#include "cthugha.h"
#include "display.h"
#include "options.h"
#include "keys.h"
#include "action.h"
#include "information.h"
#include "sound.h"
#include "translate.h"
#include "interface.h"
#include "disp-sys.h"
#include "imath.h"

#include <string.h>
#include <time.h>
#include <stdio.h>
#include <stdlib.h>
#include <math.h>

/* possible display-function */
int screen_up();	int screen_down();	int screen_2hor();
int screen_r2hor();	int screen_4hor();	int screen_2verd();
int screen_r2verd();	int screen_4kal();	int screen_hfield();
int screen_roll();	int screen_zick();	int screen_bent();
int screen_plate();

feature_entry _screens[] = {
    { screen_up,	1,	"Up",		"Up Display" },
    { screen_down,	1,	"Down",		"Upside Down"},
    { screen_2hor,	1,	"2hor",		"Hor. Split out"},
    { screen_r2hor,	1,	"r2hor",	"Hor. Split in"},
    { screen_4hor,	1,	"4hor",		"Kaleidoscope"},
    { screen_2verd,	1,	"2verd",	"90deg rot. mirror"},
    { screen_r2verd,	1,	"r2verd",	"90deg rot. mirror II"},
    { screen_4kal,	1,	"4kal",		"90deg Kaleidoscope"},
    { screen_hfield,	1,	"hfield",	"Heightfield"},
    { screen_roll,	1,	"roll",		"Roll around x-axis"},
    { screen_zick,	0,	"zick",		"Zick Zack"},
    { screen_bent,	1,	"bent",		"A bending plane"},
    { screen_plate,	1,	"plate",	"A rotating plate"}
};
feature screens = {
    "display",
    _screens,
    sizeof(_screens) / sizeof(feature_entry),
    CHANGE_RANDOM,
    f_screen
};
xy screen_sizes[] = {
    {1,1},	/* up */	
    {1,1},	/* down */
    {1,1},	/* 2hor */	
    {1,1},	/* r2hor */
    {1,1},	/* 4hor */	
    {1,1},	/* 2verd */
    {1,1},	/* r2verd */	
    {1,1},	/* 4 kal */
    {2,2},      /* hfield */     
    {1,1},	/* rooll */
    {1,1},      /* zick */
    {2,2},	/* bent */
    {2,2},	/* plate */
};


xy disp_do_tiles = {1,1};			/* do tiles or not */
xy disp_nr_tiles = {0,0};			/* nr of tiles to do */
xy disp_size = {0,0};				/* size of drawing area */
xy disp_one_size = {0,0};
xy draw_size = {0,0};				/* size of the drawn image */

xy disp_do_mirror = {-1,-1};			/* allow mirror horiz, vert */
xy disp_mirror = {0,0};			

int bytes_per_line = 0;

unsigned char * active_buffer;			/* Next on screen */
unsigned char * passive_buffer;			/* right now on screen */
unsigned char * active_buffer_rgb[3];
unsigned char * passive_buffer_rgb[3];

char screen_first[256] = "";			/* Start with this scrn-fkt */

int display_frames = 0;				/* counter for speed-tests */
time_t display_start;

int display_clear = 1;				/* clear screen */
int display_direct = 0;				/* draw directly to screen */

int display_mode = 0;				/* graphic mode */

int bpp = 8;					/* bits per pixel */
int bypp = 1;					/* bytes per pixel */

unsigned char * display_mem = NULL;		/* memory for screen-fkt to draw to */

enum draw_mode_t draw_mode = DM_direct;		/* how drawing is done */

int red_mask, green_mask, blue_mask;
int red_shift, green_shift, blue_shift;


/*****************************************************************************
 * Helping macros and fuctions for the Screen-functions
 ****************************************************************************/

#define memeq(dest,src)	    *(dest)=(src)
#define tbleq1(dest,src)    *((unsigned char*)(dest))  = bitmap_colors[(src)] 
#define tbleq2(dest,src)    *((unsigned short*)(dest)) = bitmap_colors[(src)] 
#define tbleq4(dest,src)    *((unsigned int*)(dest))   = bitmap_colors[(src)] 

inline void tblcpy1(unsigned char * dst, unsigned char * src, int n) {
    for(;n!=0;n--) {
	tbleq1(dst,*src);
	src ++; dst ++;
    }
}
inline void tblcpy2(unsigned char * dst, unsigned char * src, int n) {
    for(;n!=0;n--) {
	tbleq2(dst,*src);
	src ++; dst +=2;
    }
}
inline void tblcpy4(unsigned char * dst, unsigned char * src, int n) {
    for(;n!=0;n--) {
	tbleq4(dst,*src);
	src ++; dst +=4;
    }
}

#define rgbeqc(dst,src) {					\
    unsigned char * sr = &src;					\
    unsigned char * sg = &src-active_buffer_rgb[0] + active_buffer_rgb[1];	\
    unsigned char * sb = &src-active_buffer_rgb[0] + active_buffer_rgb[2];	\
    								\
    int r =  (*sr) >> 5;					\
    int g = ((*sg) >> 4) & 56;					\
    int b =  (*sb) & 192;					\
    *(dst) = r | g | b;						\
}

inline void rgbcpyc(unsigned char * dst, unsigned char * src, int n) {
    
    unsigned char * sr = src;
    unsigned char * sg = src-active_buffer_rgb[0] + active_buffer_rgb[1];
    unsigned char * sb = src-active_buffer_rgb[0] + active_buffer_rgb[2];

    for(; n != 0; n--) {
	int r = (*sr) >> 5;
	int g = ((*sg) >> 4) & 56;
	int b = ( *sb) & 192;
	*(dst) = r | g | b;
	
	dst ++;
	sr ++; sg ++; sb ++;
    }
}

#define rgbeq(dst,src) {					\
    unsigned char * sr = &src;					\
    unsigned char * sg = &src-active_buffer_rgb[0] + active_buffer_rgb[1];	\
    unsigned char * sb = &src-active_buffer_rgb[0] + active_buffer_rgb[2];	\
								\
    *((unsigned int*)dst) = 					\
	(red_mask   & shift(*sr, red_shift  )) |		\
	(green_mask & shift(*sg, green_shift)) |		\
	(blue_mask  & shift(*sb, blue_shift ));			\
}

inline void rgbcpy(unsigned char * dst, unsigned char * src, int n) {
    unsigned char * sr = src;
    unsigned char * sg = src-active_buffer_rgb[0] + active_buffer_rgb[1];
    unsigned char * sb = src-active_buffer_rgb[0] + active_buffer_rgb[2];
    
    for(; n != 0; n--) {
	*((unsigned int*)dst) = 
	    (red_mask   & shift(*sr, red_shift  )) |
	    (green_mask & shift(*sg, green_shift)) |
	    (blue_mask  & shift(*sb, blue_shift ));
	
	dst += bypp;
	sr ++; sg ++; sb ++;
    }
}


#define SCRN(scrn)									\
    if(flame_rgb) {									\
	if(colormapped) {								\
	    scrn_##scrn(rgbcpyc, rgbeqc, 1)						\
	} else {									\
	    scrn_##scrn(rgbcpy, rgbeq, bypp)						\
	}										\
    } else {										\
	switch(draw_mode) {								\
	case DM_direct:    								\
	    scrn_##scrn(memcpy,  memeq, 1) break;					\
	case DM_mapped1: case DM_tmp_mapped: 						\
	    scrn_##scrn(tblcpy1, tbleq1, 1) break;					\
	case DM_mapped2:								\
	    scrn_##scrn(tblcpy2, tbleq2, 2) break;					\
	case DM_mapped4:								\
	    scrn_##scrn(tblcpy4, tbleq4, 4) break;					\
	}										\
    } 											



/*
 * Initialization of display 
 *  - load palettes and PCX 
 *  - allocate buffer
 *  - does not switch to graph mode
 */
int init_display() {
    int i;
  
    if( init_display_sys() )
	return 1;

    /* The buffer has a border of 3 lines on top on bottom. These
       lines are set black before the 'flame' function is called.
       The border is not displayed to the screen
       */
    for(i=0; i < 3; i++) {
	active_buffer_rgb[i]  = cth_memory(NULL, BUFF_SIZE+6*BUFF_WIDTH, NULL) + 3*BUFF_WIDTH;
	passive_buffer_rgb[i] = cth_memory(NULL, BUFF_SIZE+6*BUFF_WIDTH, NULL) + 3*BUFF_WIDTH;
    }
    active_buffer  = active_buffer_rgb[0];
    passive_buffer = passive_buffer_rgb[0];

    /* clear buffers */
    for(i=0; i < 3; i++) {
	bzero(active_buffer_rgb[i], BUFF_SIZE);
	bzero(passive_buffer_rgb[i], BUFF_SIZE);
    }

    /* load the palettes (internal and external) */
    if( load_palettes() )
	return 1;
		
    /* load the pcx-files */
    if ( init_pcx() )
	return 1;

    /* check if any palettes in use */
    if ( palettes.nr_entries == 0) {
	printfe("No palettes specified.\n");
	return 1;
    }

    /* mirroring: do_mirror == 1 -> mirror now, and allow it */
    if(disp_do_mirror.x == 1)
	disp_mirror.x = 1;
    if(disp_do_mirror.y == 1)
	disp_mirror.y = 1;
    /* mirroring: do_mirror == -1 -> don't mirror now, but allow it */
    if(disp_do_mirror.x == -1)
	disp_do_mirror.x = 1, disp_mirror.x = 0;
    if(disp_do_mirror.y == -1)
	disp_do_mirror.y = 1, disp_mirror.y = 0;
    

    return 0;
}

/*
 * Clean upt display
 */
int exit_display() {
    exit_graph_mode();
    return 0;
}


/*
 * Turn on graph-mode
 *   - turnon graph mode
 *   - allocate XImage
 */
int init_graph_mode() {

    printfv(3, "Initializing graphics mode...\n");

    if( init_graph_mode_sys() )
	return 1;

    printfv(3, "  display size: %dx%d\n", disp_size.x, disp_size.y);
    printfv(3, "  bytes/line  : %d\n", bytes_per_line);
    printfv(3, "  bytes/pixel : %d\n", bypp);

    display_start = gettime();
	
    return 0;
}

/*
 * Switch back to text mode
 */
int exit_graph_mode() {

    printfv(3, "Closing graphics mode...\n");

    exit_graph_mode_sys();
    exit_palettes();

    return 0;
}


int change_mirror_horiz(int to) {
    disp_mirror.x = val_change(to, 0, disp_do_mirror.x, disp_mirror.x);
    return 0;
}
int change_mirror_vert(int to) {
    disp_mirror.y = val_change(to, 0, disp_do_mirror.y, disp_mirror.y);
    return 0;
}

/*****************************************************************************/

/*
 * exchange active and passive buffer
 */
void swap_buffers() {
    char * tmp; int i;

    tmp = active_buffer; active_buffer = passive_buffer; passive_buffer = tmp;

    for(i=0; i < 3; i++) {
	tmp = active_buffer_rgb[i];
	active_buffer_rgb[i] = passive_buffer_rgb[i];
	passive_buffer_rgb[i] = tmp;
    }
}



/*
 * calculate information for tile and offset 
 */
int update_draw_size() {
    xy s = screen_sizes[screens.current];

    disp_nr_tiles.x = disp_do_tiles.x ? (disp_size.x/BUFF_WIDTH/s.x) : 1;
    disp_nr_tiles.y = disp_do_tiles.y ? (disp_size.y/BUFF_HEIGHT/s.y) : 1;

    disp_one_size.x = s.x * BUFF_WIDTH;
    disp_one_size.y = s.y * BUFF_HEIGHT;

    draw_size.x = disp_nr_tiles.x * disp_one_size.x;
    draw_size.y = disp_nr_tiles.y * disp_one_size.y;

    return 0;
}

/*
 * clear border around display if necessary 
 */
static int clear_border() {
    if ( display_text_on_screen || display_clear ) {
	int bwidth = SCREEN_OFFSET_X;		/* width of left/right border */
	int bheight = SCREEN_OFFSET_Y;		/* height of upper/lower border */

	/* upper border */
	disp_clear_box_sys(0,0, disp_size.x, bheight);
	/* left border */
	disp_clear_box_sys(0, bheight, bwidth, draw_size.y);
	/* right border */
	disp_clear_box_sys(disp_size.x - bwidth, bheight, bwidth, draw_size.y);
	/* lower border */
	disp_clear_box_sys(0,disp_size.y-bheight, disp_size.x, bheight);

	/* if now text is on screen, we have to clean in the next iteration,
	   (maybe the text gets removed */
	display_clear = (display_text_on_screen) ? 1 : 0;
	return 1;
    }
    return 0;
}

/*
 * tile and mirror buffer to screen.
 *
 */
void scrn_mirror1(unsigned char * dst, unsigned char * src) {
    int j;
    for(j=disp_one_size.x; j != 0; j--) {
	* dst = * src;
	src ++; dst --;
    }
}
void scrn_mirror2(unsigned short * dst, unsigned short * src) {
    int j;
    for(j=disp_one_size.x; j != 0; j--) {
	* dst = * src;
	src ++; dst --;
    }
}
void scrn_mirror4(unsigned long * dst, unsigned long * src) {
    int j;
    for(j=disp_one_size.x; j != 0; j--) {
	* dst = * src;
	src ++; dst --;
    }
}
static void mirror_horiz() {
    int i;
    unsigned char * src = display_mem;
    unsigned char * dst = display_mem + (2*disp_one_size.x-1) * bypp;
    
    switch(bypp) {
    case 1:
	for(i=disp_one_size.y; i != 0; i--) {
	    scrn_mirror1(dst, src);
	    src += bytes_per_line;
	    dst += bytes_per_line;
	}
	break;
    case 2:
	for(i=disp_one_size.y; i != 0; i--) {
	    scrn_mirror2((unsigned short*)dst, (unsigned short*)src);
	    src += bytes_per_line;
	    dst += bytes_per_line;
	}
	break;
    case 4:
	for(i=disp_one_size.y; i != 0; i--) {
	    scrn_mirror4((unsigned long*)dst, (unsigned long*)src);
	    src += bytes_per_line;
	    dst += bytes_per_line;
	}
	break;
    }
}
static void mirror_vert() {
    int i;
    unsigned char * src = display_mem;
    unsigned char * dst = display_mem + (2*disp_one_size.y-1) * bytes_per_line;
    
    for(i = disp_one_size.y; i != 0; i--) {
	memcpy(dst, src, draw_size.x * bypp);
	src += bytes_per_line;
	dst -= bytes_per_line;
    }
}

int tile_buffer() {
    int i;


    if( (disp_nr_tiles.x > 1) && (disp_mirror.x) ) {		/* mirror */
	
	/* mirror horizontal, first block */
	mirror_horiz();

	for(i=1; i < disp_nr_tiles.x/2; i++)
	    disp_copy_box_sys(SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
			      disp_one_size.x*2, disp_one_size.y,
			      SCREEN_OFFSET_X + i*2*disp_one_size.x,
			      SCREEN_OFFSET_Y);
	/* special case for odd number of tiles */
	if( disp_nr_tiles.x & 1) 
	    disp_copy_box_sys(SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
			      disp_one_size.x, disp_one_size.y,
			      SCREEN_OFFSET_X + i*2*disp_one_size.x,
			      SCREEN_OFFSET_Y);

    } else {							/* tile only */
	for(i=1; i < disp_nr_tiles.x; i++)
	    disp_copy_box_sys(SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
			      disp_one_size.x, disp_one_size.y,
			      SCREEN_OFFSET_X + i*disp_one_size.x,
			      SCREEN_OFFSET_Y);
    }

    if( (disp_nr_tiles.y > 1) && (disp_mirror.y) ) {		/* mirror */

	/* mirror vertically, first line */
	mirror_vert(); 

	for(i=1; i < disp_nr_tiles.y/2; i++)
	    disp_copy_box_sys(SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
			      draw_size.x, 2*disp_one_size.y,
			      SCREEN_OFFSET_X, 
			      SCREEN_OFFSET_Y + i*2*disp_one_size.y);
			      
	/* special case for odd number of tiles */
	if(disp_nr_tiles.y & 1) 
	    disp_copy_box_sys(SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
			      draw_size.x, disp_one_size.y,
			      SCREEN_OFFSET_X, 
			      SCREEN_OFFSET_Y + i*2*disp_one_size.y);
			      
    } else {							/* tile only */
	for(i=1; i < disp_nr_tiles.y; i++)
	    disp_copy_box_sys(SCREEN_OFFSET_X, SCREEN_OFFSET_Y,
			      draw_size.x, disp_one_size.y,
			      SCREEN_OFFSET_X, 
			      SCREEN_OFFSET_Y + i*disp_one_size.y);
    }
    return 0;
}

/*
 * Bring the active Buffer to screen.
 */
int display() {
    int clear;

    char * display_base = pre_draw_sys();

    do {
	update_draw_size();
	display_mem = display_base + SCREEN_OFFSET;
    } while( update_screen() );			/* draw the buffer */
    
    clear = clear_border();
    tile_buffer();
    show_text();

    disp_copy_sys(clear);

    swap_buffers();
    display_frames ++;
    
    return 0;
}




/*****************************************************************************
 * Screen-functions
 ****************************************************************************/

/*
 * Draw a permutation of the lines.
 * 
 * A lot of the display functions are now based on this.
 */
static unsigned char * perm_lines[MAX_BUFF_HEIGHT];
#define scrn_perm(cpy,eq,bytes_per_pixel)		\
    for(i=BUFF_HEIGHT; i != 0; i--) {			\
        ##cpy(scrn, *perm, BUFF_WIDTH);			\
	scrn += bytes_per_line;				\
        perm ++;					\
    }
void screen_perm(void) {
    unsigned char * scrn = display_mem;
    unsigned char ** perm = perm_lines;
    int i;

    SCRN(perm)
}

#define PERM_ADDR(i)	( active_buffer + (i) * BUFF_WIDTH )


int screen_up() {
    int i;
    for(i=0; i < BUFF_HEIGHT; i++)
	perm_lines[i] = PERM_ADDR(i);
		
    screen_perm();
    return 0;
}


int screen_down() {
    int i;
    for(i=0; i < BUFF_HEIGHT; i++)
	perm_lines[i] = PERM_ADDR(BUFF_HEIGHT - i - 1);
    
    screen_perm();
    return 0;
}

int screen_2hor() {
    int i;
    for(i=0; i < BUFF_HEIGHT/2; i++) {
	/* lower half of buffer maps to upper half of screen */
	perm_lines[i] = PERM_ADDR(BUFF_HEIGHT/2 + i);
	/* lower half of buffer get turned around to lower half of screen*/
	perm_lines[i+BUFF_HEIGHT/2] = PERM_ADDR(BUFF_HEIGHT - i - 1);
    }
	
    screen_perm();
    return 0;
}

		
int screen_r2hor() {
    int i;
    for(i=0; i < BUFF_HEIGHT/2; i++) {
	/* lower half of buffer get turned around to upper half of screen*/
	perm_lines[i] = PERM_ADDR(BUFF_HEIGHT - i - 1);
	/* lower half of buffer maps to lower half of screen */
	perm_lines[i+BUFF_HEIGHT/2] = PERM_ADDR(BUFF_HEIGHT/2 + i);
    }

    screen_perm();
    return 0;
}


#define scrn_4hor_a(cpy,eq,bytes_per_pixel)			\
    for(y=BUFF_HEIGHT/2; y != 0; y--) {				\
								\
	/* left half */						\
	##cpy( scrn, tmp, BUFF_WIDTH/2);				\
	scrn += BUFF_WIDTH/2*bytes_per_pixel;			\
	tmp += BUFF_WIDTH/2;					\
								\
	/* right half */					\
	for(x=BUFF_WIDTH/2; x != 0; x--)	{			\
	    ##eq(scrn, *tmp);					\
	    scrn += bytes_per_pixel;				\
	    tmp --;						\
	}							\
	tmp += BUFF_WIDTH;					\
	scrn += bytes_per_line - BUFF_WIDTH*bytes_per_pixel;	\
    }
#define scrn_4hor_b(cpy,eq,bytes_per_pixel)			\
    for(y= BUFF_HEIGHT/2; y != 0; y--) {				\
								\
	/* left half */						\
	##cpy( scrn, tmp, BUFF_WIDTH/2);				\
	scrn += BUFF_WIDTH/2*bytes_per_pixel;			\
	tmp += BUFF_WIDTH/2;					\
								\
	/* right half */					\
	for(x=BUFF_WIDTH/2; x != 0; x--)	{			\
	    ##eq(scrn, *tmp);					\
	    scrn += bytes_per_pixel;				\
	    tmp --;						\
	}							\
	tmp -= BUFF_WIDTH;					\
	scrn += bytes_per_line - BUFF_WIDTH*bytes_per_pixel;	\
    }
int screen_4hor() {
    unsigned char * tmp, * scrn;
    int x,y;
	
    /* upper half of screen */	
    tmp = active_buffer + BUFF_SIZE/2;
    scrn = display_mem;
    SCRN(4hor_a)

    /* lower half of screen */
    tmp = active_buffer + BUFF_WIDTH*BUFF_HEIGHT - BUFF_WIDTH;
    scrn = display_mem + bytes_per_line*BUFF_HEIGHT/2;
    SCRN(4hor_b)

    return 0;
}


#define scrn_2verd(cpy,eq,bytes_per_pixel)		\
    for(y=BUFF_HEIGHT; y != 0; y--) {			\
	for(x=BUFF_WIDTH/2; x != 0; x--) {		\
	    ##eq(scrn, *tmp);				\
            scrn += bytes_per_pixel;			\
	    tmp += BUFF_WIDTH;				\
	}						\
	for(x=BUFF_WIDTH/2; x != 0; x--) {		\
	    ##eq(scrn, *tmp);				\
            scrn += bytes_per_pixel;			\
	    tmp -= BUFF_WIDTH;				\
	}						\
	tmp ++;						\
	scrn += bytes_per_line - bytes_per_pixel*BUFF_WIDTH;		\
    }

int screen_2verd() {
    if( BUFF_WIDTH/2 <= BUFF_HEIGHT) {
	int x,y;
	unsigned char * tmp = active_buffer;
	unsigned char * scrn = display_mem;
	SCRN(2verd)
    } else {
	change(f_screen, CHANGE_NEXT, 0);
	return 1;
    }
    return 0;
}


#define scrn_r2verd(cpy,eq,bytes_per_pixel)			\
    for(y=BUFF_HEIGHT; y != 0; y--) {				\
	for(x=BUFF_WIDTH/2; x != 0; x--) {			\
	    ##eq(scrn, *tmp);					\
	    scrn -= bytes_per_pixel;				\
	    tmp += BUFF_WIDTH;					\
	}							\
	for(x=BUFF_WIDTH/2; x != 0; x--) {			\
	    ##eq(scrn, *tmp);					\
	    scrn -= bytes_per_pixel;				\
	    tmp -= BUFF_WIDTH;					\
	}							\
	tmp ++;							\
	scrn -= bytes_per_line - bytes_per_pixel*BUFF_WIDTH;	\
    }
int screen_r2verd() {
    if( BUFF_WIDTH/2 <= BUFF_HEIGHT) {
	int x,y;
	unsigned char * tmp = active_buffer;
	unsigned char * scrn = display_mem + 
	    bytes_per_line*(BUFF_HEIGHT-1) + bypp*(BUFF_WIDTH-1);
	SCRN(r2verd)
    } else {
	change(f_screen, CHANGE_NEXT, 0);
	return 1;
    }
    return 0;
}


#define scrn_4kal_a(cpy,eq,bytes_per_pixel)			\
    for(y=BUFF_HEIGHT/2; y != 0; y--) {				\
	for(x=BUFF_WIDTH/2; x != 0; x--) {			\
	    ##eq(scrn, *tmp);					\
	    scrn += bytes_per_pixel;				\
	    tmp += BUFF_WIDTH;					\
	}							\
	for(x=BUFF_WIDTH/2; x != 0; x--) {			\
	    ##eq(scrn, *tmp);					\
	    scrn += bytes_per_pixel;				\
	    tmp -= BUFF_WIDTH;					\
	}							\
	tmp ++;							\
	scrn += bytes_per_line - bytes_per_pixel*BUFF_WIDTH;	\
    }
#define scrn_4kal_b(cpy,eq,bytes_per_pixel)			\
    for(y=BUFF_HEIGHT/2; y != 0; y--) {				\
	for(x=BUFF_WIDTH/2; x != 0; x--) {			\
	    ##eq(scrn, *tmp);					\
	    scrn -= bytes_per_pixel;				\
	    tmp += BUFF_WIDTH;					\
	}							\
	for(x=BUFF_WIDTH/2; x != 0; x--) {			\
	    ##eq(scrn, *tmp);					\
	    scrn -= bytes_per_pixel;				\
	    tmp -= BUFF_WIDTH;					\
	}							\
	tmp ++;							\
	scrn -= bytes_per_line - bytes_per_pixel*BUFF_WIDTH;	\
    }
int screen_4kal() {
    if( BUFF_WIDTH/2 <= BUFF_HEIGHT) {
	unsigned char * tmp, * scrn;
	int x,y;
	
	tmp = active_buffer;
	scrn = display_mem;
	/* upper half */
	SCRN(4kal_a)

	tmp = active_buffer;
	scrn = display_mem + bytes_per_line * (BUFF_HEIGHT-1) + 
	    bypp*(BUFF_WIDTH-1);
	/* lower half */
	SCRN(4kal_b)
    } else {
	change(f_screen, CHANGE_NEXT, 0);
	return 1;
    }
    return 0;
}



/*
 * 3-D display functions
 */
static const float SC = 1 << 16;
static xy height_offset[256];
static xy s1, s2, p;
static float rot[3] = {0,0,0};
static float P[4][3] = { {-1,-1,0}, {1,-1,0}, {1,1,0}, {0,0,1} };

/*
 * rotate vektor p around x/y/z axis (by rot[i])
 * result in vector r
 */
void rotate(float p[3], float rot[3], float r[3]) {
    float t1[3], t2[3];
    /* first around x axis */
    t1[0] = p[0];
    t1[1] = p[1] * cos(rot[0]) + p[2] * sin(rot[0]);
    t1[2] = -p[1] * sin(rot[0]) + p[2] * cos(rot[0]);

    /* around y axis */
    t2[0] = t1[0] * cos(rot[1]) + t1[2] * sin(rot[1]);
    t2[1] = t1[1];
    t2[2] = -t1[0] * sin(rot[1]) + t1[2] * cos(rot[1]);

    /* around z axis */
    r[0] = t2[0] * cos(rot[2]) + t2[1] * sin(rot[2]);
    r[1] = -t2[0] * sin(rot[2]) + t2[1] * cos(rot[2]);
    r[2] = t2[2];
}
inline float sqr(float a) {
    return a*a;
}

/*
 * preparations for a 3D display
 */
int prepare_3d(int maxZ) {
    int i;
    float x,y,z,l;
    float ro[4][3];
    double scale;
    unsigned char * clr = display_mem;

    /* calculate the maximal size of the "sheet" */
    x = sqr(BUFF_WIDTH/2);
    y = sqr(BUFF_HEIGHT/2);
    z = sqr((float)(BUFF_WIDTH)/640.0 * (float)maxZ);
    l = sqrt(x + y + z);

    /* make sure, the image always fits into the display region */
    scale = (float)min(BUFF_WIDTH, BUFF_HEIGHT) / l;
    
    /* rotate a little bit */
    /* TODO: use fps, some value from the sound to set speed,
       external value (joystick) to set rotation */
    rot[0] += 0.006;
    rot[1] += 0.010;
    rot[2] += 0.004;
    for(i=0; i < 4; i++)
	rotate(P[i], rot, ro[i]);
	
    for(i=0; i < 256; i++) {
	height_offset[i].x = ro[3][0] * i * (float)BUFF_WIDTH / 640.0 * SC;
	height_offset[i].y = ro[3][1] * i * (float)BUFF_WIDTH / 640.0 * SC;
    }

    /* calculate the step sizes */
    s1.x = (ro[1][0] - ro[0][0]) * scale * SC/2.0;
    s1.y = (ro[1][1] - ro[0][1]) * scale * SC/2.0;

    s2.x = (ro[2][0] - ro[1][0]) * scale * SC/2.0;
    s2.y = (ro[2][1] - ro[1][1]) * scale * SC/2.0;

    /* starting point */
    p.x = -BUFF_WIDTH/2 * s1.x -BUFF_HEIGHT/2 * s2.x;
    p.y = -BUFF_WIDTH/2 * s1.y -BUFF_HEIGHT/2 * s2.y;

    /* clear screen */
    for(i=2*BUFF_HEIGHT; i != 0; i--) {
	bzero(clr, 2*BUFF_WIDTH*bypp);
	clr += bytes_per_line;
    }

    return 0;
}

/*
 * hfield: buffer_value determines height
 */
#define scrn_hfield(cpy,eq,bytes_per_pixel)				\
   for(i=BUFF_HEIGHT; i != 0; i--) {					\
       xy t = p;							\
       for(j=BUFF_WIDTH; j != 0; j--) {					\
	   unsigned char * d = dst +					\
	       ((p.y + height_offset[*src].y) >> 16) * bytes_per_line +	\
	       ((p.x + height_offset[*src].x) >> 16) * bytes_per_pixel;	\
	   unsigned char c = *src | 128;				\
	   ##eq(d, c);							\
	   src++;							\
	   p.x += s1.x; p.y += s1.y;					\
       }								\
       p.x = t.x + s2.x;						\
       p.y = t.y + s2.y;						\
   }
int screen_hfield() {
    unsigned char * src = active_buffer;
    unsigned char * dst = display_mem + 
	BUFF_WIDTH*bypp + bytes_per_line*BUFF_HEIGHT;
    int i,j;
    
    if( (2*BUFF_WIDTH > disp_size.x) || (2*BUFF_HEIGHT > disp_size.y) || flame_rgb) {
	change(f_screen, CHANGE_NEXT, 0);
	return 1;
    }

    prepare_3d(256);

    SCRN(hfield)

    return 0;
}

/*
 * bent: height is a sine-wave along x axis
 */
#define scrn_bent(cpy,eq,bytes_per_pixel)				\
   for(i=BUFF_HEIGHT; i != 0; i--) {					\
       xy t = p;							\
       for(j=BUFF_WIDTH; j != 0; j--) {					\
	   unsigned char * d = dst +					\
	       ((p.y + height_offset[height[j]].y) >> 16) * bytes_per_line +	\
	       ((p.x + height_offset[height[j]].x) >> 16) * bytes_per_pixel;	\
	   unsigned char c = *src | 128;				\
	   ##eq(d, c);							\
	   src++;							\
	   p.x += s1.x; p.y += s1.y;					\
       }								\
       p.x = t.x + s2.x;						\
       p.y = t.y + s2.y;						\
   }
int screen_bent() {
    unsigned char * src = active_buffer;
    unsigned char * dst = display_mem + 
	BUFF_WIDTH*bypp + bytes_per_line*BUFF_HEIGHT;
    int i,j;
    static int height[MAX_BUFF_WIDTH];
    static float t = 0;
    static float h = 0.0;
    static float stp = 0.0;

    if( (2*BUFF_WIDTH > disp_size.x) || (2*BUFF_HEIGHT > disp_size.y) || flame_rgb) {
	change(f_screen, CHANGE_NEXT, 0);
	return 1;
    }

    prepare_3d(256);

    h = h * 0.95 + min(2.0*sound_amplitude,120.0) * 0.05;

    for(i=BUFF_WIDTH; i != 0; i--) {
	height[i] = (int)(h * sin(t) *			\
			  sin((double)(i)/(double)BUFF_WIDTH*3.0*M_PI)) + 128;
    }
    stp = stp * 0.95 + max(fps*2, 1) * 0.05;
    t += 4.0/stp;

    SCRN(bent)

    return 0;
}

/*
 * plate: height contant to 0
 */
#define scrn_plate(cpy,eq,bytes_per_pixel)				\
   for(i=BUFF_HEIGHT; i != 0; i--) {					\
       xy t = p;							\
       for(j=BUFF_WIDTH; j != 0; j--) {					\
	   unsigned char * d = dst +					\
	       (p.y >> 16) * bytes_per_line +				\
	       (p.x >> 16) * bytes_per_pixel;				\
	   unsigned char c = *src | 128;				\
	   ##eq(d, c);							\
	   src++;							\
	   p.x += s1.x; p.y += s1.y;					\
       }								\
       p.x = t.x + s2.x;						\
       p.y = t.y + s2.y;						\
   }
int screen_plate() {
    unsigned char * src = active_buffer;
    unsigned char * dst = display_mem + 
	BUFF_WIDTH*bypp + bytes_per_line*BUFF_HEIGHT;
    int i,j;

    if( (2*BUFF_WIDTH > disp_size.x) || (2*BUFF_HEIGHT > disp_size.y) || flame_rgb) {
	change(f_screen, CHANGE_NEXT, 0);
	return 1;
    }

    prepare_3d(1);

    SCRN(plate)

    return 0;
}

/*
 * Map the buffer on a cylinder (around x-axis), and roll it.
 * Buffer is mapped 2 times on the cylinder, so that no discontinuities arise
 *
 * other possible things with this idea: waves, ...
 */
int screen_roll() {
    int i;
    static double theta = 0;		/* rotation angle */

    for(i=0; i < BUFF_HEIGHT; i++) {
	int p = i - BUFF_HEIGHT/2;		
	double phi = acos( (double)p / (double)(BUFF_HEIGHT/2));
	int b = (int)((theta + phi) / M_PI * (double)BUFF_HEIGHT);
	b %= 2*BUFF_HEIGHT; 
	if (b < 0)		b += 2*BUFF_HEIGHT; 
	if (b >= BUFF_HEIGHT)	b = 2*BUFF_HEIGHT - b - 1;

	perm_lines[i] = PERM_ADDR(b);
    }
    theta += M_PI / 40.0;		/* roate a little bit */
    if (theta > 2.0*M_PI) 
	theta -= 2.0*M_PI;

    screen_perm();

    return 0;
}


/*
 * this is not looking good
 */
#define ZICK_SMOOTH	10
static int zicks[MAX_BUFF_HEIGHT];
#define scrn_zick(cpy,eq,bytes_per_pixel)		\
    for(i=BUFF_HEIGHT; i != 0; i--) {			\
	zicks[i] = ((sound_data[i][0]) + ZICK_SMOOTH*zicks[i]) / (ZICK_SMOOTH+1);	\
        d = (d + zicks[i])/2;		\
        D = d * bytes_per_pixel;			\
        if(d == 0) {					\
	    ##cpy( scrn, src, BUFF_WIDTH);		\
	} else if(d > 0) {				\
	    memset(scrn, 0, D);				\
            ##cpy(scrn+D, src, BUFF_WIDTH-d);		\
        } else {					\
	    ##cpy(scrn, src-d, BUFF_WIDTH+d);		\
	    memset(scrn+bytes_per_line+D, 0, -D);	\
	}						\
        src += BUFF_WIDTH;				\
	scrn += bytes_per_line;				\
    }
int screen_zick() {
    int i;
    unsigned char * scrn = display_mem;
    unsigned char * src = active_buffer;
    static int first = 1;

    static int d = 0; 
    int D;

    if(first) {
	for(i=0; i < BUFF_HEIGHT+1; i++)
	    zicks[i] = 0;

    }

    SCRN(zick)

    return 0;
}


