/* This file is Copyright 1993 by Clifford A. Adams */
/* sgfile.c
 *
 * group scan mode file code (also handles restriction entries)
 */

/* trim the list */
#include "EXTERN.h"
#include "common.h"
#ifdef SCAN
#include "intrp.h"
#include "search.h"
#include "sgroup.h"
#include "sgdata.h"
#include "sgmisc.h"
#include "term.h"
#include "util.h"
#include "url.h"
#include "INTERN.h"
#include "sgfile.h"

/* complex expressions taken from ngsrch.c */

static COMPEX sgcompex;

static bool sg_file_initialized	INIT(FALSE);

static bool sg_begin_found INIT(FALSE);
/* label to start working with */
static char *sg_begin_label INIT(Nullch);

/* initializes sg_file stuff */
void
sg_file_init()
{
    if (sg_file_initialized)
	return;			/* only necessary once */
    init_compex(&sgcompex);
    sg_file_initialized = TRUE;
}

bool sg_match_err	INIT(FALSE);

/* Wildmat function:
 *      Based on Rich $alz's wildmat, reduced to the simple case of *
 *      and text.  The complete version can be found in Rich's INN,
 *      among other places; write to <rsalz@uunet.uu.net>
 */

/* an improbable number */
#define ABORT			-42


/*
**  Match text and p, return TRUE, FALSE, or ABORT.
*/
static int
sg_DoMatch(text, p)
register char	*text;
register char	*p;
{
    register int	matched;

    for ( ; *p; text++, p++) {
	if (*p == '*') {
	    while (*++p == '*')
		/* Consecutive stars act just like one. */
		continue;
	    if (*p == '\0')
		/* Trailing star matches everything. */
		return TRUE;
	    while (*text)
		if ((matched = sg_DoMatch(text++, p)) != FALSE)
		    return matched;
	    return ABORT;
	}
	if (*text != *p) {
	    if (*text == '\0')
		return ABORT;
	    return FALSE;
	}
    }
    return *text == '\0';
}

void
sg_use_pattern(pattern)
char *pattern;
{
    register long i;
    long e;		/* misc. entry */
    register char *s = pattern;

    sg_match_err = FALSE;		/* no errors (yet :-) */
    if (!s || !*s) {
	printf("\ngroup scan pattern: empty regular expression\n") FLUSH;
	return;
    }
    if (*pattern == '!') {	/* exclude these matches */
	if (!*(pattern+1)) {
	    printf("\ngroup scan pattern: empty expression after '!'\n") FLUSH;
	    return;
	}

	/* note: skip zero'th entry */
	for (i=1;i<sg_num_ents;i++) {
	    if (sg_ents[i].type == 1)		/* newsgroup type */
		if (sg_DoMatch(sg_groups[sg_ents[i].groupnum].name,pattern+1)
		    == TRUE)
			sg_ents[i].type = -1;	/* make entry invalid */
	}
    } else {	/* include these matches */
	for (i=0;i<sg_num_groups;i++) {
	    if (sg_DoMatch(sg_groups[i].name,pattern) == TRUE) {
		int j;
		for (j=1;j<sg_num_ents;j++)
		    if (sg_ents[j].type == 1)	/* newsgroup type */
			if (sg_ents[j].groupnum == i)	/* same group */
			    break;	/* out of for loop */
		if (j>=sg_num_ents) {	/* was not there before */
		    e = sg_add_ent(1,sg_groups[i].name);
		    sg_ents[e].groupnum = i;
		}
	    }
	}
    }
}

void
sg_use_restrict(restrict,title)
char *restrict;
char *title;
{
    (void)sg_new_context();
    sg_contexts[sg_num_contexts-1].title = savestr(title);

    sg_use_group_line(restrict);
    sg_delete_invalid_ents();			/* in case some were deleted */
}

/* returns TRUE on success, FALSE otherwise */
bool
sg_use_file(fname,title)
char *fname;		/* file name (optional trailing label) */
			/* a trailing label is specified like:
			   "title" filename >label
			   where fname = "filename >label" */
char *title;		/* for top title/status bar */
{
    static char lbuf[LBUFLEN];
    static char fnamebuf[LBUFLEN];
    FILE *fp;
    char *s,*p;
    int i;
    char *open_name;
    bool save_temp;
    bool begin_top;	/* if FALSE, look for "begin group"
			   before interpreting */
    char *label;	/* label to start interpreting at.  If NULL,
			   interpretation starts at top of file. */

    save_temp = FALSE;
    begin_top = TRUE;	/* default assumption (might be changed later) */
    label = Nullch;
    p = Nullch;

    if (!fname)
	return(FALSE);	/* bad argument */
    /* deal with label-only filenames */
    if ((*fname == '-') && (*(fname+1) == '>')) {
	label = fname+2;
	/* get the current filename */
	if (sg_num_contexts==0) {
	    /* really shouldn't happen */
	    printf("\nsg_use_file: top level filename error\n") FLUSH;
	    return(FALSE);
	}
	/* use temporary files first (so URLs don't need to be reloaded) */
	for (i=sg_num_contexts-1;i>=0;i--)
	    if ((p=sg_contexts[i].tmpfilename)	/* EQUALS */
		|| (p=sg_contexts[i].filename))	/* EQUALS */
		break;
	fname = p;	/* use the same file */
	if (!fname)
	    return(FALSE);	/* no enclosing filename */
    }
    /* check for labels within a filename */
    for (p=fname;*p;p++) {
	if ((*p == ' ') && (*(p+1) == '-') && (*(p+2) == '>')) {
	    label = p+3;
/* check length later for overflow */
	    strncpy(fnamebuf,fname,p-fname);
	    fnamebuf[p-fname] = '\0';
	    fname = fnamebuf;
	}
    }

    s = fname;
    open_name = s;
    /* open URLs and translate them into local temporary filenames */
    if (!strncasecmp(fname,"URL:",4)) {
#ifdef USEURL
	s = fname;
	open_name = temp_filename();
	if (!url_get(fname+4,open_name))
	    open_name = NULL;
	save_temp = TRUE;
	begin_top = FALSE;	/* we will need a "begin group" */
#else
	printf("This copy of strn does not have URL support.\n") FLUSH;
	open_name = NULL;
#endif
    } else if (*s == ':') {	/* relative to last file's directory */
	if (sg_num_contexts==0) {
	    /* really shouldn't happen */
	    printf("\nsg_use_file: top level filename error\n") FLUSH;
	    return(FALSE);
	}
	for (i=sg_num_contexts-1;i>=0;i--)
	    if ((p=sg_contexts[i].filename))	/* EQUALS */
		break;
	if (i<0) {	/* no file found */
	    printf("\nsg_use_file internal error: current not found\n") FLUSH;
	    /* consider doing an assertion? something is really wrong */
	    return(FALSE);
	}
	strcpy(lbuf,p);
	p = rindex(lbuf,'/');
	if (!p) {
	    printf("\nsg_use_file: current has no slash\n") FLUSH;
	    return(FALSE);
	}
	p++;					/* point to after slash */
	strcpy(p,s+1);
	s = lbuf;
	open_name = s;
    }
    if (!open_name)
	return(FALSE);
/*    printf("\nAttempting to open file %s\n",open_name) FLUSH; /* */
    sg_begin_found = begin_top;
    sg_begin_label = label;
    fp = fopen(filexp(open_name),"r");
    if (!fp)
	return(FALSE);		/* unsuccessful */
    (void)sg_new_context();
    sg_contexts[sg_num_contexts-1].title = savestr(title);
    if (label)
	sg_contexts[sg_num_contexts-1].label = savestr(label);
    sg_contexts[sg_num_contexts-1].filename = savestr(s);
    if (save_temp)
	sg_contexts[sg_num_contexts-1].tmpfilename = savestr(open_name);
/* Later considerations:
 * 1. Long lines
 * 2. Backslash continuations
 */
    while ((s = fgets(lbuf,1020,fp))) {		/* EQUALS */
	if (!s)		/* end of file */
	    break;
	if (!sg_do_line(s))
	    break;	/* end of useful file */
    }
    fclose(fp);
    if (!sg_begin_found)
	printf("\"begin group\" not found.\n") FLUSH;
    if (sg_begin_label)
	printf("label not found: %s\n",sg_begin_label);
    sg_delete_invalid_ents();			/* in case some were deleted */
    return(TRUE);
}

/* rereads the current file */
/* returns TRUE on success, FALSE otherwise */
bool
sg_reuse_file()
{
    char *fname;
    char *title;
    bool flag;
    
    /* If this is a restriction list, there will be no local filename */
    if (!sg_contexts[sg_num_contexts-1].filename)
	return(TRUE);			/* nothing to do */
    /* if we were using a label, re-create the label */
    if (sg_contexts[sg_num_contexts-1].label) {
	char lbuf[LBUFLEN];
	sprintf(lbuf,"%s ->%s",sg_contexts[sg_num_contexts-1].filename,
	        sg_contexts[sg_num_contexts-1].label);
	fname = savestr(lbuf);
    } else
	fname = savestr(sg_contexts[sg_num_contexts-1].filename);
    title = savestr(sg_contexts[sg_num_contexts-1].title);
    sg_old_context();
    flag = sg_use_file(fname,title);
    free(fname);
    free(title);
    return(flag);
}

/* if non-NULL, the description (printing name) of the entry */
static char *sg_line_desc;

/* returns FALSE when no more lines should be interpreted */
bool
sg_do_line(line)
char *line;
{
    char *s;
    char *p;
    long e;		/* entry index */

    s = line;

    if (s[strlen(s)-1] == '\n')
	s[strlen(s)-1] = '\0';		/* delete newline */
    while ((*s == ' ') || (*s == '\t'))
	s++;		/* delete whitespace */
    if (*s == '\0')
	return(TRUE);	/* empty line */

    if (!sg_begin_found) {
	if (strncasecmp(s,"begin group",11))
	    return(TRUE);	/* wait until "begin group" is found */
	sg_begin_found = TRUE;
    }
    if (sg_begin_label) {
	if ((*s == '>') && (*(s+1) == ':') && (strEQ(s+2,sg_begin_label))) {
	    sg_begin_label = Nullch;	/* interpret starting at next line */
	}
	return(TRUE);
    }
    if (sg_line_desc) {
	free(sg_line_desc);
	sg_line_desc = (char*)NULL;
    }
    if (*s == '"') {	/* description name */
	/* try handling backslashes later?  Perhaps other special cases? */
	s++;
	for (p=s;*p && *p != '"';p++)
	    ;	/* EMPTY */
	if (!*p) {
	    printf("group scan: unfinished quote.  Line was:\n%s\n",
		   line) FLUSH;
	    return(TRUE);
	}
	*p = '\0';
	sg_line_desc = savestr(s);
	*p = '"';
	++p;
	s = p;
    }
    while ((*s == ' ') || (*s == '\t'))
	s++;		/* delete whitespace */
    if (!strncasecmp(s,"end group",9))
	return(FALSE);
    if (!strncasecmp(s,"URL:",4)) {
	/* description defaults to name */
	if (sg_line_desc)
	    e = sg_add_ent(2,sg_line_desc);
	else
	    e = sg_add_ent(2,s);
	sg_ents[e].restrict = savestr(s);
    } else {
      switch (*s) {
	case '#':	/* comment */
	    break;
	case ':':	/* relative to sg_root */
	case '~':	/* ...or full file names */
	case '%':
	case '/':
	case '-':	/* label within same file (treated like filename) */
	    /* description defaults to name */
	    if (sg_line_desc)
		e = sg_add_ent(2,sg_line_desc);
	    else
		e = sg_add_ent(2,s);
	    sg_ents[e].restrict = savestr(s);
	    break;
	case '@':	/* virtual newsgroup file */
	    /* description defaults to name */
	    if (sg_line_desc)
		e = sg_add_ent(4,sg_line_desc);
	    else
		e = sg_add_ent(4,s+1);
	    sg_ents[e].restrict = savestr(s+1);
	    break;
	case '>':
	    if (*(s+1)==':')
		return(FALSE);	/* label found, end of previous block */
	    break;	/* just ignore the line (print warning later?) */
	case '&':	/* special textual type */
	    sg_special(sg_line_desc,s+1);
	    break;
	default:
	    /* if there is a description, this must be a restriction list */
	    if (sg_line_desc) {
		e = sg_add_ent(3,sg_line_desc);
		sg_ents[e].restrict = savestr(s);
		break;
	    }
	    /* one or more newsgroups instead */
	    sg_use_group_line(s);
	    break;
      }
    }
    return(TRUE);	/* continue reading */
}

/* interprets a line of newsgroups, adding or subtracting each pattern */
/* Newsgroup patterns are separated by spaces and/or commas */
void
sg_use_group_line(line)
char *line;
{
    char *s,*p;
    char ch;

/*    printf("sg_use_group_line(%s) called\n",line); /* */
    s = line;
    if (!s || !*s)
	return;
    /* newsgroup patterns will be separated by space(s) */
    /* July 27, 1993: why not allow commas too? */
    while (*s) {
	for (p=s;*p && ((*p != ' ') && (*p != ',')) ;p++)
	    ; /* EMPTY */
	ch = *p;
	*p = '\0';
	sg_use_pattern(s);
	*p = ch;
	s = p;
	while ((*s == ' ') || (*s == ','))
	    s++;
    }
}

/* edits the current file */
/* consider some kind of goto-label functionality later? */
void
sg_edit_file()
{
    char *fname;	/* initialized iff used */
    char *s;
    int i;
    char lbuf[1024];	/* hold the filename */
    char wherebuf[1024];	/* where were we? */

    for (i=sg_num_contexts-1;i>=0;i--)
	if ((fname=sg_contexts[i].tmpfilename) ||	/* EQUALS */
	   (fname=sg_contexts[i].filename)) {	/* EQUALS */
		if (!strncasecmp(fname,"URL:",4))
		    continue;	/* don't edit a URL directly */
		else
		    break;
	}
    if (i<0) {	/* no file found */
	return;
    }
    if (!fname || !*fname)
	return;		/* empty, do nothing (error later?) */
    strcpy(lbuf,filexp(fname));
    sprintf(cmd_buf,"%s %s",
	filexp(getval("VISUAL",getval("EDITOR",defeditor))),lbuf);
    printf("\nEditing group scan topic file %s:\n%s\n",fname,cmd_buf) FLUSH;
    resetty();			/* make sure tty is friendly */
    getwd(wherebuf);	/* find out where we are */
    s = rindex(lbuf,'/');
    if (!s)
/* consider an error message later? */
	return;		/* error, but not critical */
    if (s == lbuf)	/* wow, file in root directory */
	s++;		/* will need the slash */
    *s = '\0';	/* cut off after directory name */
    if (chdir(lbuf)) {
	printf(nocd,lbuf) FLUSH;
	return;
    }
    doshell(sh,cmd_buf);/* invoke the shell */
    if (chdir(wherebuf)) {
	printf(nocd,wherebuf) FLUSH;
	return;
    }
    noecho();			/* and make terminal */
    crmode();			/*   unfriendly again */
}

void
sg_special(desc,line)
char *desc;	/* description or NULL */
char *line;	/* remainder of special commmand */
{
    char *s;
    long e;

    if (!line || !*line)
	return;		/* show error later? */
    if (!strncasecmp(line,"text",4)) {
	s = line + 4;
	/* for now, there is only the plain text type */
	/* later, try looking for things like "text/plain" or others? */
	while (*s && (*s != ' ') && (*s != '\t'))
	    s++;	/* skip to whitespace */
	while ((*s == ' ') || (*s == '\t'))
	    s++;	/* skip whitespace */
	if (!*s)
	    return;	/* complain later? */
	if (desc)
	    e = sg_add_ent(5,desc);
	else
	    e = sg_add_ent(5,s);
	sg_ents[e].restrict = savestr(s);
	return;
    }
    printf("Bad special group scan command: |%s|\n",line) FLUSH;
}

/* copied from svfile.c because I'm lazy */
/* returns filename after interpreting special stuff like :name */
/* also grabs URLs and translates them into temporary filenames */
/* returns a saved string, or Nullch on an error */
char *
sg_interp_fname(fname)
char *fname;
{
    char *s,*p;
    int i;
    bool flag;
    static char lbuf[LBUFLEN];

    if (!strncasecmp(fname,"URL:",4)) {
#ifdef USEURL
	s = temp_filename();
	flag = url_get(fname+4,s);
	if (flag) {
	    return(s);
	} else
	    return(NULL);
#else
	printf("This copy of strn does not have URL support.\n") FLUSH;
	return(NULL);
#endif
    }
    s = fname;
    if (*s == ':') {	/* relative to last file's directory */
	if (sg_num_contexts==0) {
	    /* this could happen if a virtual group is started
	     * from the article level.
	     */
	    printf("(:file found without group scan context)\n") FLUSH;
	    printf("Ignoring...\n") FLUSH;
	    return(Nullch);
	}
	p = sg_contexts[sg_num_contexts-1].filename;
	for (i=sg_num_contexts-1;i>=0;i--)
	    if ((p=sg_contexts[i].filename))	/* EQUALS */
		break;
	if (i<0) {	/* no file found */
	    return(Nullch);
	}
	strcpy(lbuf,p);
	p = rindex(lbuf,'/');
	if (!p) {
	    printf("(error: current location has no '/')\n") FLUSH;
	    return(Nullch);
	}
	p++;					/* point to after slash */
	strcpy(p,s+1);
	return(savestr(lbuf));
    }
    return(savestr(s));
}

void
sg_show_text(fname)
char *fname;		/* may also be a URL */
{
    char *real_fname;	/* possibly translated to a temp name */
    char *pager;
    char lbuf[LBUFLEN];

    real_fname = sg_interp_fname(fname);
    if (!real_fname) {
	printf("File could not be shown.\n");
	printf("Press any key to continue.\n");
	(void)get_anything();
	eat_typeahead();
	return;
    }
    pager = getval("SCANPAGER",getval("PAGER","more"));
    sprintf(lbuf,"%s %s",pager,real_fname);
    resetty();
    doshell(sh,filexp(lbuf));
    noecho();
    crmode();
    (void)get_anything();
    /* remove any temporary file(s) */
    if (strnEQ(real_fname,"/tmp/strn",9)) {
	UNLINK(real_fname);
    }
    free(real_fname);
}
#endif /* SCAN */
