/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*             Author :  Paul Taylor and Alan W Black                    */
/*             Date   :  May 1996                                        */
/*-----------------------------------------------------------------------*/
/*             EST_Wave Class header file                                */
/*                                                                       */
/*=======================================================================*/
#ifndef __Wave_H__
#define __Wave_H__

#include <stdio.h>
#include "EST_String.h"
#include "EST_rw_status.h"
#include "EST_String.h"
#include "EST_types.h"

#define Wave_DEFAULT_SR 16000 // when nothing at all is specified elsewhere

class EST_Wave {
 private:
    short *s; // data

    int bo;
    int p_num_channels;
    int p_num_samples;
    int p_sample_rate;
    int p_word_size;

    int default_sample_rate;
    int default_num_channels;

    EST_String default_sample_type;
    EST_String p_file_type;
    EST_String p_sample_type;
    EST_String p_name;
    
    void default_vals();
    void free_wave();
    void copy(const EST_Wave &w);
    void copy_setup(const EST_Wave &w);
public:
    EST_Wave();
    EST_Wave(const EST_Wave &a);
    ~EST_Wave();

    void clear() {free_wave();}

    short operator()(int i, int channel) const
       { return s[(i*p_num_channels)+(channel-1)]; }
    short operator()(int i) const
       { return s[(i*p_num_channels)+(1-1)]; }
    short &a(int i, int channel = 1)
       { return s[(i*p_num_channels)+(channel-1)]; }
    short *data() const { return s; }

    int num_samples() const { return p_num_samples;}
    int num_channels() const { return p_num_channels; }
    int sample_rate() const { return p_sample_rate; }
    float t(int i) const;

    EST_String sample_type() const { return p_sample_type; }
    EST_String file_type() const { return p_file_type; }
    EST_String name() const { return p_name; }

    void set_num_samples(const int n);
    void set_num_channels(const int n);
    void set_sample_rate(const int n);
    void set_file_type(const EST_String &ft) { p_file_type = ft; }
    void set_data(short *d,int num_samples,int sr,int nc);
    void set_data_copy(short *d,int num_samples,int sr,int nc);

    void set_default_num_channels(const int n);
    void set_default_sample_rate(const int n);
    void set_name(const EST_String &n) {p_name = n;}

    void bo_swap();
    void resize(int n);

//    float length(); // REORG - superceded by end()
    int length() const { return num_samples();}
    float end(); // length in seconds
    
    EST_read_status load(const EST_String &name, int offset=0, int length = 0);

    EST_read_status load_file(const EST_String &name, const EST_String
			      filetype, int sample_rate, const
			      EST_String sample_type, int bo, int nc,
			      int offset = 0, int length = 0);

    EST_write_status save(const EST_String &name, EST_String
			  EST_filetype = "");

    EST_write_status save_file(const EST_String
			       &name, EST_String filetype,
			       EST_String sample_type, int bo,
			       int offset, int length);
    
    void resample(int rate);
    void rescale(float gain,int normalize=0);

    EST_Wave& operator = (const EST_Wave& w);
    EST_Wave& operator +=(const EST_Wave &a);  // add to existing wave in serial (concat)
    EST_Wave& operator |=(const EST_Wave &a);  // add Wave in parallel (add new channel)
    friend ostream& operator << (ostream& s, const EST_Wave &sig);
};

typedef EST_TList<EST_Wave> EST_WaveList;

#endif /* __Wave_H__ */
