/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, and modify this software and its            */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Paul Taylor                              */
/*                    Date   :  February 1996                            */
/*-----------------------------------------------------------------------*/
/*                    Event RFC Utilities                                */
/*                                                                       */
/*=======================================================================*/

#include "EST_String.h"
#include "EST.h"
#include "EST_Event.h"
#include "EST_rfc.h"
#include "EST_math.h"
#include "EST_Token.h"

int add_dummy_silence(EST_Stream &rfc_ev);
int legal_event(EST_Track &fz, EST_Stream_Item &e);
int single_event(EST_Stream &ev_list);
int comp_extract(EST_Track &fz, EST_Track &part, float start, float end);
int conn_ends(EST_Stream &ev);

void add_neighbours(EST_Stream &rfc_ev);

// find event portions of fz in contour, cut out, and send one by one
// to individual labeller.


EST_Stream rfc_to_tilt(EST_Stream &ev_rfc)
{
    EST_Stream_Item *e, newsi;
    EventSI rfc;
    EventTilt *tilt;
    EST_Stream ev_tilt;
    float a_tilt, d_tilt, t_tilt;

    cout << "printing events\n";
    print_rfc_events(ev_rfc);

    if (ev_rfc.stream_name() != "rfc_event")
    {
	cerr << "Can't create Tilt parameters stream type: " 
	    << ev_rfc.stream_name() << endl;
	return ev_rfc;
    }
    add_neighbours(ev_rfc);

    for (e = ev_rfc.head(); e != 0; e = next(e))
    {
	tilt = new EventTilt;
	tilt->init();
	rfc = *RFCS(*e);
	cout << "new rfc = " << rfc << endl;
	if (pos_event(*e))
	{
	    cout << "RFC\n" << rfc << endl;
	    a_tilt = (float)(rfc.rise_amp() + rfc.fall_amp()) / 
		(float)(rfc.rise_amp() - rfc.fall_amp());
	    d_tilt = (float)(rfc.rise_dur() - rfc.fall_dur()) / 
		(float)(rfc.rise_dur() + rfc.fall_dur());

	    cout << "a tilt " << a_tilt << " d_tilt " << d_tilt << endl;
	    t_tilt = (a_tilt + d_tilt) / 2;
	    if (isnan(t_tilt))
		t_tilt = 0.0;
	    cout << "t tilt " << t_tilt << endl;

	    tilt->set_amp(-rfc.fall_amp() + rfc.rise_amp());
	    tilt->set_dur(rfc.fall_dur() + rfc.rise_dur());
	    tilt->set_tilt(t_tilt);
	    tilt->set_pos(0.0);
	    cout << "t amp " << tilt->amp() << endl;
	    cout << "rfc amp " << -rfc.fall_amp() + rfc.rise_amp() << endl;
	}
	cout << "start amp" << rfc.start_f0() << endl;
	tilt->set_start_pos(rfc.start_pos());
	tilt->set_start_f0(rfc.start_f0());
	
	newsi = *e;
	cout << "un tilt: " << *e << "f:" << (int)e->feature("pos") << endl;
	cout << "copied tilt: " << newsi << "f:" << (int)newsi.feature("pos") << endl;
	newsi.set_contents(tilt, gc_eventtilt);
	ev_tilt.append(newsi);
    }
    write_tilt_fields(ev_tilt);
    ev_tilt.set_stream_name("tilt_event");

    cout << "Last " << ev_tilt;
    
    return ev_tilt;
}

EST_Stream tilt_to_rfc(EST_Stream &ev_tilt)
{
    EST_Stream_Item *e, newsi;
    EventSI *rfc;
    EventTilt tilt;
    EST_Stream ev_rfc;
    float a_rise, d_rise;
//    const float MIN_RFC_SIZE = 0.01; // 10ms minimum on element size

    if (ev_tilt.stream_name() != "tilt_event")
    {
	cerr << "Can't create rfc parameters stream type: " 
	    << ev_tilt.stream_name() << endl;
	return ev_tilt;
    }
    
    for (e = ev_tilt.head(); e != 0; e = next(e))
    {
	rfc = new EventSI;
	rfc->init();
	tilt = *TiltC(*e);
	cout << "mapping " << tilt << "name " << e->name() << endl;;
	if (pos_event(*e))
	{
	    a_rise = tilt.amp() * (1 + tilt.tilt()) / 2.0;
	    d_rise = tilt.dur() * (1 + tilt.tilt()) / 2.0;

	    cout << "D_rise = " << d_rise << endl;

	    if ((tilt.tilt() < 0.9) && (tilt.tilt() > -0.9))
	    {
		rfc->set_peak_f0(a_rise + tilt.start_f0());
		rfc->set_peak_pos(d_rise + tilt.start_pos());
	    }
	    else 
	    {
		rfc->set_peak_f0(0.0);
		rfc->set_peak_pos(0.0);
	    }
	}
	rfc->set_start_pos(tilt.start_pos());
	rfc->set_start_f0(tilt.start_f0());
	
	newsi = *e;
	newsi.set_contents(rfc, gc_eventsi);
	ev_rfc.append(newsi);
    }
    write_tilt_fields(ev_rfc);
    ev_rfc.set_stream_name("rfc_event");

    for (e = ev_rfc.head(); next(e) != 0; e = next(e))
    {
	if (con_event(*e))
	    RFCS(*e)->type = "CONN";
	else if (sil_event(*e))
	    RFCS(*e)->type = "SIL";
	else
	{
	    if (RFCS(*e)->peak_pos() > 0.0)
		RFCS(*e)->type = "RISEFALL";
	    else if (RFCS(*e)->start_f0() < RFCS(*next(e))->start_f0())
		RFCS(*e)->type = "RISE";
	    else 
		RFCS(*e)->type = "FALL";
	}
    }
    if (sil_event(*e)) // now deal with last element
	RFCS(*e)->type = "SIL";
    else
	cerr << "stream doesn't end in silence: unpredictable behaviour\n";


    EventSI *p_ev = 0;
    for (e = ev_rfc.head(); next(e) != 0; e = next(e))
    {
	RFCS(*e)->sp = p_ev;
	p_ev = RFCS(*e);
    }

    EventSI *n_ev = 0;
    for (e = ev_rfc.tail(); prev(e) != 0; e = prev(e))
    {
	RFCS(*e)->sn = n_ev;
	n_ev = RFCS(*e);
    }

    return ev_rfc;
}

static int report_rfc_error(int error, EST_Stream_Item *e);

int validate_rfc_stream(EST_Stream &ev)
{
    EST_Stream_Item *e;
    int error = 0;
    float min_valid_f0 = 1.0;
    float prev_end = 0.0;

    for (e = ev.head(); e != 0; e = next(e))
    {
	if (e->end() <= prev_end)
	{
	    report_rfc_error(4, e);
	    error = 4;
	    continue;
	}
	prev_end = e->end();
	if (e->contents() == 0)
	{
	    report_rfc_error(1, e);
	    error = 1;
	    continue;
	}
	if (e->name() == "sil")
	    continue;
	if (RFCS(*e)->start_f0() < min_valid_f0)
	{
	    report_rfc_error(2, e);
	    error = 2;
	    continue;
	}
	if ((RFCS(*e)->type == "RISEFALL") &&(prev(e) != 0))
	    if (prev(e)->end() > RFCS(*e)->peak_pos())
	    {
		report_rfc_error(3, e);
		error = 3;
		continue;
	    }
    }
    return error;
}

static int report_rfc_error(int error, EST_Stream_Item *e)
{
    switch (error)
    {
    case 1:
	cerr << "No val() field set for element " << *e << endl;
	break;
    case 2:
	cerr << "F0 start value too low: " << *RFCS(*e) << endl;
	break;
    case 3:
	cerr << "Peak position is before Event start " << *RFCS(*e) << endl;
	break;
    case 4:
	cerr << "Event has end before previous event end: " << *e << " :" <<
	    *RFCS(*e) << endl;
	break;
    }
    return error;
}

void print_rfc_events(EST_Stream &ev)
{
    EST_Stream_Item *e;

    if (ev.stream_name() != "rfc_event")
    {
	cerr << "Can't create print rfc info from stream type: " 
	    << ev.stream_name() << endl;
	return;
    }
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	cout.precision(4);
	cout.width(6);
	cout.setf(ios::left,ios::adjustfield);
	if (pos_event(*e))
	{
	    cout << e->end() << "\t121\t" << e->name() << "; ";
	    cout << RFCS(*e)->rise_amp() << " " << RFCS(*e)->rise_dur() << " ";
	    cout << RFCS(*e)->fall_amp() << " " << RFCS(*e)->fall_dur()<<" : ";
	    cout << RFCS(*e)->start_f0() << " " << RFCS(*e)->start_pos()<< " ";
	    cout << RFCS(*e)->type << "\n";
	}
	else 
	{
	    cout << e->end() << "\t121\t" << e->name();
	    cout << RFCS(*e)->amp() << " " << RFCS(*e)->dur() << " ";
	    cout << RFCS(*e)->type << "\n";
	}
    }
    return;
}

void print_rfc2_events(EST_Stream &ev)
{
    EST_Stream_Item *e;

    if (ev.stream_name() != "rfc_event")
    {
	cerr << "Can't create print rfc info from stream type: " 
	    << ev.stream_name() << endl;
	return;
    }
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	cout.precision(4);
	cout.width(6);
	cout.setf(ios::left,ios::adjustfield);
	cout << e->end() << "\t121\t" << e->name() << "; ";
	cout << RFCS(*e)->start_f0() << " " << RFCS(*e)->peak_pos() << " ";
	cout << RFCS(*e)->peak_f0() << " type:" << RFCS(*e)->type << endl;
    }
}

void print_rfc3_events(EST_Stream &ev)
{
    EST_Stream_Item *e;

    if (ev.stream_name() != "rfc_event")
    {
	cerr << "Can't create print rfc info from stream type: " 
	    << ev.stream_name() << endl;
	return;
    }

    cout << "#\n";
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	cout.precision(4);
	cout.width(6);
	cout.setf(ios::left,ios::adjustfield);

	if (RFCS(*e)->type == "RISE")
	{
	    cout << e->end() << "\t121\t" << "rise; ";
	    cout << RFCS(*e)->start_f0() <<endl;
	}
	else if (RFCS(*e)->type == "FALL")
	{
	    cout << e->end() << "\t121\t" << "fall; ";
	    cout << RFCS(*e)->start_f0() <<endl;
	}
	else if (RFCS(*e)->type == "RISEFALL")
	{
	    cout << RFCS(*e)->peak_pos() << "\t121\t" << "rise; ";
	    cout << RFCS(*e)->start_f0() <<endl;

	    cout << e->end() << "\t121\t" << "fall; ";
	    cout << RFCS(*e)->peak_f0() <<endl;
	}
	else
	{
	    cout << e->end() << "\t121\t" << e->name() << "; ";
	    cout << RFCS(*e)->start_f0() <<endl;
	}
    }
}

void print_tilt_events(EST_Stream &ev)
{
    EST_Stream_Item *e;
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	cout.precision(4);
	cout.width(6);
	cout.setf(ios::left,ios::adjustfield);
	if (pos_event(*e))
	{
	    cout << e->end() << "\t121\t" << e->name() << "; ";
	    cout << TiltC(*e)->amp() << " " << TiltC(*e)->dur() << " ";
	    cout << TiltC(*e)->tilt() << " " << TiltC(*e)->pos() << " : ";
	    
	    cout << TiltC(*e)->start_f0() << " "; 
	    cout << TiltC(*e)->start_pos() << "\n";
	}
    }
}

// Write values from RFC structure to field string
void write_tilt_fields(EST_Stream &ev)
{
    EST_Stream_Item *e;
    EST_String f;
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	cout << "e = " << *e << endl;
	if (pos_event(*e))
	{
	    cout << "pos tilt\n";
	    f = "tilt: " + ftoString(TiltC(*e)->start_f0(), 3, 8, 1) + " " +
		ftoString(TiltC(*e)->amp(), 3, 6, 1) + " " +
		    ftoString(TiltC(*e)->dur(), 3, 6, 1) + " " +
			ftoString(TiltC(*e)->tilt(), 3, 6, 1) + " " +
			    ftoString(TiltC(*e)->pos(), 3, 6, 1);
	}

	else if (con_event(*e))
	    f = "  tilt: " + ftoString(TiltC(*e)->start_f0(), 3, 8, 1);
	else
	    f = "tilt: " + ftoString(TiltC(*e)->start_f0(), 3, 8, 1);

//	cout << "adding " << f << endl << endl;

	e->fields.clear();
	e->fields.append(f);
    }
}

// write tilt params with relative rather than absolute start f0 values
void write_tilt2_fields(EST_Stream &ev)
{
    EST_Stream_Item *e;
    EST_String f;
    float rel_f0;

    for (e = ev.head(); e != 0; e = next(e))
    {
	rel_f0 = (next(e) == 0) ? 0.0 : TiltC(*next(e))->start_f0() - TiltC(*e)->start_f0();

	if (pos_event(*e))
	    f = "tilt: " + ftoString(rel_f0, 3, 8, 1) + " " +
		ftoString(TiltC(*e)->amp(), 3, 6, 1) + " " +
		    ftoString(TiltC(*e)->dur(), 3, 6, 1) + " " +
			ftoString(TiltC(*e)->tilt(), 3, 6, 1) + " " +
			    ftoString(TiltC(*e)->pos(), 3, 6, 1);

	else if (con_event(*e))
	    f = "  tilt: " + ftoString(rel_f0, 3, 8, 1);
	else
	    f = "tilt: " + ftoString(rel_f0, 3, 8, 1);

	e->fields.append(f);
    }
}

void write_rfc_fields(EST_Stream &ev, EST_String format)
{
    EST_Stream_Item *e;
    EST_String f;
    const int pres = 3;
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	cout.width(6);
	cout.precision(4);
	if (format == "rfc1")
	{
	if (con_event(*e))
		f = "  RFC1: " + ftoString(RFCS(*e)->rise_amp(), pres, 8, 1);
	    else if (sil_event(*e))
		f = "RFC1: " + ftoString(RFCS(*e)->rise_amp(), pres, 8, 1);
	    else
		f = "RFC1: "+ftoString(RFCS(*e)->rise_amp(), pres, 8, 1) +" " +
		    ftoString(RFCS(*e)->rise_dur(), pres, 6, 1) + " " +
			ftoString(RFCS(*e)->fall_amp(), pres, 7, 1) + " " +
			    ftoString(RFCS(*e)->fall_dur(), pres, 6, 1);
	}
	else if (format == "rfc2")
	{
	    if (con_event(*e))
		f = "  rfc2: " + ftoString(RFCS(*e)->start_f0(), pres, 8, 1);
	    else if (sil_event(*e))
		f = "rfc2: " + ftoString(RFCS(*e)->start_f0(), pres, 8, 1);
	    else
		f = "rfc2: "+ftoString(RFCS(*e)->start_f0(), pres, 8, 1) +" " +
		    ftoString(RFCS(*e)->peak_pos(), pres, 6, 1) + " " +
			ftoString(RFCS(*e)->peak_f0(), pres, 7, 1);
	}
	e->fields.append(f);
    }
}

void add_neighbours(EST_Stream &rfc_ev)
{
    // set up next and previous pointers in events
    EST_Stream_Item *e;
//    e = rfc_ev.head();
//    RFCS(*e)->sp = 0;
    for (e = rfc_ev.tail(); prev(e) != 0; e = prev(e))
	RFCS(*e)->sp = RFCS(*prev(e));
    RFCS(*e)->sp = 0;
    for (e = rfc_ev.head(); next(e) != 0; e = next(e))
	RFCS(*e)->sn = RFCS(*next(e));
    RFCS(*e)->sn = 0;

}

int read_rfc_fields(EST_Stream &ev)
{
    const int NUM_FIELDS = 6;
    EST_Stream_Item *e;
    EST_String str[NUM_FIELDS];
    EventSI *p;
    EST_String f;

    if (!sil_event(*ev.tail()))
    {
	cerr << "Error: event stream does not end in silence\n";
	cerr << "Unpredictable consequences for further processing\n";
	return -1;
    }
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	p = new EventSI;
	e->set_contents(p, gc_eventsi);
	f = e->fields.first();
	if (f.contains(RXwhite, 0)) // remove leading whitespace
	    f = f.after(RXwhite);
	split(f, str, NUM_FIELDS, RXwhite);
	if (str[0] != "rfc2:")
	{
	    cerr << "Non rfc2 field in label stream: " << f << endl;
	    cerr << "Type was:" << str[0] <<": in field\n";
	    return -1;
	}
	RFCS(*e)->set_start_pos(e->start());
	RFCS(*e)->set_start_f0(atof(str[1]));
//	cout << "thing " << RFCS(*e)->start_f0() << endl;
	
	if (pos_event(*e))
	{
	    RFCS(*e)->set_peak_pos(atof(str[2]));
	    RFCS(*e)->set_peak_f0(atof(str[3]));
	}
	else 
	{
	    RFCS(*e)->set_peak_f0(-1.0);
	    RFCS(*e)->set_peak_pos(-1.0);
	}
    }
    add_neighbours(ev);
    
    for (e = ev.head(); next(e) != 0; e = next(e))
    {
	if (con_event(*e))
	    RFCS(*e)->type = "CONN";
	else if (sil_event(*e))
	    RFCS(*e)->type = "SIL";
	else
	{
	    if (RFCS(*e)->peak_pos() > 0.0)
		RFCS(*e)->type = "RISEFALL";
	    else if (RFCS(*e)->start_f0() < RFCS(*next(e))->start_f0())
		RFCS(*e)->type = "RISE";
	    else 
		RFCS(*e)->type = "FALL";
	}
    }
    // the last event should alway be silence, even if this is "dummy" silence
    RFCS(*e)->type = "SIL";
    ev.set_stream_name("rfc_event");
    return 1;
}

int read_tilt_fields(EST_Stream &ev)
{
    const int NUM_FIELDS = 6;
    EST_Stream_Item *e;
    EST_String str[NUM_FIELDS];
    EventTilt *p;
    EST_String f, t;
    EST_TokenStream ts;

    ts.set_SingleCharSymbols(":");
    ts.set_PunctuationSymbols("");
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	p = new EventTilt;
	e->set_contents(p, gc_eventtilt);
//	cout << "fields " << e->fields() << endl;;
	ts.open_string(e->fields.first());
	t = ts.get().string();
//	cout << "t = " << t << endl;
	if (t != "tilt")
	{
	    cerr << "Non tilt field in label stream: " << t << endl;
	    return -1;
	}
	t = ts.get().string();
//	cout << "t = " << t << endl;

	TiltC(*e)->set_start_f0(atof(ts.get().string()));	
//	cout << "name = " << e->name() << endl;
	if (pos_event(*e))
	{
//	    cout << "t = " << ts.peek().string() << endl;
	    TiltC(*e)->set_amp(atof(ts.get().string()));
//	    cout << "t = " << ts.peek().string() << endl;
	    TiltC(*e)->set_dur(atof(ts.get().string()));
//	    cout << "t = " << ts.peek().string() << endl;
	    TiltC(*e)->set_tilt(atof(ts.get().string()));
//	    cout << "t = " << ts.peek().string() << endl;
	    TiltC(*e)->set_pos(atof(ts.get().string()));
	}
	TiltC(*e)->set_start_pos(e->start());
	ts.close();
    }
    ev.set_stream_name("tilt_event");
    return 1;
}

EST_String guess_ev_format(EST_Stream &ev)
{
    const int NUM_FIELDS = 6;
    EST_Stream_Item *e;
    EST_String str[NUM_FIELDS];
    EST_String f, t;
    EST_TokenStream ts;
    
    for (e = ev.head(); e != 0; e = next(e))
    {
	f = e->fields.first();
//	cout << "guess " << f << endl;

	if (f.contains("tilt"))
	    return "tilt";
	else if (f.contains("rfc1"))
	    return "rfc1";
	else if (f.contains("rfc2"))
	    return "rfc2";

	cerr << "Unknown field format in event stream: " << f << endl;
	return t;
    }
    return "";
}

int read_ev_fields(EST_Stream &ev)
{
    EST_String lab_format;

    lab_format = guess_ev_format(ev);
    if (lab_format == "tilt")
	return read_tilt_fields(ev);
    else if (lab_format == "rfc2")
	return read_rfc_fields(ev);
    
    cerr << "Can;t read field type: " << lab_format << endl;
    return -1;
}

EST_Stream_Item zero_tilt()
{
    EST_Stream_Item e;
    //    EST_String x = "0 0 0 0 100 0";
    e.init("Event");
    e.set_name("pos");
    e.fields.append(" tilt: 0 0 0 0");
    //    e.set_field_names(x);
    return e;
}

void strip_silence(EST_Stream &ev_list)
{
    EST_Stream_Item *e, *r;;
    for (e = ev_list.tail(); e != 0; e = prev(e))
	if (sil_event(*e))
	{
	    r = e;
	    e = next(e);
	    ev_list.remove(r);
	}
}

EST_Stream tune_values_old(EST_Stream &ev_list)
{
    EST_Stream_Item *e, tmp;
    EST_Stream tune;
    
    int event_count = 0;
    strip_silence(ev_list);
    
    for (e = ev_list.tail(); e != 0; e = prev(e))
	if (pos_event(*e))
	    event_count++;
    
    //    cout << "Event count " << event_count << endl;
    //    cout << "events " << ev_list;
    
    if (event_count == 0)
    {
	tmp = zero_tilt();
	tune.append(tmp);
	tune.append(tmp);
    }
    else if (event_count == 1)
    {
	for (e = ev_list.tail(); e != 0; e = prev(e))
	    if (pos_event(*e))
	    {
		tune.append(*e);
		break;
	    }
	tmp = zero_tilt();
	tune.append(tmp);
	
    }
    else 
    {
	e = ev_list.tail(); // check if last event is a boundary
	if (pos_event(*e))
	{
	    tune.append(*e);
	    e = prev(e);
	}
	else 
	{
	    //	    tmp = zero_tilt();
	    //	    tune.append(tmp);
	    //	    tmp = 
	    tune.append(zero_tilt());
	}
	for (; e != 0; e = prev(e))
	    if (pos_event(*e))
	    {
		tune.prepend(*e);
		break;
	    }
	//	print_tune(tune);
    }
    
    return tune;
}

int single_event(EST_Stream &ev_list)
{
    EST_Stream_Item *e;
    int sing = 0;

    for (e = ev_list.head(); e != 0; e = next(e))
        if (pos_event(*e))
            sing = 1;

    return sing;
}

// convert tilt description to tune description by rule. The rules are
// as follows: 
// 1. The tune is made of two parts, the first representing
//    the nuclear accent and the second the boundary.

// 2. If the last event of the stream is a "b", make the second part
//    equal to this event's tilt values. Otherwise set the second part to 
//    "0 0 0 0"

// 3. Search backwards for the last accent. Make the first part equal to
//    its tilt values. If no accent is found set thsi equal to "0 0 0 0"

// 4. If the last event is "ab", set the first and second part to the
//      event's tilt values.


EST_Stream tune_values(EST_Stream &ev_list)
{
    EST_Stream_Item *e, tmp;
    EST_Stream tune;

    strip_silence(ev_list);
    e = ev_list.tail();

    // test to make sure that at least one event is present
    if (!single_event(ev_list))
      {
          tune.append(zero_tilt());
          tune.append(zero_tilt());
      }
    else if (e->name() == "ab")
    {
      tune.append(*e);
      tune.append(*e);
    }
  else if (e->name() == "a")
    {
      tune.append(*e);
      tune.append(zero_tilt());
    }
  else if (e->name() == "b")
    {
      tune.append(*e);
      e = prev(e);
      for (; e != 0; e = prev(e))
	if (pos_event(*e))
	  {
	    tune.append(*e);
	    break;
	  }
      if (e == 0)
	  tune.prepend(zero_tilt());
    }
  else if (e->name() == "c")
    {
      for (; e != 0; e = prev(e))
	if (pos_event(*e))
	  {
	    tune.append(*e);
	    break;
	  }
      tune.append(zero_tilt());
    }
  else
    cerr << "Illegal last element: " << e->name() << endl;

  return tune;
}

EST_Stream phrase_tune(EST_Stream &ev_list)
{
    EST_Stream_Item *e, tmp;
    EST_Stream part_ev, tune;
    
    for (e = ev_list.head(); e != 0; e = next(e))
    {
	if (sil_event(*e))
	    continue;
	part_ev.clear();
	for (;(e != 0) && (sil_event(*e)); e = next(e))
	{
	    tmp = *e;
	    part_ev.append(tmp);
	}
	//	cout << "Part " << part_ev;
	
	tune = tune_values(part_ev);
	print_tune(tune);
    }
    
    return ev_list;
}

void print_tune(EST_Stream &tune)
{
    EST_Stream_Item *t;
    read_tilt_fields(tune);
//    cout << "tune: " << tune << endl;
    for (t = tune.head(); t != 0; t = next(t))
	cout << TiltC(*t)->amp() << " " << TiltC(*t)->dur() << " " <<
	    TiltC(*t)->tilt() << " " << TiltC(*t)->start_f0() << " ";
    
    cout << endl;
}

int pos_event(EST_Stream_Item &evi)
{	
//    cout << "pos " << evi.name() << ":" << (int)evi.feature("pos") << endl;
    return ((int)evi.feature("pos"));
//    return ((int)evi.feature("pos") == 1);
}
int sil_event(EST_Stream_Item &evi)
{
    return ((evi.name() == "sil") || (evi.name() == "SIL"));
}
int con_event(EST_Stream_Item &evi)
{
    return ((evi.name() == "c") || (evi.name() == "C"));
}
