/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1995,1996                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                       Author: Paul Taylor                             */
/*                       Date  : April 1995                              */
/*-----------------------------------------------------------------------*/
/*                     Generate feature vectors                          */
/*                                                                       */
/*=======================================================================*/
#include "EST.h"

void cepstra_defaults(EST_Option &op);
EST_Track make_spectrogram(EST_Wave &sig, EST_Option &op);

int main(int argc, char *argv[])
{
    EST_String out_file("-");
    EST_StrList files;
    EST_Option al, op;
    int d_ord;
    int regression_length=4;
    float mean, sd;
    int i;

    EST_Wave sig;

    // cepstra[0] is the actual cepstra, [1] is the first deriv etc.
    // REORG - make these Lists or Vectors
    EST_Track cepstra_array[MAX_DELTA_ORDER+1];
    EST_Track spec;
    EST_Track energy_array[MAX_DELTA_ORDER+1]; // as above
    EST_Track final_track;			   // the result 
    
    print_help(argc, argv);
    cepstra_defaults(op);
    
    // mean, d_mean, d_d_mean etc are for cepstra and its derivatives
    // energy_mean is for energy (!) etc.

    parse_command_line(argc, argv, "-C -cepstra -cep_method:s -energy -o:s\n\
                 -esmooth:f -delta:i -N -c:s -dd:f -f:d -itype:s -l:s \n\
                 -lop:s -ltype:s -nnorm -off:f -ops -order:i -otype:s \n\
                 -range:f -s:f -mean:f -sd:f -d_mean:f -d_sd:f \n\
                 -d_d_mean:f -d_d_sd:f -itype:s -istype:s -iswap \n\
                 -ibo:s -cepstra_mean:f -cepstra_sd:f \n\
                 -cepstra_d_mean:f -cepstra_d_sd:f -cepstra_d_d_mean:f \n\
                 -cepstra_d_d_sd:f -energy_mean:f -energy_sd:f \n\
                 -energy_d_mean:f -energy_d_sd:f -energy_d_d_mean:f \n\
                 -energy_d_d_sd:f -o:s -snorm -style:s", files, al);
    
    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";
    init_lib_ops(al, op);

    if (!(al.present("-cepstra") || al.present("-energy") 
	  || al.present("-formant")))
    {
	cerr << "No feature vector types specified\n";
	exit (0);
    }
    
    // don't know if this should really be here.
    if (al.present("-lop"))
	if (init_cepstra(op, al.val("-lop", 0)) == -1)
	    exit(0);		// initialise broad class labeller.
    
    if (read_wave(sig, files.first(), al) == -1)
	exit(-1);
    
    // make un-normalised cepstra ( normalise after finding delta coeffs)
    // ------------------------------------------------------------------
    if (al.present("-cepstra"))
    {
	cepstra_array[0] = make_cepstra_unnorm(sig, op);
	EST_String temp = cepstra_array[0].name();
    
	// now the derivatives (delta_order will be 0 if none are required)
	for(d_ord=1; d_ord<=op.ival("delta_order",1); d_ord++){
	    
	    cepstra_array[d_ord] = cepstra_array[d_ord-1];
	    delta(cepstra_array[d_ord], regression_length);
	    
	}
	
	// normalise cepstra if required
	// -----------------------------
	EST_String d_name="";
	for(d_ord=0; d_ord<=op.ival("delta_order",1); d_ord++){
	    
	    if (op.val("cepstra_unnorm", 0) != "true"){
		
		if (op.val("cepstra_self_norm", 0) == "true")
		    normalise(cepstra_array[d_ord]);
		else{
		    mean = op.fval("cepstra" + d_name + "_mean", 1);
		    sd = op.fval("cepstra" + d_name + "_sd", 1);
		    for (i = 0; i < cepstra_array[d_ord].num_channels(); ++i)
			normalise(cepstra_array[d_ord], mean, sd, i);
		}
	    }
	    
	    d_name += "_d";
	    
	}				// for(d_ord=0; ...
    }    
    
    //      energy
    // -------------------
    
    if (al.present("-energy")){
	// energy
	energy_array[0] = rms_energy(sig, op.fval("bc_frame_shift", 1), 
				     op.fval("bc_frame_length", 1) );

	energy_array[0].set_field_name("energy", 0);

	// now the derivatives (delta_order will be 0 if none are required)
	for (d_ord=1; d_ord<=op.ival("delta_order",1); d_ord++){
	    energy_array[d_ord] = energy_array[d_ord-1];
	    delta(energy_array[d_ord], regression_length);
	}
	
/*	if (op.present("energy_smooth"))
	    time_mean_smooth(energy_array[0], op.fval("energy_smooth"));
*/	
	// normalise energy if required
	// ----------------------------
	EST_String d_name="";
	for(d_ord=0; d_ord<=op.ival("delta_order",1); d_ord++){
	    
	    if (op.val("energy_unnorm", 0) != "true"){
		
		if (op.val("energy_self_norm", 0) == "true")
		    normalise(energy_array[d_ord]);
		else{
		    mean = op.fval("energy" + d_name + "_mean", 1);
		    sd = op.fval("energy" + d_name + "_sd", 1);
		    for (i = 0; i < energy_array[d_ord].num_channels(); ++i)
			normalise(energy_array[d_ord], mean, sd, i);
		}
	    }
	    d_name += "_d";
	}				// for(d_ord=0; ...

    }					// if (al.present("-energy"))
    
    //      formants could go here
    // --------------------------------
    
    // slap it all into the final EST_Track and save it
    // ------------------------------------------
    
    //    final_Track.resize(cepstra_array[0].num_frames(), 0);
    
    // fill in parameters needed to save the EST_Track
    
    for (d_ord = 0; d_ord <= op.ival("delta_order",1); d_ord++)
    {
	if (al.present("-cepstra"))
	    final_track |= cepstra_array[d_ord];
	if (al.present("-energy"))
	    final_track |= energy_array[d_ord];
	//add formants when finished.
    }
    final_track.fill_time(op.fval("bc_frame_shift", 1));
    final_track.save(out_file, al.val("-otype", 0));
    return 0;
}

void override_lib_ops(EST_Option &a_list, EST_Option &al)
{ 
    // overwrite options with command line args
    a_list.override_val("bc_frame_shift", al.val("-s", 0));
    a_list.override_val("bc_frame_length", al.val("-d", 0));
    a_list.override_val("bc_frame_order", al.val("-order", 0));
    a_list.override_val("cepstra_self_norm", al.val("-snorm", 0));
    a_list.override_val("energy_self_norm", al.val("-snorm", 0));
    a_list.override_val("cepstra_unnorm", al.val("-nnorm", 0));
    a_list.override_val("energy_unnorm", al.val("-nnorm", 0));
    a_list.override_val("wave_file_type", al.val("-itype", 0));
    a_list.override_val("broad_class_type", al.val("-class", 0));
    a_list.override_val("label_file_type", al.val("-ltype", 0));
    a_list.override_val("label_offset", al.val("-off", 0));
    a_list.override_val("label_range", al.val("-range", 0));
    a_list.override_val("delta_order", al.val("-delta", 0));
    a_list.override_val("energy_smooth",al.val("-esmooth", 0));
    // 24/10/96 pault: changed it to use non-esps code - not sure
    // of consequences
    a_list.override_val("cepstral_method", al.val("-cep_method", 0)); 
    
    
    // -----------------------------------------------------------------------
    
    // backwards compatibility -- these will be removed eventually
    if( al.present("-mean") || al.present("-d_mean") || al.present("-d_d_mean") || \
       al.present("-sd") || al.present("-d_sd") || al.present("-d_d_sd") ){
	cerr << "WARNING : you are using old option names !";
	cerr << "use -cepstra_mean instead of -mean etc. etc." << endl;
	cerr << " -mean etc. will not be supported in future !!" << endl;
    }
    a_list.override_val("cepstra_mean", al.val("-mean", 0));
    a_list.override_val("cepstra_d_mean", al.val("-d_mean", 0));
    a_list.override_val("cepstra_d_d_mean", al.val("-d_d_mean", 0));
    a_list.override_val("cepstra_sd", al.val("-sd", 0));
    a_list.override_val("cepstra_d_sd", al.val("-d_sd", 0));
    a_list.override_val("cepstra_d_d_sd", al.val("-d_d_sd", 0));
    
    // -----------------------------------------------------------------------
    
    a_list.override_val("cepstra_mean", al.val("-cepstra_mean", 0));
    a_list.override_val("cepstra_d_mean", al.val("-cepstra_d_mean", 0));
    a_list.override_val("cepstra_d_d_mean", al.val("-cepstra_d_d_mean", 0));
    a_list.override_val("cepstra_sd", al.val("-cepstra_sd", 0));
    a_list.override_val("cepstra_d_sd", al.val("-cepstra_d_sd", 0));
    a_list.override_val("cepstra_d_d_sd", al.val("-cepstra_d_d_sd", 0));
    
    a_list.override_val("energy_mean", al.val("-energy_mean", 0));
    a_list.override_val("energy_d_mean", al.val("-energy_d_mean", 0));
    a_list.override_val("energy_d_d_mean", al.val("-energy_d_d_mean", 0));
    a_list.override_val("energy_sd", al.val("-energy_sd", 0));
    a_list.override_val("energy_d_sd", al.val("-energy_d_sd", 0));
    a_list.override_val("energy_d_d_sd", al.val("-energy_d_d_sd", 0));
    
    if (al.val("-style", 0) == "label")
	a_list.override_val("label_file_type", 
			    al.val("-otype", 0));
    else if (al.val("-style", 0) == "track")
	a_list.override_val("track_file_type", 
			    al.val("-otype", 0));
    else 
	a_list.override_val("tfr_file_type", 
			    al.val("-otype", 0));
}

void cepstra_defaults(EST_Option &op)
{ 
    // overwrite options with command line args
    
    // times are now all in seconds
    op.override_val("bc_frame_shift", "0.01");
    op.override_val("bc_frame_length", "0.032"); 
    op.override_val("bc_frame_order", "12");
    
    op.override_val("cepstra_mean", "0.224547");
    op.override_val("cepstra_d_mean", "0.0");
    op.override_val("cepstra_d_d_mean", "0.0");
    
    // NB: a default mean of 0.0 and sd of 0.5 means that
    //     normalisation will have no effect, since we subtract the
    //     mean and divide by twice the standard deviation
    
    op.override_val("cepstra_sd", "0.404159");
    op.override_val("cepstra_d_sd", "0.5");
    op.override_val("cepstra_d_d_sd", "0.5");
    
    op.override_val("delta_order", "0");
    
    op.override_val("energy_mean", "0.0");
    op.override_val("energy_d_mean", "0.0");
    op.override_val("energy_d_d_mean", "0.0");
    
    op.override_val("energy_sd", "0.5");
    op.override_val("energy_d_sd", "0.5");
    op.override_val("energy_d_d_sd", "0.5");
    
}

void print_help(int argc, char *argv[])
{
    for (int i = 0; i < argc; ++i)
	if (strncmp(argv[i],"-help", 2) == 0)
	{
	    cout << "Usage:\n";
	    cout << "fvgen   <input file> <output file> <options>\n";
	    cout << "use \"-\" to make input and output files stdin/out\n";
	    cout << endl;
	    cout << "-h      options help\n";
	    cout << "-ops    print relevant system operations\n";
	    cout << endl;
	    cout << "-c      <otions file> for system params \n";
	    cout << "-N      do NOT read environment operations file\n";
	    cout << endl;
	    cout << "-otype  output file type (one of 'esps'  'fspec'  'xmg'   'snns'\n";
	    cout << "                          'varsnns' 'xgraph' 'ascii', default is 'ascii')\n";
	    cout << endl;
	    cout << "-d      <float> input frame length in milliseconds\n";
	    cout << "-s      <float> input frame spacing in milliseconds\n";
	    cout << "-f      <int> sample rate in Hz (for headerless data)\n";
	    cout << "-cepstra            add cepstral coefficients\n";
	    cout << "-order  cepstral order\n";
	    cout << endl;
	    cout << "-nnorm  do not normalise feature vector\n";
	    cout << "-snorm  self normalise feature vector\n";
	    cout << endl;
	    cout << "-energy             add energy\n";
	    cout << "-delta  <int> number of delta co-efficients to append (1 or 2)\n";
	    cout << endl;
	    cout << "-cepstra_mean       cepstral mean\n";
	    cout << "-cepstra_d_mean     cepstral 1st diff mean\n";
	    cout << "-cepstra_d_d_mean   cepstral 2nd diff mean\n";
	    cout << "-cepstra_sd         cepstral standard deviation\n";
	    cout << "-cepstra_d_sd       cepstral 1st diff standard deviation\n";
	    cout << "-cepstra_d_d_sd     cepstral 2nd diff standard deviation\n";
	    cout << endl;
	    cout << "-energy_mean       energy mean\n";
	    cout << "-energy_d_mean     energy 1st diff mean\n";
	    cout << "-energy_d_d_mean   energy 2nd diff mean\n";
	    cout << "-energy_sd         energy standard deviation\n";
	    cout << "-energy_d_sd       energy 1st diff standard deviation\n";
	    cout << "-energy_d_d_sd     energy 2nd diff standard deviation\n";
	    cout << endl;
	    exit(0);
	}
    
}
