/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                         Copyright (c) 1996                            */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                       Author: Paul Taylor                             */
/*                       Date  : April 1996                              */
/*-----------------------------------------------------------------------*/
/*                    Label conversion main file                         */
/*                                                                       */
/*=======================================================================*/

#include <stdlib.h>
#include "EST_cluster.h"
#include "EST_multistats.h"

int main(int argc, char *argv[])
{
    int i;
    EST_StrList files;
    EST_String out_file;
    EST_Option al, op;
    EST_FMatrix c;
    EST_FMatrix *in;
    EST_FVector v;
    int num_pop;
    EST_TBI *p;

    print_help(argc, argv);

    parse_command_line(argc, argv, "-covar:s -d -mah -mean:s -o:s -pen -var:s",
		 files, al);

    init_lib_ops(al, op);
    out_file = al.present("-o") ? al.val("-o") : (EST_String)"-";

    num_pop = files.length();
    in = new EST_FMatrix[num_pop];

    EST_FMatrix total, popmean, d;

    if (al.present("-d"))
    {
	for (p = files.head(), i = 0; i < num_pop; ++i, p = next(p))
	{
	    if (in[i].load(files(p)) != read_ok)
		exit (0);
	    //	cout << "i = " << i << endl << in[i] << endl;
	}
	
	total = add_populations(in, num_pop);
	popmean = population_mean(in, num_pop);
	c = sample_covariance(total);
    }
    else
    {
	popmean.load(al.val("-mean"));
	c.load(al.val("-covar"));
    }
    
    if (al.present("-pen"))
    {
	v = sample_variance(total);
	d = penrose_distance(popmean, v);
	cout << "Penrose\n" << d;
    }
    else if (al.present("-mah"))
    {
	d = mahalanobis_distance(popmean, c);
	cout << "Mahalanobis\n" << d;
    }

    d.save(out_file);
    return 0;
}

void override_lib_ops(EST_Option &a_list, EST_Option &al)
{
    if (al.val("-event", 0) == "true")
	a_list.override_val("print_event", "true");
    if (al.val("-syllable", 0) == "true")
	a_list.override_val("print_syllable", "true");
    if (al.val("-link", 0) == "true")
	a_list.override_val("print_link", "true");
    if (al.val("-ins", 0) == "true")
	a_list.override_val("print_ins", "true");
    if (al.val("-del", 0) == "true")
	a_list.override_val("print_del", "true");
    if (al.val("-map", 0) == "true")
	a_list.override_val("print_map", "true");
    if (al.val("-f", 0) == "true")
	a_list.override_val("print_functions", "true");
    if (al.val("-d", 0) == "true")
	a_list.override_val("print_derivation", "true");
    if (al.val("-total", 0) == "true")
	a_list.override_val("print_total", "true");
}

void print_help(int argc, char *argv[])
{
    for (int i = 0;i < argc; ++i)
	if (strncmp(argv[i],"-help", 2) == 0)
	{
	    cout << "Usage:\n";
	    cout << "ch_lab  <input file> <output file> <options>\n";
	    cout << "use \"-\" to make input and output files stdin/out\n";
	    cout << "-N      do NOT read environment operations file\n";
	    cout << "-S      <float> frame spacing of output\n";
	    cout << "-class  <class name> Name of class defined in op file\n";
	    cout << "-color  <name> Color of output track\n";
	    cout << "-extend <factor> extend track file beyod label file";
	    cout << "-itype  <input file type 'xmg' 'ascii' 'esps' 'htk'>\n";
	    cout << "-length <ms float> length of track produced";
	    cout << "-name   <name> eg 'Fo', 'Phoneme'\n";
	    cout << "-off    vertical offset of track\n";
	    cout << "-ops    print relevant system options\n";
	    cout << "-otype  <output file type 'xmg' 'ascii' 'esps' 'htk'>\n";
	    cout << "-pad    <string> Pad with \"high\" or \"low\" values\n";
	    cout << "-pos    <label> list of labels to be regarded as `pos`";
	    cout << "-range  <float> difference between high and low values\n";
	    cout << "-shift   <float> shift the times of the labels\n";
	    cout << "-style  output style:, e.g. track\n";
	    cout << "-h      options help\n";
	    exit (0);
	}
}
