/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                    Copyright (c) 1994,1995,1996                       */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                       Author :  Simon King                            */
/*                       Date   :  October 1996                          */
/*-----------------------------------------------------------------------*/
/*                         Filter functions                              */
/*                                                                       */
/*=======================================================================*/
#include <math.h>
#include <iostream.h>
#include <fstream.h>
#include "EST_sigpr.h"

bool FIRfilter(EST_Wave &sigin, EST_FVector numerator, int delay_correction){
    
    if(delay_correction < 0){
	cerr << "Can't have negative delay !";
	return false;
    }

    if(numerator.n() <= 0){
	cerr << "Can't filter EST_Wave with given filter" << endl;
	return false;
    }

    EST_Wave sigout;
    int i,j,n=sigin.num_samples();
    sigout.resize(n);


    // could speed up here with three loops :
    // 1 for first part (filter overlaps start of wave, one 'if')
    // 2 for middle part (filter always within wave, no 'if's)
    // 3 for last part (filter ovelaps end of wave, one 'if')


    for (i = 0; i < n; ++i){
	sigout.a(i) = 0;

	int jlow=0;
	int jhigh=numerator.n();

	if(i+delay_correction >= n)
	    jlow = i + delay_correction - n + 1;

	if(i+delay_correction - jhigh < 0)
	    jhigh = i + delay_correction;

	for(j=jlow; j<jhigh; j++)
	    if( ((i+delay_correction - j) >= 0) &&
		((i+delay_correction - j) < n) )
		sigout.a(i) += (int)( (float)(sigin.a(i+delay_correction - j))
				       * numerator(j));
	
    }
    sigout.set_sample_rate(sigin.sample_rate());
    sigout.set_file_type(sigin.file_type());
    sigin = sigout;

    return true;
}



EST_FVector design_FIR_filter(EST_FVector &frequency_response, int filter_order){

    // frequency_response contains the desired filter reponse,
    // on a scale 0...sampling frequency
    
    // check filter_order is odd
    if((filter_order & 1) == 0){
	cerr << "Requested filter order must be odd" << endl;
	return EST_FVector(0);
    }
	
    // check frequency_response has dimension 2^N
    int N = fastlog2(frequency_response.num_points());
    if(frequency_response.num_points() !=  (int)pow(2,(float)N)){
	cerr << "Desired frequency response must have dimension 2^N" << endl;
	return EST_FVector(0);
    }

    int i;
    EST_FVector filt(frequency_response);
    EST_FVector dummy(frequency_response.num_points());
    for(i=0;i<dummy.num_points();i++)
	dummy(i) = 0.0;

    int e=slowIFFT(filt,dummy);
    if(e != 0){
	cerr << "Failed to design filter because FFT failed" << endl;
	return EST_FVector(0);
    }

    EST_FVector reduced_filt(filter_order);

    int mid = filter_order/2;

    reduced_filt(mid) = filt(0);
    for(i=1; i<=mid ;i++){
	// Hann window for zero ripple
	float window =  0.5 + 0.5 * cos(3.1415926535897932 *float(i) / float(mid));
	reduced_filt(mid+i) = filt(i) * window;
	reduced_filt(mid-i) = filt(i) * window;
    }

    return reduced_filt;
}



EST_FVector design_high_or_low_pass_FIR_filter(int sample_rate, int cutoff_freq, int order,
					   float gain1, float gain2){


    // change to bandpass filter .....

    if(sample_rate <= 0){
	cerr << "Can't design a FIR filter for a sampling rate of "
	     << sample_rate << endl;
	return EST_FVector(0);
    }

    int i;
    int N=10; // good minimum size

    int fft_size = (int)pow(2, N);
    while(fft_size < order*4){ // rule of thumb !?
	N++;
	fft_size = (int)pow(2, N);
    }

    // freq response is from 0 to sampling freq and therefore
    // must be symmetrical about 1/2 sampling freq

    EST_FVector freq_resp(fft_size);
    int normalised_cutoff = (fft_size * cutoff_freq)/sample_rate;
    for(i=0;i<normalised_cutoff;i++){
	freq_resp(i) = gain1;
	freq_resp(fft_size-i-1) = gain1;
    }
    for(;i<fft_size/2;i++){
	freq_resp(i) = gain2;
	freq_resp(fft_size-i-1) = gain2;
    }

    return design_FIR_filter(freq_resp,order);
}


EST_FVector design_lowpass_FIR_filter(int sample_rate, int cutoff_freq, int order){
    return design_high_or_low_pass_FIR_filter(sample_rate, cutoff_freq, order, 1.0, 0.0);
}

EST_FVector design_highpass_FIR_filter(int sample_rate, int cutoff_freq, int order){
    return design_high_or_low_pass_FIR_filter(sample_rate, cutoff_freq, order, 0.0, 1.0);
}


bool FIRlowpass_filter(EST_Wave &sigin, int cutoff_frequency, int order){

    EST_FVector filt = design_lowpass_FIR_filter(sigin.sample_rate(),cutoff_frequency,order);
    return FIRfilter(sigin,filt,filt.num_points()/2);

}


bool FIRhighpass_filter(EST_Wave &sigin, int cutoff_frequency, int order){

    EST_FVector filt = design_highpass_FIR_filter(sigin.sample_rate(),cutoff_frequency,order);
    return FIRfilter(sigin,filt,filt.num_points()/2);

}


