/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                    Copyright (c) 1994,1995,1996                       */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                       Author :  Simon King (taken from Tony Robinson) */
/*                       Date   :  July 1994                             */
/*-----------------------------------------------------------------------*/
/*                       windowing functions                             */
/*                                                                       */
/*=======================================================================*/
#include <math.h>
#include <iostream.h>
#include <fstream.h>
#include "EST_system.h"
#include "EST_sigpr.h"

EST_FVector 
Rectangular_window(int size){

    // this may be a little silly
    EST_FVector window;
    window.resize(size);
    int i;

    for(i = 0; i < size; i++)
	window(i) = 1.0;
    
    return window;
}

EST_FVector 
Triangular_window(int size){

    EST_FVector window;
    window.resize(size);
    int i,j=size/2;
    float k = 2.0 / (float)size;

    if( (size & 1) != 0) // odd
	window(j)=1.0;

    for(i = 0; i < j; i++){
	window(i) = i * k;
	window(size-i-1) = window(i);
    }
    return window;
    
}

EST_FVector 
Hanning_window(int size){

    EST_FVector window;
    window.resize(size);
    int i,j=size/2;
    float k = 2.0 * M_PI / size;

    if( (size & 1) != 0) // odd
	window(j)=1.0;

    for(i = 0; i < j; i++){
	window(i) = 0.5 - 0.5 * cos(k * (i + 0.5));
	window(size-i-1) = window(i);
    }

    return window;

}



EST_FVector 
Hamming_window(int size){

    EST_FVector window;
    window.resize(size);
    int i,j=size/2;
    float k = 2.0 * M_PI / size;

    if( (size & 1) != 0) // odd
	window(j)=1.0;

    for(i = 0; i < j; i++){
	window(i) = 0.54 - 0.46 * cos(k * (i + 0.5));
	window(size-i-1) = window(i);
    }

    return window;

}


void 
Triangular_window(EST_FVector &data){
    EST_FVector w = Triangular_window(data.n());
    data *= w;
}
    
void 
Hanning_window(EST_FVector &data){
    EST_FVector w = Hanning_window(data.n());
    data *= w;
}

void 
Hamming_window(EST_FVector &data){
    EST_FVector w = Hamming_window(data.n());
    data *= w;
}

static EST_TValuedEnumDefinition<EST_WindowType, const char *, EST_window_info> window_names[] =
{
  { wf_none,		{ "none" },				{NULL}},
  { wf_rectangle,	{"rectangle", "rect", "rectangular"},	{Rectangular_window}},
  { wf_triangle,	{"triangle", "tri", "triangular"},	{Triangular_window}},
  { wf_hanning,		{"hanning", "han"},			{Hanning_window}},
  { wf_hamming,		{"hamming", "ham"},			{Hamming_window}},
  { wf_none,		{ NULL }},
};

EST_TNamedEnumI<EST_WindowType, EST_window_info> EST_WindowTypeMap(window_names);


#if defined(INSTANTIATE_TEMPLATES)

#include "../base_class/EST_TNamedEnum.cc"

template class EST_TNamedEnumI<EST_WindowType, EST_window_info>;
template class EST_TValuedEnumI<EST_WindowType, const char *, EST_window_info>;

#endif
