 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Fri May  9 1997                              */
 /************************************************************************/
 /*                                                                      */
 /* Example of declaration and use of track maps.                        */
 /*                                                                      */
 /************************************************************************/

#include <iostream.h>
#include <stdlib.h>
#include "EST_TrackMap.h"
#include "EST_Track.h"

// A static track map which might be used by LPC analysis code to
// describe it's results.

static EST_ChannelMapping lpc_mapping =
{
  { channel_length, 0 },
  { channel_power, 1 },
  { channel_f0, 2},
  { channel_coef0,  3 },
  { channel_unknown},		// just to mark the end
};


EST_TrackMap LPCTrackMap(lpc_mapping);

int main(void)

{
  int i;

  // create a track
  EST_Track track(10,7);

  // and say it has the structre defined above.
  track.assign_map(&LPCTrackMap);

  // fields in the map will have names. field not in the map are named
  // as "trackN", we can override either like this...
  track.set_field_name("My Funky F0", 2);
  track.set_field_name("Mean Wangle Period", 5);

  // set them to values by channel type
  for(i=first_channel_type; i< num_channel_types; i++)
    {
      EST_ChannelType type = (EST_ChannelType)i;
      if (track.has_channel(type))
	track.a(3, type) = i*i/3.0;
    }

  // or by position

  track.a(3, 5) = 626262.62;

  cout.precision(4);	// print to 4 significant digits

  cout << "Frame 3 listing by position\n";

  for(i=0; i<track.num_channels(); i++)
    cout << "  Channel " << i << " is '"  << track.field_name(i) << "' = " << track.a(3,i) << "\n";
  

  cout << "Frame 3 listing by type\n";

  for(i=first_channel_type; i< num_channel_types; i++)
    {
      EST_ChannelType type = (EST_ChannelType)i;
      if (track.has_channel(type))
	cout << "  Channel type " << type << " (" << EST_default_channel_names.name(type) << ") =  " << track.a(3, type) << "\n";
      else
	cout << "  Channel type " << type << " (" << EST_default_channel_names.name(type) << ") not valid in this track\n";
    }

  // or we can look it up by name

  if (track.has_channel("Mean Wangle Period"))
    cout << "And Mean Wangle Period = " << track.a(3, "Mean Wangle Period") << "\n";
  else
    cout << "Seem to have lost Mean Wangle Period\n";

  // Now, read in a track from a file.

  EST_Track file_track;

  if (file_track.load("data/ch_track.htk") != format_ok)
    cerr << "error loading data/ch_track.htk\n";
  else
    cerr << "track load OK\n";

  // To work out what types of channel are in this track we
  // call create_map() which uses a name-to-channel mapping
  // to decide what each channel is from it's name.

  // The default mapping is EST_default_channel_names defined in EST_TrackMap.C
  // However,  Since this test track file has some unusal track names, we
  // can't use the default

  // We have a mapping which describes the test track file in 
  //	data/channel_names.map
  // so we load it (see named_enum_example.C for how this works)

  EST_ChannelNameMap TestTrackNames(channel_unknown);

  if (TestTrackNames.load("data/channel_names.map", EST_default_channel_names ) != format_ok)
    cerr << "error loading name map";
  else
    cerr << "name load OK\n";

  // OK, now we can see what channesl are in our loaded track file.
  // if we were using the default name map we would pass no argument to
  // create_map.

  file_track.create_map(TestTrackNames);

  cout << "Frame 3 listing of loaded track by type\n";

  for(i=first_channel_type; i< num_channel_types; i++)
    {
      EST_ChannelType type = (EST_ChannelType)i;
      if (file_track.has_channel(type))
	cout << "  Channel type " << type << " (" << EST_default_channel_names.name(type) << ") =  " << file_track.a(3, type) << "\n";
      else
	cout << "  Channel type " << type << " (" << EST_default_channel_names.name(type) << ") not valid in this track\n";
    }


  return (0);
}

