/*********************************************************************
 *
 *	Copyright (C) 2001 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Abstract Data Types
 * MODULE:      Skip List Test
 * FILE:        TestSkipList.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      nf      12/26/01        Initial version
 *
 * DESCRIPTION:
 *      Implements test-wrapper for the SkipList class.
 *
 * $Id: TestSkipList.java,v 1.5 2002/01/06 07:39:38 nfiedler Exp $
 *
 ********************************************************************/

package com.bluemarsh.adt.test;

import com.bluemarsh.adt.SkipList;
import java.util.*;

/**
 * Class to test the SkipList class.
 *
 * @author  Nathan Fiedler
 */
public class TestSkipList {

    /**
     * Test-wrapper for the skip list class.
     *
     * @param  args  array of command-line string arguments
     */
    public static void main(String args[]) {
        System.out.println();
        System.out.println("  + = good");
        System.out.println("  - = bad");
        System.out.println("  ! = information");
        System.out.println("  ? = verify manually");
        System.out.println();

        SkipList list = new SkipList();
        if (list.isEmpty()) {
            System.out.println("+ list is empty");
        } else {
            System.out.println("- list is not empty, what's up with that?");
            return;
        }

        System.out.println("! populating list");
        for (int i = 1; i <= 1000; i++) {
            list.insert(i, new Integer(i));
        }

        if (!list.isEmpty()) {
            System.out.println("+ list is not empty");
        } else {
            System.out.println("- list is empty, what's up with that?");
            return;
        }

        System.out.println("? list hash = " + list.hashCode());
        if (list.equals(list)) {
            System.out.println("+ list equals itself");
        } else {
            System.out.println("- list does not equal itself, eh?");
            return;
        }

        if (list.size() == 1000) {
            System.out.println("+ list has 1000 elements");
        } else {
            System.out.println("- list count is not 1000");
            return;
        }

        System.out.println("! searching for all 1000 elements");
        boolean success = true;
        for (int ii = 1; ii <= 1000; ii++) {
            Integer igr = (Integer) list.search(ii);
            if (igr == null || igr.intValue() != ii) {
                System.out.println("- could not find element " + ii);
                success = false;
                break;
            }
        }
        if (success) {
            System.out.println("+ found all 1000 elements");
        } else {
            return;
        }

        if (list.contains(new Integer(55))) {
            System.out.println("+ contains() seems to work");
        } else {
            System.out.println("- contains() failed");
            return;
        }

        List contents = new ArrayList();
        for (int ii = 1; ii <= 1000; ii += 10) {
            contents.add(new Integer(ii));
        }
        if (list.containsAll(contents)) {
            System.out.println("+ containsAll() seems to work");
        } else {
            System.out.println("- containsAll() failed");
            return;
        }

        Iterator iter = list.iterator();
        int jj = 1;
        while (iter.hasNext()) {
            Integer igr = (Integer) iter.next();
            if (igr == null || igr.intValue() != jj) {
                System.out.println("- iterator element incorrect at " + jj +
                                   ", got " + igr);
                success = false;
                break;
            }
            jj++;
        }
        if (success) {
            System.out.println("+ iterator() seems to work");
        } else {
            return;
        }

        System.out.println("! clearing list");
        list.clear();
        if (list.size() == 0 && list.isEmpty()) {
            System.out.println("+ list is empty");
        } else {
            System.out.println("- list is not empty, warning, warning!!!");
            return;
        }

        System.out.println("! making list of 20 elements");
        SkipList list2 = new SkipList();
        for (int i = 1; i <= 20; i++) {
            list2.insert(i, new Integer(i));
        }
        Object[] arr = list2.toArray();
        if (arr.length != 20) {
            System.out.println("- toArray() result does not have 20 elements");
            return;
        } else {
            System.out.print("? [" + arr[0]);
            for (int i = 1; i < arr.length; i++) {
                System.out.print(", ");
                System.out.print(arr[i]);
            }
            System.out.println("]");
            System.out.println("+ toArray() seems to work");
        }

        Integer[] intarr = (Integer[]) list2.toArray(new Integer[0]);
        if (arr.length != 20) {
            System.out.println("- toArray(Object[]) result does not have 20 elements");
            return;
        } else {
            System.out.print("? [" + arr[0]);
            for (int i = 1; i < arr.length; i++) {
                System.out.print(", ");
                System.out.print(arr[i]);
            }
            System.out.println("]");
            System.out.println("+ toArray(Object[]) seems to work");
        }

        System.out.println("! making list of 10 elements (each 10 larger than the next)");
        SkipList list3 = new SkipList();
        for (int i = 0; i <= 100; i += 10) {
            list3.insert(i, new Integer(i));
        }
        success = true;
        for (int i = 0; i < 10; i++) {
            int k = (int) (Math.random() * 100);
            Integer o = (Integer) list3.searchLeastSmaller(k);
            if (o != null &&
                o.intValue() % 10 == 0 &&
                o.intValue() - k < 10) {
                System.out.println("+ found " + o + " when looking for " + k);
            } else {
                System.out.println("- got wrong thing: " + o + " for " + k);
                success = false;
            }
        }
        // special case of exact match for searchLeastSmaller
        int k = 20;
        Integer o = (Integer) list3.searchLeastSmaller(k);
        if (o != null && o.intValue() == 20) {
            System.out.println("+ found " + o + " when looking for " + k);
        } else {
            System.out.println("- got wrong thing: " + o + " for " + k);
            success = false;
        }
        if (success) {
            System.out.println("+ searchLeastSmaller() seems to be working");
        } else {
            System.out.println("- searchLeastSmaller() is not working");
            return;
        }

        success = true;
        for (int i = 0; i < 10; i++) {
            k = (int) (Math.random() * 100);
            o = (Integer) list3.searchNextLarger(k);
            if (o != null &&
                o.intValue() % 10 == 0 &&
                k - o.intValue() < 10) {
                System.out.println("+ found " + o + " when looking for " + k);
            } else {
                System.out.println("- got wrong thing: " + o + " for " + k);
                success = false;
            }
        }

        // special case of exact match for searchNextLarger
        k = 20;
        o = (Integer) list3.searchNextLarger(k);
        if (o != null && o.intValue() == 30) {
            System.out.println("+ found " + o + " when looking for " + k);
        } else {
            System.out.println("- got wrong thing: " + o + " for " + k);
            success = false;
        }
        if (success) {
            System.out.println("+ searchNextLarger() seems to be working");
        } else {
            System.out.println("- searchNextLarger() is not working");
            return;
        }

        System.out.println();
        System.out.println("+ Tests completed successfully.");
    } // main
} // TestSkipList
