/*********************************************************************
 *
 *	Copyright (C) 1999 Nathan Fiedler
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * PROJECT:     Utilities
 * MODULE:      Soft Hashtable
 * FILE:        IntCacheTable.java
 *
 * AUTHOR:      Nathan Fiedler
 *
 * REVISION HISTORY:
 *      Name    Date            Description
 *      ----    ----            -----------
 *      NF      10/19/99        Initial version of CacheTable
 *      nf      10/23/99        Made into IntCacheTable
 *      nf      10/24/99        Rewrote to use IntHashtable
 *
 * DESCRIPTION:
 *      This class implements a hashtable that uses soft references
 *      for each key and value stored in the table. This allows the
 *      objects to be garbage-collected if the JVM runs low on memory.
 *
 ********************************************************************/

package com.bluemarsh.util;

import java.lang.ref.SoftReference;

/**
 * This class implements a hashtable, which maps keys to values. Any 
 * integer can be used as a key, and any non-null object can be used
 * as a value. The twist with
 * this hashtable is that the value objects are stored in this table
 * using a <code>SoftReference</code>. This allows the value objects
 * to be garbage-collected if the Java VM is running out of memory.
 * The caller will not know when the value object has been collected
 * until the <code>get()</code> method is called to retrieve that
 * object, in which case <code>null</code> will be returned.
 * <p>
 * This class may be useful for storing objects while enough memory is
 * available. As memory in the JVM runs low, the value objects will
 * be discarded. Thus, the name "IntCacheTable", meaning it holds values
 * so long as memory is not a problem.
 * <p>
 * An instance of IntCacheTable has two parameters that affect its
 * efficiency: its <em>capacity</em> and its <em>load factor</em>. The
 * load factor should be between 0.0 and 1.0. When the number of entries
 * in the hashtable exceeds the product of the load factor and the current
 * capacity, the capacity is increased by calling the <code>rehash</code>
 * method. Larger load factors use memory more efficiently, at the expense
 * of larger expected time per access.
 * <p>
 * If many entries are to be made into a IntCacheTable, creating it with a
 * sufficiently large capacity may allow the entries to be inserted more
 * efficiently than letting it perform automatic rehashing as needed to
 * grow the table. 
 *
 * @author  Nathan Fiedler
 * @version 1.0  10/19/99
 */
public class IntCacheTable implements Cloneable {
    /** The hash table data. This is a coalesced chain of entries. */
    protected IntHashtable cache;

    /**
     * Constructs a new, empty hashtable with a default capacity of 101
     * and load factor of 0.75.
     */
    public IntCacheTable() {
        cache = new IntHashtable();
    } // IntCacheTable

    /**
     * Constructs a new, empty hashtable with the specified initial
     * capacity and default load factor of 0.75.
     *
     * @param  initialCapacity  the initial capacity of the hashtable
     * @exception IllegalArgumentException if the initial capacity is
     *            less than zero
     */
    public IntCacheTable(int initialCapacity) {
        cache = new IntHashtable(initialCapacity);
    } // IntCacheTable

    /**
     * Constructs a new, empty hashtable with the specified initial 
     * capacity and the specified load factor.
     *
     * @param  initialCapacity  the initial capacity of the hashtable
     * @param  loadFactor       a number between 0.0 and 1.0
     * @exception  IllegalArgumentException  if the initial capacity is
     *             less than zero, if the load factor is less than or
     *             equal to zero, or if the load factor is greater than
     *             one.
     */
    public IntCacheTable(int initialCapacity, float loadFactor) {
        cache = new IntHashtable(initialCapacity, loadFactor);
    } // IntCacheTable

    /**
     * Constructs a new IntCacheTable with the given existing hashtable.
     *
     * @param  cache  Existing hashtable.
     */
    protected IntCacheTable(IntHashtable cache) {
        this.cache = cache;
    } // IntCacheTable

    /**
     * Clears this hashtable so that it contains no keys or values.
     */
    public void clear() {
        cache.clear();
    } // clear

    /**
     * Creates a shallow copy of this hashtable. All the structure of the 
     * hashtable itself is copied, but the values are not cloned. 
     * This is a relatively expensive operation.
     *
     * @return  a clone of the hashtable, or null if clone not supported
     */
    public Object clone() {
        return new IntCacheTable((IntHashtable)cache.clone());
    } // clone

    /**
     * Returns the value to which the specified key is mapped in this
     * hashtable. It is possible the value object has been garbage
     * collected and this may return null, in that case.
     *
     * @param   key   a key in the hashtable
     * @return  the value to which the key is mapped in this hashtable
     *          or null if the key is not mapped to any value
     * @see     #put
     */
    public Object get(int key) {
        Object o = cache.get(key);
        if ((o != null) && (o instanceof SoftReference)) {
            return ((SoftReference)o).get();
        }
        return o;
    } // get

    /**
     * Tests if this hashtable maps no keys to values.
     *
     * @return  <code>true</code> if this hashtable maps no keys to values;
     *          <code>false</code> otherwise.
     */
    public boolean isEmpty() {
	return cache.isEmpty();
    } // isEmpty

    /**
     * Maps the specified key to the specified value in this hashtable.
     * Neither the key nor the value can be null.
     * <p>
     * The value can be retrieved by calling the <code>get</code> method 
     * with a key that is equal to the original key. 
     *
     * @param  key    the hashtable key
     * @param  value  the value
     * @return  the previous value of the specified key in this hashtable,
     *          or null if it did not have one
     * @see     #get
     */
    public Object put(int key, Object value) {
        Object o = cache.put(key, new SoftReference(value));
        if ((o != null) && (o instanceof SoftReference)) {
            return ((SoftReference)o).get();
        }
        return o;
    } // put

    /**
     * Removes the key and its corresponding value from this 
     * hashtable. This method does nothing if the key is not
     * in the hashtable.
     *
     * @param   key  the key that needs to be removed
     * @return  the value to which the key had been mapped in this hashtable,
     *          or null if the key did not have a mapping
     */
    public synchronized Object remove(int key) {
        Object o = cache.remove(key);
        if ((o != null) && (o instanceof SoftReference)) {
            return ((SoftReference)o).get();
        }
        return o;
    } // remove

    /**
     * Returns the number of keys in this hashtable.
     *
     * @return  the number of keys in this hashtable.
     */
    public int size() {
	return cache.size();
    } // size

    /**
     * Returns the string representation of this hashtable.
     * The key/value pairs are inserted into the returned
     * string in no particular order.
     *
     * @return  string value of this table
     */
    public String toString() {
        StringBuffer buff = new StringBuffer("IntCacheTable=[");
        buff.append(cache.toString());
        buff.append("]");
        return buff.toString();
    } // toString
} // IntCacheTable
