package org.apache.turbine.services.intake;

/* ====================================================================
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 2001 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution,
 *    if any, must include the following acknowledgment:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowledgment may appear in the software itself,
 *    if and wherever such third-party acknowledgments normally appear.
 *
 * 4. The names "Apache" and "Apache Software Foundation" and 
 *    "Apache Turbine" must not be used to endorse or promote products 
 *    derived from this software without prior written permission. For 
 *    written permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache",
 *    "Apache Turbine", nor may "Apache" appear in their name, without 
 *    prior written permission of the Apache Software Foundation.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 */

// JDK classes
import java.util.*;
import java.lang.reflect.Method;
// Turbine classes
import org.apache.turbine.om.Retrievable;
import org.apache.turbine.util.RunData;
import org.apache.turbine.util.StringUtils;
import org.apache.turbine.util.ParameterParser;
import org.apache.turbine.util.Log;
import org.apache.turbine.util.TurbineException;
import org.apache.turbine.services.velocity.TurbineVelocity;
import org.apache.turbine.services.resources.TurbineResources;
import org.apache.turbine.services.intake.model.*;
import org.apache.turbine.services.pull.ApplicationTool;
import org.apache.turbine.util.pool.Recyclable;

/**
 * A Pull tool to make intake objects available to a template
 *
 * @author <a href="mailto:jmcnally@collab.net">John D. McNally</a>
 * @version $Id: IntakeTool.java,v 1.11 2001/04/19 00:46:08 jmcnally Exp $
 */
public class IntakeTool 
    implements ApplicationTool, Recyclable
{
    public static final String DEFAULT_KEY = "_0";
    private HashMap groups;
    private RunData data;
    // private boolean allValid;
    // private String omToolKey;
    // private OMTool omTool;

    /** The cache of PullHelpers. **/
    private Map pullMap = new HashMap();

    public IntakeTool()
    {
        String[] groupNames = TurbineIntake.getGroupNames();
        groups = new HashMap((int)(1.25*groupNames.length + 1));
        pullMap = new HashMap((int)(1.25*groupNames.length + 1));
        // omToolKey = TurbineResources.getString("tool.intake.om");

        for (int i=groupNames.length-1; i>=0; i--) 
        {
            pullMap.put(groupNames[i], new PullHelper(groupNames[i]));
        }
    }

    /**
     * Prepares intake for a single request
     */
    public void init(Object runData)
    {
        data = (RunData)runData;

        String[] groupNames = TurbineIntake.getGroupNames();
        for (int i=groupNames.length-1; i>=0; i--) 
        {
            try
            {
                List foundGroups = TurbineIntake.getGroup(groupNames[i])
                    .getObjects(data);
                if ( foundGroups != null ) 
                {            
                    Iterator iter = foundGroups.iterator();
                    while (iter.hasNext()) 
                    {
                        Group group = (Group)iter.next();
                        groups.put(group.getObjectKey(), group);
                    }
                }
            }
            catch(Exception e)
            {
                Log.error(e);
            }
        }        
    }

    /**
     * Implementation of ApplicationTool interface is not needed for this
     * tool as it is request scoped
     */
    public void refresh()
    {
        // empty
    }

    /**
     * Inner class to present a nice interface to the template designer
     */
    public class PullHelper
    {
        String groupName;

        private PullHelper(String groupName)
        {
            this.groupName = groupName;
        }

        public Group getDefault()
            throws Exception
        {
            return setKey(DEFAULT_KEY);
        }
        
        public Group setKey(String key)
            throws Exception
        {
            return setKey(key, true);
        }

        public Group setKey(String key, boolean create)
            throws Exception
        {
            Group g = null;

            String inputKey = TurbineIntake.getGroupKey(groupName) + key;
            if ( groups.containsKey(inputKey)) 
            {
                g = (Group)groups.get(inputKey);
            }
            else if (create)
            {
                g = TurbineIntake.getGroup(groupName); 
                groups.put(inputKey, g);
                g.init(key, data);
            }

            return g;
        }
        

        public Group mapTo(Retrievable obj)
            throws Exception
        {
        
            Group g = null;

            try 
            {
                String inputKey = TurbineIntake.getGroupKey(groupName) 
                    + obj.getQueryKey();
                if ( groups.containsKey(inputKey)) 
                {
                    g = (Group)groups.get(inputKey);
                }
                else 
                {
                    g = TurbineIntake.getGroup(groupName); 
                    groups.put(inputKey, g);
                }
                return g.init(obj);
            }
            catch(Exception e)
            {
                Log.error(e);
            }

            return null;        
        }
    }
    
    
    public Object get(String groupName)
        throws Exception
    {
        return pullMap.get(groupName);
    }

    public boolean isAllValid()
    {
        boolean allValid = true;
        Iterator iter = groups.values().iterator();
        while (iter.hasNext()) 
        {
            Group group = (Group)iter.next();
            allValid &= group.isAllValid();
        }
        return allValid;
    }

    public Group get(String groupName, String key)
        throws Exception
    {
        return ((PullHelper)get(groupName)).setKey(key);
    }

    public Group get(String groupName, String key, boolean create)
        throws Exception
    {
        return ((PullHelper)get(groupName)).setKey(key, create);
    }

    /**
     * Removes group.  Primary use is to remove a group that has
     * been processed by an action and is no longer appropriate
     * in the view (screen).
     */
    public void remove(Group group)
    {
        groups.remove(group.getObjectKey());
        group.removeFromRequest();
        TurbineIntake.releaseGroup(group);
    }


    // ****************** Recyclable implementation ************************

    private boolean disposed;

    /**
     * Recycles the object for a new client. Recycle methods with
     * parameters must be added to implementing object and they will be
     * automatically called by pool implementations when the object is
     * taken from the pool for a new client. The parameters must
     * correspond to the parameters of the constructors of the object.
     * For new objects, constructors can call their corresponding recycle
     * methods whenever applicable.
     * The recycle methods must call their super.
     */
    public void recycle()
    {
        disposed = false;
    }

    /**
     * Disposes the object after use. The method is called
     * when the object is returned to its pool.
     * The dispose method must call its super.
     */
    public void dispose()
    {
        Iterator iter = groups.values().iterator();
        while ( iter.hasNext() ) 
        {
            Group g = (Group)iter.next();
            TurbineIntake.releaseGroup(g);
        }

        groups.clear(); 
        data = null;

        disposed = true;
    }

    /**
     * Checks whether the recyclable has been disposed.
     * @return true, if the recyclable is disposed.
     */
    public boolean isDisposed()
    {
        return disposed;
    }


}






