/*
Copyright (C) 1998, 1999, 2000 Wabasoft

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License as published by the Free Software
Foundation; either version 2 of the License, or (at your option) any later version. 

This program is distributed in the hope that it will be useful, but WITHOUT ANY
WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A
PARTICULAR PURPOSE. See the GNU General Public License for more details. 

You should have received a copy of the GNU General Public License along with
this program; if not, write to the Free Software Foundation, Inc., 675 Mass Ave,
Cambridge, MA 02139, USA. 
*/

/*
Copyright (C) 2000, 2001 SMARTDATA, http://www.smartdata.ch/

More help for porting is include in the file nmport_c.c. Please read it before
porting or modify this port.

Linux port.
===========

*/

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/time.h>
#include <fcntl.h>
#include <dirent.h>
#include <unistd.h>
#include <pwd.h>
#include <time.h>
#include <errno.h>
#include <stdio.h>
#include <sys/socket.h>
#include <netdb.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include "../waba.h"

/* to have a explanation of the following structure, see the the file nmport_c.c
   and my small explanation of how to access class variable below                */
ClassHook classHooks[] =
{
  { "waba/ui/Window", WindowDestroy, 1 },
  { "waba/ui/MainWindow", NULL, 1 },
  { "waba/fx/Graphics", GraphicsDestroy, 15 },
  { "waba/fx/Image", ImageDestroy, 1 },
  { "waba/io/Catalog", CatalogDestroy, 9 },
  { "waba/io/File", FileDestroy, 1 },
  { "waba/io/Socket", SocketDestroy, 1 }, 
  { "waba/io/SerialPort", SerialPortDestroy, 1 }, 
  { NULL, NULL }
};

// Debugging Memory Sizes
#ifdef DEBUGMEMSIZE
void debugMemSize() {
        char buf[40];

        xmemzero(buf, 40);

        // get head infos
	sprintf( buf, "c:%d/%d o:%d/%d",
		 classHeapUsed, classHeapSize,
		 heap.memSize - getUnusedMem(), heap.memSize );

	// print the information
	DPRINTF( "MEM SIZE: %s\n", buf );

        }
#endif

/*------------------------------------------------------------------------

Access Class Variables from C
=============================

To access variables of waba classes from C, the following rules applies :

    - the function objectPtr() return the address of an array which map all
      the variables of the gived class
    - the 1st variable of a class is the class itself. We don't use it in
      the native functions (or, at least, I don't have found a use somewhere).
      I suppose it's the table of the methods.
    - accessing the variable number n of a class is done by accessing the
      element n of the array returned by objetPtr().
    - depending of the type of the variable, we get the correct value with
      one of the field of the type Var. Here is the definition of the type
      Var (taken from waba.c) :
        typedef union
          {
          int32 intValue;
          float32 floatValue;
          void *classRef;
          uchar *pc;
          void *refValue;
          WObject obj;
        } Var;

Hook Variables
==============

Variables need by the implementation of the native function, named in
waba "hook variables", are stocked directly after the other class
variables.  Accessing them is like accessing the class variable. Just
compute the correct index index for the 1st "hook variable" is 1 more
than the number of variable of the class. For example, if a class has
2 variables, the 1st "hook variable" is at index 3. (0 is the class
itself, 1 and 2 the variables of the class and 3 and more the hook
variables).

The number of hook variables need to be inserted inside the classHook
array.

Access parameter of a Class Method from the Native C Function
=============================================================

All the parameter of a class method is passed to the native C function
via the stack. To access them, the C function need to be declared with
one parameter of type "Var". For example :

    static Var MainWinExit( Var stack[] )

correspond to the method

    public native void exit(int exitCode);

of waba/ui/MainWindow.

Stack[0] is a reference to the object instance itself. In this case,
the object waba/ui/MainWindow. We need to use this parameter to access
also the class variables.

stack[1] represent the 1st parameter gived to the method. In our
sample, stack[1] correspond to the parameter exitCode.

--------------------------------------------------------------------------*/

/*************************************************************************
                               class waba/ui/Window
*************************************************************************/

// since Window inherits from other classes, we need to calculate the
// right base offset to start with when reading/writing to variables
int _winHookOffset = -1;

/* implementation of native waba/ui/Window._nativeCreate() */
Var WindowCreate( Var stack[] ) {
    Var v; 
    WObject win;

    // get the current class object
    win = stack[0].obj;
    if( _winHookOffset == -1 ) {

	// first time => compute the offset to go after the "class variables"

	WClass *wc;

	wc = getClass(createUtfString("waba/ui/Window"));
	_winHookOffset = 1 + wc->numVars - WOBJ_WindowHookVars;

    }

    // init the user interface and store the created main windowin the hook var
    WOBJ_WindowMain(win) = ui_init();

    // init the variables of the class
    WOBJ_ControlX(win) = 0;
    WOBJ_ControlY(win) = 0;
    WOBJ_ControlWidth(win) = hwr_width;
    WOBJ_ControlHeight(win) = hwr_height;

    /* return nothing */
    v.obj = 0;
    return v; 

}

/* implementation of the function called by the garbage collector when a
   window object is destroyed                                             */
void WindowDestroy(WObject mainWin) {

    DPUTS("??? WindowDestroy");

}

/*************************************************************************
                               class waba/ui/MainWindow
*************************************************************************/

// since MainWindow inherits from other classes, we need to calculate the
// right base offset to start with when reading/writing to variables
int _mainWinHookOffset = -1;

/* implementation of native waba/ui/MainWindow._nativeCreate() */
Var MainWinCreate( Var stack[] ) {
    Var v;

    // we use a global variable, because we need a link to the main window in ..._b.c
    globalMainWin = stack[0].obj;

    if( _mainWinHookOffset == -1 ) {

	// first time => compute the offset to go after the "class variables"

	WClass *wc;

	wc = getClass(createUtfString("waba/ui/MainWindow"));
	_mainWinHookOffset = 1 + wc->numVars - WOBJ_MainWinHookVars;

    }

    /* put the timer to 0 ( => no timer ) */
    WOBJ_MainWinTimer( globalMainWin ) = 0;

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/ui/MainWindow.exit() */
Var MainWinExit( Var stack[] ) {
    Var v;
    int16 exitCode;

    /* get the exit code from the stack */
    exitCode = stack[1].intValue;

    /* exit the user interface by giving it the exit code */
    ui_exit(exitCode);

    /* return nothing. Is this correct ? */
    v.obj = 0;
    return v;

}

/* implementation of native waba/ui/MainWindow._setTimerInterval() */
Var MainWinSetTimerInterval( Var stack[] ) {
    Var v;
    WObject win;
    gint32 nbOfMillisecond;

    /* get the current class object */
    win = stack[0].obj;

    /* get the number of millisecond we want for the timer */
    nbOfMillisecond = stack[1].intValue;

    /* set the timeout */
    ui_TimeoutSet( win, nbOfMillisecond );

    /* return nothing */
    v.obj = 0;
    return v;

}

/*************************************************************************
                               class waba/fx/Image
*************************************************************************/

/* implementation of native waba/fx/Image._nativeCreate() */
Var ImageCreate( Var stack[] ) {
    WObject image;
    Var v;
#ifdef GUI_GTK
    GdkGC *theGC;
    GdkPixmap *thePixmap;
    gint32 width, height;
    uchar *rgbbuf;
    gint x;

    /* get the Image object from the stack */
    image = stack[0].obj;

    /* get the width and height of the image */
    width = WOBJ_ImageWidth( image );
    height = WOBJ_ImageHeight( image );
    
    /* create the image pixmap */
    if( width > 0 && height > 0 ) {

      thePixmap = gdk_pixmap_new( NULL, width, height, gdk_visual_get_best_depth() );

    } else {

      thePixmap = NULL;

    }

    /* memorise the created image pixmap in the image var hooks */
    WOBJ_ImagePixmap( image ) = thePixmap;

    /* allocate the buffer to put the RGB */
    rgbbuf = xmalloc( sizeof( guchar ) * width * height * 3 );

    /* init the RGB buffer to no color */
    for( x = 0; x < (width * height * 3) ; x++ ) {
      rgbbuf[x] = 255;
    }

    /* create a new Graphics Context */
    theGC = gdk_gc_new( thePixmap );

    /* draw the rgb buffer to init the pixmap */
    gdk_draw_rgb_image( thePixmap,
			theGC,
			0, 0, width, height,
			GDK_RGB_DITHER_NORMAL,
			rgbbuf,
			width * 3 );

    /* releast the memory */
    xfree( rgbbuf );
    gdk_gc_unref( theGC );

#else

DPUTS("??? ImageCreate");

#endif

    /* return nothing */
    v.obj = 0;
    return v;

}


/* Intel-architecture getUInt32 and getUInt16 */
/* This have to do with BIG ENDIAN/LO ENDIAN  */
#define inGetUInt32(b) (uint32)( (uint32)((b)[3])<<24 | \
				 (uint32)((b)[2])<<16 | \
				 (uint32)((b)[1])<<8  | \
				 (uint32)((b)[0]) )
#define inGetUInt16(b) (uint16)( (uint16)((b)[1])<<8  | \
				 (uint16)((b)[0]) )

#ifdef GUI_GTK

/* implementation of function for loading a "windows" bitmap file         */
/* WARNING ! a "windows" bitmap file is not the same as a X11 bitmap file */
void ImageLoadBMP( WObject image, uchar *p ) {
    uint32 bitmapOffset, infoSize, width, height, bpp;
    uint32 i, x, y, compression, numColors, scanlen;
    uchar *rgbbuf;
    uchar *pos;
    uchar *b;
    uchar *bsave;
    int32 bit;
    uchar startMask;
    uchar currentMask;
    int32 currentMaskShift;
    uint32 nbUsedColors;
    uchar *theUsedColors;
    uchar *c;
    GdkGC *theGC;
    GdkPixmap *thePixmap;

    // content of a "windows" bitmap file :
    // header (54 bytes)
    // 0-1   magic chars 'BM'
    // 2-5   uint32 filesize (not reliable)
    // 6-7   uint16 0
    // 8-9   uint16 0
    // 10-13 uint32 bitmapOffset
    // 14-17 uint32 info size
    // 18-21 int32  width
    // 22-25 int32  height
    // 26-27 uint16 nplanes
    // 28-29 uint16 bits per pixel
    // 30-33 uint32 compression flag
    // 34-37 uint32 image size in bytes
    // 38-41 int32  biXPelsPerMeter
    // 32-45 int32  biYPelsPerMeter
    // 46-49 uint32 colors used
    // 50-53 uint32 important color count

    if (p[0] != 'B' || p[1] != 'M')
	return; // not a BMP file -> we don't load it

    bitmapOffset = inGetUInt32(&p[10]);
    infoSize = inGetUInt32(&p[14]);
    if (infoSize != 40)
	return; // old-style BMP -> we don't load it

    width = inGetUInt32(&p[18]);
    height = inGetUInt32(&p[22]);
    if (width > 65535 || height > 65535)
	return; // bad width/height -> we don't load it

    bpp = inGetUInt16(&p[28]);
    if (bpp != 1 && bpp != 4 && bpp != 8)
	return; // not a 2, 16 or 256 color image -> we don't load it

    compression = inGetUInt32(&p[30]);
    if (compression != 0)
	return; // compressed image -> we don't load it

    numColors = 1 << bpp;
    scanlen = ((width * bpp) + 7) / 8; // # bytes
    scanlen = ((scanlen + 3) / 4) * 4; // end on 32 bit boundary

    // colormap
    //
    // 0-3 uint32 col[0]
    // 4-7 uint32 col[1]
    // ...
    nbUsedColors = inGetUInt32(&p[50]);
    theUsedColors = xmalloc( sizeof( guchar ) * nbUsedColors * 3 );
    c = theUsedColors;
    b = (guchar *)&p[54];
    for( i = 0; i < nbUsedColors; i++ ) {
	*c++ = *b++;  /* red */
	*c++ = *b++;  /* green */
	*c++ = *b++;  /* blue */
	b++;   /* skip the non-used parameter */
    }

    /* allocate the buffer to put the image */
    rgbbuf = xmalloc( sizeof( guchar ) * width * height * 3 );

    /* Set up the RGB buffer. */
    b = (guchar *)&p[bitmapOffset];
    bsave = b;
    bit = 0;
    startMask = 0;
    for( i = 0; i < bpp; i++ ) {
	startMask = startMask >> 1;
	startMask |= 0x80;
    }
    currentMask = startMask;
    currentMaskShift = 0;
    pos = rgbbuf + width * ( height - 1 ) * 3;  /* start at last line, because the image is inversed */

    for( y = 0; y < height; y++ ) {
	for( x = 0; x < width; x++ ) {
	    guchar dummy;
	    guchar current;
	    gint tmpIndex;

	    current = *b & currentMask;
	    currentMask = currentMask >> bpp;
	    currentMaskShift += bpp;

	    tmpIndex = current >> ( 8 - currentMaskShift );

	    *pos++ = theUsedColors[ tmpIndex * 3 + 2 ];  /* Blue. */
	    *pos++ = theUsedColors[ tmpIndex * 3 + 1 ];  /* Green. */
	    *pos++ = theUsedColors[ tmpIndex * 3 + 0 ];  /* Red. */

	    if( currentMask == 0 ) {

		currentMask = startMask;
		currentMaskShift = 0;
		b++;

	    }
	}

	pos -= 2 * width * 3;
	currentMask = startMask;
	currentMaskShift = 0;

	// go to the next line
	bsave += scanlen;
	b = bsave;

    }

    /* create the image pixmap */
    if( width > 0 && height > 0 ) {

      thePixmap = gdk_pixmap_new( NULL, width, height, gdk_visual_get_best_depth() );

    } else {

      thePixmap = NULL;

    }

    /* memorise the created image pixmap in the image var hooks */
    WOBJ_ImagePixmap( image ) = thePixmap;

    /* create a new Graphics Context */
    theGC = gdk_gc_new( thePixmap );

    /* draw the rgb buffer to init the pixmap */
    gdk_draw_rgb_image( thePixmap,
			theGC,
			0, 0, width, height,
			GDK_RGB_DITHER_NORMAL,
			rgbbuf,
			width * 3 );

    /* release memory */
    xfree( theUsedColors );
    xfree( rgbbuf );
    gdk_gc_unref( theGC );

    /* update the variable of the Image class */
    WOBJ_ImageWidth(image) = width;
    WOBJ_ImageHeight(image) = height;

}

#endif

/* implementation of native waba/fx/Image._nativeLoad() */
Var ImageLoad(Var stack[]) {
    Var v;

#ifdef GUI_GTK

    WObject pathString, charArray;
    WObject image;
    int32 count;
    uint16 *chars;
    uchar p[50];
    uchar *inMemory;

    /* get the class variables from the stack */
    image = stack[0].obj;

    /* init the variable of the class */
    WOBJ_ImagePixmap(image) = NULL;
    WOBJ_ImageWidth(image) = 0;
    WOBJ_ImageHeight(image) = 0;

    /* get the name of the file to load from the parameter*/
    pathString = stack[1].obj;

    /* convert the string to an "array of char" */
    charArray = WOBJ_StringCharArrayObj( pathString );
    count = WOBJ_arrayLen(charArray);
    chars = (uint16 *)WOBJ_arrayStart(charArray);

    /* unicode => chars */
    while( count > 0 ) {

	int32 i, n;

	/* limit the copy to the size of p */
	n = sizeof( p ) - 1;
	if( n > count ) {
	    n = count;
	}

	/* copy each char from the unicode chars variable to the "normal" p variable */
	for( i = 0; i < n; i++ ) {
	    p[i] = (char)chars[i];
	}

	/* the terminator */
	p[i] = 0;

	/* update the counters */
	count -= n;
	chars += n;

    }

    /* load the file */
    inMemory = readFileIntoMemory( p, 0, NULL );
    if( inMemory == NULL ) {

	/* problem during the load => end the load */
	v.obj = 0;
	return v;

    }

    /* comvert the file loaded in memory to a RGB bitmap object */
    ImageLoadBMP( image, inMemory );

    /* free the memory by removing the file loaded just before */
    xfree( inMemory );

#else

    DPUTS("??? ImageLoad");

#endif

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Image.useImagePalette() */
Var ImageUseImagePalette( Var stack[] ) {
    Var v;
    
    DPUTS("??? ImageUseImagePalette");

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Image.setPixels() */
Var ImageSetPixels( Var stack[] ) {
    Var v;

#ifdef GUI_GTK

    WObject image, colorMapArray, pixelsArray;
    int32 bitsPerPixel, bytesPerRow, nbPixelsPerRow, numRows, yDest, numColors;
    int32 i;
    uint32 *colorMap;
    uchar *pixels;
    GdkPixmap *thePixmap;
    GdkGC *theGC;
    uchar *c;
    uchar startMask;
    uchar currentMask;
    int32 currentMaskShift;
    uint32 nbUsedColors;
    uchar *theUsedColors;
    uint32 x, y;
    uchar *rgbbuf;
    uchar *pos;
    uchar *b;

    /* get the parameters from the stack */
    v.obj = 0;
    image = stack[0].obj;
    bitsPerPixel = stack[1].intValue;
    colorMapArray = stack[2].obj;
    bytesPerRow = stack[3].intValue;
    numRows = stack[4].intValue;
    yDest = stack[5].intValue;
    pixelsArray = stack[6].obj;

    /* validate parameters */
    if (colorMapArray == 0 || pixelsArray == 0) {
      return v;
    }
    numColors = WOBJ_arrayLen(colorMapArray);
    if (bitsPerPixel == 1 && numColors == 2)
      ;
    else if (bitsPerPixel == 4 && numColors == 16)
      ;
    else if (bitsPerPixel == 8 && numColors == 256)
      ;
    else {
      return v;
    }
    if (WOBJ_arrayLen(pixelsArray) < bytesPerRow * numRows) {
      return v;
    }

    /* get the reference to the current iamge pixmap */
    thePixmap = WOBJ_ImagePixmap(image);
    if( thePixmap == NULL ) {
      return v;
    }

    /* compute the number of "pixels" per row */
    nbPixelsPerRow = bytesPerRow * 8 / bitsPerPixel;

    /* get the two parameters array */
    colorMap = (uint32 *)WOBJ_arrayStart(colorMapArray);
    pixels = (uchar *)WOBJ_arrayStart(pixelsArray);

    /* create the colormap */
    nbUsedColors = WOBJ_arrayLen( colorMapArray );
    theUsedColors = xmalloc( sizeof( guchar ) * nbUsedColors * 3 );
    if( theUsedColors == NULL ) {
      return v;
    }
    c = theUsedColors;
    for( i = 0; i < nbUsedColors; i++ ) {
	*c++ = colorMap[i] & 0xFF;             /* blue */
	*c++ = ( colorMap[i] >> 8 ) & 0xFF;    /* green */
        *c++ = ( colorMap[i]  >> 16 ) & 0xFF;  /* red */
    }

    /* allocate the buffer to put the image */
    rgbbuf = xmalloc( sizeof( guchar ) * nbPixelsPerRow * numRows * 3 );

    /* compute the mask to use */
    startMask = 0;
    for( i = 0; i < bitsPerPixel; i++ ) {
	startMask = startMask >> 1;
	startMask |= 0x80;
    }
    currentMask = startMask;
    currentMaskShift = 0;

    /* init the counters */
    pos = rgbbuf;
    b = pixels;

    /* Set up the RGB buffer */
    for( y = 0; y < numRows; y++ ) {
	for( x = 0; x < nbPixelsPerRow; x++ ) {
	    guchar dummy;
	    guchar current;
	    gint tmpIndex;

	    current = *b & currentMask;
	    currentMask = currentMask >> bitsPerPixel;
	    currentMaskShift += bitsPerPixel;

	    tmpIndex = current >> ( 8 - currentMaskShift );

	    *pos++ = theUsedColors[ tmpIndex * 3 + 2 ];  /* Blue. */
	    *pos++ = theUsedColors[ tmpIndex * 3 + 1 ];  /* Green. */
	    *pos++ = theUsedColors[ tmpIndex * 3 + 0 ];  /* Red. */

	    if( currentMask == 0 ) {

		currentMask = startMask;
		currentMaskShift = 0;
		b++;

	    }
	}

	currentMask = startMask;
	currentMaskShift = 0;

    }

    /* create a new Graphics Context */
    theGC = gdk_gc_new( thePixmap );

    /* draw the rgb buffer to init the pixmap */
    gdk_draw_rgb_image( thePixmap,
			theGC,
			0, yDest, nbPixelsPerRow, numRows,
			GDK_RGB_DITHER_NORMAL,
			rgbbuf,
			nbPixelsPerRow * 3 );

    /* release memory */
    xfree( theUsedColors );
    xfree( rgbbuf );
    gdk_gc_unref( theGC );

#else

    DPUTS("??? ImageSetPixels");

#endif

    return v;

}

/* implementation of native waba/fx/Image.free() */
Var ImageFree( Var stack[] ) {
    Var v;
    
    DPUTS("??? ImageFree"); 

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of the function called by the garbage collector when a
   Image object is destroyed                                             */
void ImageDestroy( WObject image ) {

    DPUTS("??? ImageDestroy");

}

/*************************************************************************
                               class waba/io/Socket
*************************************************************************/

// var[0] = Class
// var[1] = hook var - Socket Id (int32)
//

/* implementation of native waba/io/Socket._nativeCreate() */
Var SocketCreate(Var stack[]) {
    Var v; 
    WObject sock, host;
    UtfString hostname;
    unsigned long ipAddr;
    int port, sockfd;
    struct sockaddr_in * saddr;
    struct hostent * hptr;

    v.obj = 0;
    sock = stack[0].obj;
    host = stack[1].obj;
    port = stack[2].obj;

    saddr = (struct sockaddr_in *)malloc(sizeof(struct sockaddr_in));
    if (!saddr)
	return v;
    xmemzero(saddr, sizeof(struct sockaddr_in));

    saddr->sin_family = AF_INET;
    saddr->sin_port = htons(port);

    hostname = stringToUtf(host, STU_NULL_TERMINATE | STU_USE_STATIC);
    if (hostname.len == 0)
    {
	free(saddr);
	return v;
    }

    ipAddr = inet_addr(hostname.str);
    if (ipAddr != INADDR_NONE)
	xmemmove(&saddr->sin_addr, &ipAddr, sizeof(unsigned long));
    else
    {
	hptr = gethostbyname(hostname.str);
	if (!hptr)
	{
	    free(saddr);
	    return v;
	}
	xmemmove(&saddr->sin_addr, hptr->h_addr, hptr->h_length);
    }

    if ((sockfd = socket(AF_INET, SOCK_STREAM, 0)) < 0)
    {
	free(saddr);
	return v;
    }

    if (connect(sockfd, (struct sockaddr *)saddr, sizeof(struct sockaddr)) == -1)
    {
	close(sockfd);
	free(saddr);
	return v;
    }

    WOBJ_SocketFD(sock) = sockfd;

    return v;

}

/* implementation of function for closing a socket */
int32 mySocketClose( WObject socket ) {

    int sockfd;
    sockfd = WOBJ_SocketFD(socket);

    return (shutdown(sockfd, 2) == 0) ? 0 : 1;
}

/* implementation of the function called by the garbage collector when a
   Socket object is destroyed                                             */
void SocketDestroy( WObject socket ) {

    mySocketClose( socket );

}

/* implementation of native waba/io/Socket.close() */
Var SocketClose( Var stack[] ) {
    Var v;
    WObject socket;

    v.obj = 0;
    socket = stack[0].obj;
    v.intValue = mySocketClose(socket);

    return v;

}

/* implementation of native waba/io/Socket.isOpen() */
Var SocketIsOpen( Var stack[] ) {
    Var v; 
    struct stat buf;
    WObject socket;
    int sockfd;

    v.obj = 0;
    socket = stack[0].obj;
    sockfd = WOBJ_SocketFD(socket);

    if (fstat(sockfd, &buf) < 0)
	    return v;

    v.intValue = 1;

    return v;

}

/* implementation of native waba/io/Socket.setReadTimeout() */
Var SocketSetReadTimeout(Var stack[]) {
    Var v; 
    
    DPUTS("??? SocketSetReadTimeout");

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of function for reading/writing in a socket */
Var SocketReadWriteBytes( Var stack[], int isRead ) {
    Var v; 

    WObject socket, byteArray;
    int sock;
    int32 start, count, countSoFar, n;
    uchar * bytes;

    v.intValue = -1;

    socket = stack[0].obj;
    byteArray = stack[1].obj;
    start = stack[2].intValue;
    count = stack[3].intValue;
    sock = WOBJ_SocketFD(socket);

    if (sock == -1)
	return v;

    if (arrayRangeCheck(byteArray, start, count) == 0)
	return v;

    bytes = (uchar *)WOBJ_arrayStart(byteArray);
    bytes = &bytes[start];
    countSoFar = 0;

    while (countSoFar < count)
    {
	int chunkSize;

	chunkSize = count - countSoFar;

	if (chunkSize > 0x7000)
	    chunkSize = 0x7000;

	if (isRead)
	{
	    n = read(sock, bytes, chunkSize);
	    if (chunkSize > 0 && n == 0)
		mySocketClose(socket);
	}
	else
	    n = write(sock, bytes, chunkSize);

	if (n > 0)
	{
	    countSoFar += n;
	    bytes += n;
	}
	else
	{
	    if (countSoFar != 0)
		v.intValue = countSoFar;
	    else if (n < 0)
		v.intValue = 0;
	    else
		v.intValue = -1;
	    return v;
	}
    }

    v.intValue = count;
    return v;

}

/* implementation of native waba/io/Socket.readBytes() */
Var SocketRead( Var stack[] ) {

    return SocketReadWriteBytes( stack, 1 );

}

/* implementation of native waba/io/Socket.writeBytes() */
Var SocketWrite(Var stack[]) {

    return SocketReadWriteBytes( stack, 0 );

}

/*************************************************************************
                               class waba/fx/Sound
*************************************************************************/

// all functions static

/* implementation of native waba/fx/Sound.tone() */
Var SoundTone( Var stack[] ) {
    Var v;

    DPUTS("??? SoundTone");

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Sound.beep() */
Var SoundBeep( Var stack[] ) {
    Var v;

    DPUTS("??? SoundBeep"); 

    /* return nothing */
    v.obj = 0;
    return v;

}

/*************************************************************************
                               class waba/fx/Font
*************************************************************************/

/* no native function to implement for Font. Cool ! */

/*************************************************************************
                               class waba/fx/FontMetrics
*************************************************************************/

/* implementation of native waba/fx/FontMetrics._nativeCreate() */
Var FontMetricsCreate( Var stack[] ) {
    WObject font, fontMetrics, surface;
    ui_FontType *theFont;
    Var v;

    /* get the FontMetrics from the stack */
    fontMetrics = stack[0].obj;

    /* get the FontMetrics variable */
    font = WOBJ_FontMetricsFont(fontMetrics);
    surface = WOBJ_FontMetricsSurface(fontMetrics);

    if( font == 0 || surface == 0 ) {

	/* no fontMetric, no surface => reset the class's variables and exit */

	WOBJ_FontMetricsAscent(fontMetrics) = 0;
	WOBJ_FontMetricsDescent(fontMetrics) = 0;
	WOBJ_FontMetricsLeading(fontMetrics) = 0;

	/* return nothing */
	v.obj = 0;
	return v;

    }

    /* create the font */
    theFont = ui_FontCreate( font );

    /* update the class's variables */
    ui_FontSetProperties( theFont, fontMetrics );

    /* release the memory */
    ui_FontDelete( theFont );

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of function for getting properties of a FontMetrics */
Var FontMetricsGetWidth(int type, Var stack[]) {
    WObject font, fontMetrics, surface;
    WObject string, charArray;
    ui_FontType *theFont;
    int32 width;
    int32 start, count;
    uint16 *chars;
    Var v;

    /* get the FontMetrics from the stack */
    fontMetrics = stack[0].obj;

    /* get the FontMetrics variable */
    font = WOBJ_FontMetricsFont(fontMetrics);
    surface = WOBJ_FontMetricsSurface(fontMetrics);

    /* create the font to retreive the wanted value */
    theFont = ui_FontCreate( font );

    /* initialisation */
    width = 0;

    switch( type ) {

    case FM_STRINGWIDTH:

	/* we want the width of the string */

	/* get the string from the stack */
	string = stack[1].obj;
	if (string == 0)
	    /* no string */
	    break;

	/* convert the string to an "array of char" */
	charArray = WOBJ_StringCharArrayObj(string);
	if (charArray == 0)
	    /* no string */
	    break;

	/* get the len of the string */
	count = WOBJ_arrayLen(charArray);

	/* unicode => chars */
	chars = (uint16 *)WOBJ_arrayStart(charArray);
	while (count > 0) {

	    char buf[40];
	    int32 i, n;

	    /* limit the copy to the size of p */
	    n = sizeof(buf) - 1;
	    if (n > count)
		n = count;

	    /* copy each char from the unicode chars variable to the "normal" p variable */
	    for (i = 0; i < n; i++) {
		buf[i] = (char)chars[i];
	    }

	    /* the terminator */
	    buf[i] = 0;

	    /* update the wanted width */
	    width += ui_TextWidth( buf, theFont );

	    /* update the counters */
	    count -= n;
	    chars += n;

	  }

	break;

    case FM_CHARWIDTH:

      {

	gchar ch;
	/* we want the width of the char */

	/* get the wanted char */
	ch = stack[1].intValue;

	width = ui_CharWidth( ch, theFont );

      }

      break;

    default:
	width = 0;

    }

    /* release the memory */
    ui_FontDelete( theFont );

    /* return the wanted value */
    v.intValue = width;
    return v;

}

/* implementation of native waba/fx/FontMetrics.getTextWidth(String s) */
Var FontMetricsGetStringWidth(Var stack[]) {

  return FontMetricsGetWidth(FM_STRINGWIDTH, stack);

}

/* implementation of native waba/fx/FontMetrics.getTextWidth(char chars[], int start, int count) */
Var FontMetricsGetCharArrayWidth(Var stack[]) {

    return FontMetricsGetWidth(FM_CHARARRAYWIDTH, stack);

}

/* implementation of native waba/fx/FontMetrics.getCharWidth() */
Var FontMetricsGetCharWidth(Var stack[]) {

    return FontMetricsGetWidth(FM_CHARWIDTH, stack);
  
}

/*************************************************************************
                               class waba/fx/Graphics
*************************************************************************/

static WClass *windowClass = 0;
static WClass *imageClass = 0;

/* implementation of function for getting the type of the surface (window or image) */
int SurfaceGetType( WObject surface ) {
    WClass *wclass;

    if( surface == 0 ) {

	/* no surface gived */
	return SURF_UNKNOWN;

    }

    // cache class pointers for performance
    if( !windowClass ) {
    
	/* not loaded => load class */
	windowClass = getClass( createUtfString( "waba/ui/Window" ));

    }

    if( !imageClass ) {
    
	/* not loaded => load class */
	imageClass = getClass( createUtfString( "waba/fx/Image" ));

    }

    /* get the class of the given surface */
    wclass = WOBJ_class( surface );

    /* check the class of the given surface */
    if( compatible( wclass, windowClass )) {

	/* it's a window */
	return SURF_WINDOW;

    }
    if( compatible( wclass, imageClass )) {

	/* it's an image */
	return SURF_IMAGE;

    }

    /* not a known surface */
    return SURF_UNKNOWN;

}

/* implementation of native waba/fx/Graphics._nativeCreate() */
Var GraphicsCreate( Var stack[] ) {
    WObject gr, surface;
    Var v;
    ui_MainWindowType *mw;
    ui_PixmapType *imagePixmap;
    ui_FontType *theFont;

    ui_PixmapType *drawingPixmap;
    ui_Rectangle rect;
    gint surfaceType;

    /* get the Graphics object from the stack */
    gr = stack[0].obj;

    /* get the Graphics's variables */
    surface = WOBJ_GraphicsSurface(gr);

    /* get the type of the surface to create */
    surfaceType = SurfaceGetType( surface );

    /* initialisations */
    mw = NULL;
    imagePixmap = NULL;

    if( surfaceType == SURF_WINDOW ) {

      /* SURF_WINDOW surface type => get the WindowMain */
      mw = WOBJ_WindowMain( surface );

    } else if ( surfaceType == SURF_IMAGE ) {

      /* SURF_IMAGE surface type => get the ImagePixmap */
      imagePixmap = WOBJ_ImagePixmap( surface );

    }

    /* put the WindowMain in the Graphics object var hooks */
    WOBJ_GraphicsWindowMain(gr) = mw;

    /* put the ImagePixmap in the Graphics object var hooks */
    WOBJ_GraphicsPixmap(gr) = imagePixmap;

    /* define the clip region */
    WOBJ_GraphicsClipRegionX(gr) = rect.x = 0;
    WOBJ_GraphicsClipRegionY(gr) = rect.y = 0;
    WOBJ_GraphicsClipRegionW(gr) = rect.width = hwr_width;
    WOBJ_GraphicsClipRegionH(gr) = rect.height = hwr_height;

    /* create the pixmap for drawing */
    if( mw != NULL ) {

      /* main window exist => create a drawing pixmap from it */
      drawingPixmap = ui_createDrawingPixmap( mw, rect );

    } else if( imagePixmap != NULL ) {

      /* pixmap from image pixmap */
      drawingPixmap = imagePixmap;

    }

    /* put the drawingPixmap in the Graphics object var hooks */
    WOBJ_GraphicsDrawingPixmap(gr) = drawingPixmap;

    /* define a black color at start */
    WOBJ_GraphicsRED(gr) = 0x00;
    WOBJ_GraphicsGREEN(gr) = 0x00;
    WOBJ_GraphicsBLUE(gr) = 0x00;

    /* create the default font */
    theFont = ui_FontCreate( 0 );
    WOBJ_GraphicsFont(gr) = theFont;

    /* no translation by default */
    WOBJ_GraphicsTransX(gr) = 0;
    WOBJ_GraphicsTransY(gr) = 0;

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of the function called by the garbage collector when a
   Graphics object is destroyed                                             */
void GraphicsDestroy( WObject gr ) {

    DPUTS("??? GraphicsDestroy");

}

/* implementation of native waba/fx/Graphics.free() */
Var GraphicsFree( Var stack[] ) {
    Var v; 
    
    DPUTS("??? GraphicsFree"); 

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Graphics.setColor() */
Var GraphicsSetColor( Var stack[] ) {
    Var v;
    WObject gr;

    /* get the Graphics object from the stack */
    gr = stack[0].obj;

    /* get the wanted values from the stack */
    WOBJ_GraphicsRED(gr) = stack[1].intValue;
    WOBJ_GraphicsGREEN(gr) = stack[2].intValue;
    WOBJ_GraphicsBLUE(gr) = stack[3].intValue;

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Graphics.setForeColor() */
Var GraphicsSetForeColor( Var stack[] ) {
    Var v;
    
    DPUTS("??? GraphicsSetForeColor");

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Graphics.setBackColor() */
Var GraphicsSetBackColor( Var stack[] ) {
    Var v;
    
    DPUTS("??? GraphicsSetBackColor");

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Graphics.setTextColor() */
Var GraphicsSetTextColor( Var stack[] ) {
    Var v;
    
    DPUTS("??? GraphicsSetTextColor");

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Graphics.setFont() */
Var GraphicsSetFont(Var stack[]) {
    WObject gr, font;
    Var v;

    gr = stack[0].obj;
    font = stack[1].obj;
    /*
    if( WOBJ_GraphicsIsValid( gr ) == 1 ) {
    */{
        ui_FontType *theFont;

	/* get the pointer to the used font */
	theFont = WOBJ_GraphicsFont( gr );

	/* delete the old font */
#ifdef GUI_PGUI
	DPUTS( "??? GraphicsSetFont/delete the old font : not possible now, because pgserver use them (with PGFX_PERSISTENT)" );
#else
	ui_FontDelete( theFont );
#endif

	/* create the wanted font */
	theFont = ui_FontCreate( font );
	WOBJ_GraphicsFont(gr) = theFont;

    }

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Graphics.setDrawOp() */
Var GraphicsSetDrawOp( Var stack[] ) {
    WObject gr;
    Var v;

    /* get the Graphics object from the stack */
    gr = stack[0].obj;

    /*
    if( WOBJ_GraphicsIsValid( gr ) == 1 ) {
    */{

	/* OK, it's a valid Graphics object => store the wanted drawing operation */
	WOBJ_GraphicsDrawOp( gr ) = stack[1].intValue;

    }

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Graphics.setClip() */
Var GraphicsSetClip( Var stack[] ) {
    Var v;
    WObject gr;
    ui_Rectangle rect;

    /* get the Graphics object from the stack */
    gr = stack[0].obj;

    /* clip X and Y are stored in absolute coordinates */
    WOBJ_GraphicsClipRegionX(gr) = rect.x = WOBJ_GraphicsTransX(gr) + stack[1].intValue;
    WOBJ_GraphicsClipRegionY(gr) = rect.y = WOBJ_GraphicsTransY(gr) + stack[2].intValue;
    WOBJ_GraphicsClipRegionW(gr) = rect.width = stack[3].intValue;
    WOBJ_GraphicsClipRegionH(gr) = rect.height = stack[4].intValue;

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Graphics.getClip() */
Var GraphicsGetClip( Var stack[] ) {
    Var v;
    WObject gr;
    WObject rect;

    /* initialisation */
    v.obj = 0;

    /* get the Graphics object from the stack */
    gr = stack[0].obj;

    /* get the pointer to the rectangle to return */
    rect = stack[1].obj;

    /* test if rect is OK */
    if( rect == 0 /* || WOBJ_GraphicsIsValid(gr) != 1*/ ) {
      return v;
    }

    /* return the wanted values : alltime maximum of the window ??? */
    WOBJ_RectX(rect) = 0;
    WOBJ_RectY(rect) = 0;
    WOBJ_RectWidth(rect) = hwr_width;
    WOBJ_RectHeight(rect) = hwr_height;

    /* return the rectangle */
    v.obj = rect;
    return v;

}

/* implementation of native waba/fx/Graphics.clearClip() */
Var GraphicsClearClip( Var stack[] ) {
    WObject gr;
    Var v;
    ui_Rectangle rect;

    /* get the Graphics object from the stack */
    gr = stack[0].obj;

    /* clip X and Y are stored in absolute coordinates */
    WOBJ_GraphicsClipRegionX(gr) = rect.x = 0;
    WOBJ_GraphicsClipRegionY(gr) = rect.y = 0;
    WOBJ_GraphicsClipRegionW(gr) = rect.width = hwr_width;
    WOBJ_GraphicsClipRegionH(gr) = rect.height = hwr_height;

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/fx/Graphics.translate() */
Var GraphicsTranslate( Var stack[] ) {
    WObject gr;
    Var v;

    /* get the Graphics object from the stack */
    gr = stack[0].obj;

    /* add the gived offset for X and Y*/
    WOBJ_GraphicsTransX(gr) = WOBJ_GraphicsTransX(gr) + stack[1].intValue;
    WOBJ_GraphicsTransY(gr) = WOBJ_GraphicsTransY(gr) + stack[2].intValue;;
 
    /* return nothing */
    v.obj = 0 ;
    return v;

}

/* implementation of function for drawing depending of the gived parameter */
Var GraphicsDraw( int16 type, Var stack[] ) {
    Var v;
    ui_PixmapType *drawingPixmap;
    WObject gr;
    WObject string, charArray;
    gint32 x, y, count, start;
    gint32 transX, transY;
    gint32 drawOp;
    guint16 *chars;
    ui_FontType *theFont;
    ui_GraphicsContextType *theGC;
    ui_Rectangle rect;
    gint surfaceType;

    /* get the Graphics object from the stack */
    gr = stack[0].obj;

    /* get the type of the surface to draw */
    surfaceType = SurfaceGetType( WOBJ_GraphicsSurface( gr ));

    /* initialisation depending on the surface to draw */

    /* create the new context */
    theGC = ui_createGC( gr );

    if( theGC == NULL ) {

      /* the graphics context is not valid => stop */

      /* return nothing */
      v.obj = 0 ;
      return v;
      
    }

    if ( surfaceType == SURF_WINDOW ) {

      /* SURF_WINDOW surface type */

      /* get the drawing pixmap from the hook */
      drawingPixmap = WOBJ_GraphicsDrawingPixmap( gr );

    } else if ( surfaceType == SURF_IMAGE ) {

      /* SURF_IMAGE */

      /* get the drawing pixmap from the var hooks */
      drawingPixmap = WOBJ_GraphicsPixmap( gr );

    }

    /* set the color to the wanted color to use */
    ui_SetColor( theGC, WOBJ_GraphicsRED(gr), WOBJ_GraphicsGREEN(gr), WOBJ_GraphicsBLUE(gr) );

    /* get the region of the draw */
    rect.x = WOBJ_GraphicsClipRegionX(gr);
    rect.y = WOBJ_GraphicsClipRegionY(gr);
    rect.width = WOBJ_GraphicsClipRegionW(gr);
    rect.height = WOBJ_GraphicsClipRegionH(gr);

    /* get the translation to do */
    transX = WOBJ_GraphicsTransX(gr);
    transY = WOBJ_GraphicsTransY(gr);

    /* activate the clip */
    ui_ClipTo( theGC, rect );

    /* get the drawing operation to do */
    drawOp = WOBJ_GraphicsDrawOp(gr);

    /* set the drawing operation to do */
    ui_setTheDrawFunction( theGC, drawOp );

    /* draw depending of the type of drawing */
    switch( type ) {

    case GR_FILLRECT:   /* fill the given rectangle */
	{

	    gint x, y, w, h;

	    /* get the position/size of the rectangle from the stack */
	    x = stack[1].intValue + transX;
	    y = stack[2].intValue + transY;
	    w = stack[3].intValue;
	    h = stack[4].intValue;

	    /* draw the rectangle */
	    ui_DrawFilledRect( drawingPixmap, theGC, x, y, w, h );

	}
	break;

    case GR_DRAWLINE:	/* draw a line */
	{

	    gint x1, y1, x2, y2;

	    /* get the 2 points of the line from the stack */
	    x1 = stack[1].intValue + transX;
	    y1 = stack[2].intValue + transY;
	    x2 = stack[3].intValue + transX;
	    y2 = stack[4].intValue + transY;

	    /* draw the line */
	    ui_DrawLine( drawingPixmap, theGC, x1, y1, x2, y2 );

	}
	break;

      case GR_DOTS:    /* draw a horitontal or vertical dotted line  */
	{

	    gint x1, y1, x2, y2, stepx, stepy, tempSwap;
	    gint x, y;

	    /* get the 2 points of the line from the stack */
	    x1 = stack[1].intValue + transX;
	    y1 = stack[2].intValue + transY;
	    x2 = stack[3].intValue + transX;
	    y2 = stack[4].intValue + transY;

	    if( x1 > x2 ) {

	      /* reverse to have allways x1 lower than x2 */
	      tempSwap = x1;
	      x1 = x2;
	      x2 = tempSwap;

	    }

	    if( y1 > y2 ) {

	      /* reverse to have allways y1 lower than y2 */
	      tempSwap = y1;
	      y1 = y2;
	      y2 = tempSwap;

	    }

	    if( x1 == x2 ) {

	      /* vertical line */

	      stepx = 0;
	      stepy = 2;

	    } else if( y1 == y2 ) {

	      /* horizontal line */

	      stepx = 2;
	      stepy = 0;

	    } else {

	      /* neither horizontal, nor vertical => not correct */

	      break;

	    }

	    /* now, we can loop to make the draw */
	    for( x = x1, y = y1;
		 ( x <= x2 ) && ( y <= y2 );
		 x += stepx, y += stepy ) {

	      /* draw the current point */
	      ui_DrawPixel( drawingPixmap, theGC, x, y );

	    }
	}
	break;

    case GR_DRAWSTRING:    /* draw a string */
    case GR_DRAWCHARS:     /* draw chars */
	/* select the font */
	theFont = WOBJ_GraphicsFont(gr);

        if( type == GR_DRAWSTRING ) {

	  /* get the string to draw from the stack */
	  string = stack[1].obj;
	  charArray = WOBJ_StringCharArrayObj(string);

	  /* compute the x coordinate */
	  x = stack[2].intValue + transX;

	  /* compute the y coordinate by adding an Y offset */
	  y = stack[3].intValue + transY + ui_FontGetYOffset( theFont );

	  /* start at 0, and compute the size of the string from the charArray */
	  start = 0;
	  count = WOBJ_arrayLen(charArray);

	} else { 

	  /* get the chars to draw from the stack */
	  charArray = stack[1].obj;

	  /* get the starting position from the stack */
	  start = stack[2].intValue;

	  /* get the number of chars from the stack */
	  count = stack[3].intValue;

	  /* compute the x coordinate */
	  x = stack[4].intValue + transX;

	  /* compute the y coordinate by adding an Y offset */
	  y = stack[5].intValue + transY + ui_FontGetYOffset( theFont );

	  /* check the validity of the chars array */
	  if( arrayRangeCheck( charArray, start, count ) == 0 ) {

	    break;

	  }
	}

	/* convert the string to an "array of char" */
	chars = (uint16 *)WOBJ_arrayStart(charArray);
	chars = &chars[start];

	/* unicode => chars */
	while( count > 0 ) {

	    uchar buf[40];
	    int32 i, n;

	    /* limit the copy to the size of p */
	    n = sizeof( buf ) - 1;
	    if( n > count ) {
		n = count;
	    }

	    /* copy each char from the unicode chars variable to the "normal" p variable */
	    for( i = 0; i < n; i++ ) {
		buf[i] = (char)chars[i];
	    }

	    /* the terminator */
	    buf[i] = 0;

	    /* draw the string */
	    ui_DrawText( drawingPixmap, theGC, x, y, buf, theFont );

	    /* update the counters */
	    count -= n;
	    chars += n;

	    /* update the x position to draw */
	    x += ui_TextWidth( buf, theFont );

	}
	break;

    case GR_COPYRECT:    /* copy of a rectangle from a Surface to another Surface */

#ifdef GUI_GTK

	{

	    WObject srcSurf, dstSurf;
	    int x, y, w, h, dstX, dstY;

	    /* get the original surface */
	    srcSurf = stack[1].obj;

	    /* get the coordinate/size of the source rectangle to copy */
	    x = stack[2].intValue;
	    y = stack[3].intValue;
	    w = stack[4].intValue;
	    h = stack[5].intValue;

	    /* get the position where to put the copy in the destination */
	    dstX = stack[6].intValue + transX;
	    dstY = stack[7].intValue + transY;

	    /* get the destination Surface */
	    dstSurf = WOBJ_GraphicsSurface(gr);
	    if( dstSurf == srcSurf ) {

	      /* destination and source surface are same => directly draw the pixmap */
	      gdk_draw_pixmap( drawingPixmap, theGC, drawingPixmap, x, y, dstX, dstY, w, h );

	    } else {

	      /* destination and source different => get the source pixmap from an Image */

	      GdkPixmap *srcPixmap;

	      /* get the source pixmap */
	      srcPixmap = WOBJ_ImagePixmap( srcSurf );

	      /* draw the source pixmap to the destination pixmap */
	      gdk_draw_pixmap( drawingPixmap, theGC, srcPixmap, x, y, dstX, dstY, w, h );

	    }
	}

#else

      DPRINTF( "GraphicsDraw type %d not implemented\n", type );

#endif

	break;

    case GR_FILLPOLY:

#ifdef GUI_GTK

      /* draw a filled polygon */

      {

	WObject xArray, yArray;
	GdkPoint *thePoints;
	gint i, numberOfPoints;
	gint32 *x, *y;

	/* get the x and y arrays from the stack */
	xArray = stack[1].obj;
	yArray = stack[2].obj;

	if (xArray == 0 || yArray == 0) {

	  /* one of the arrays is not good => stop */
	  break;

	}

	/* get the number of points from the stack */
	numberOfPoints = stack[3].intValue;

	if( numberOfPoints < 3 
	    || numberOfPoints > WOBJ_arrayLen( xArray ) 
	    || numberOfPoints > WOBJ_arrayLen( yArray )) {

	  /* problem : number of points too low, or number of points in x/y not correct */
	  break;

	}

	/* allocation of the needed memory */
	thePoints = xmalloc( numberOfPoints * sizeof( GdkPoint ));

	if( thePoints == NULL ) {

	  /* problem allocating memory ==> stop */
	  break;

	}

	/* get a reference to the two arrays */
	x = (gint32 *)WOBJ_arrayStart(xArray);
	y = (gint32 *)WOBJ_arrayStart(yArray);

	for( i = 0; i < numberOfPoints; i++ ) {

	  /* copy all the points to the GdkPoint array */

	  thePoints[i].x = x[i] + transX;
	  thePoints[i].y = y[i] + transY;

	}
 
	/* finally, draw the polygon */
	gdk_draw_polygon( drawingPixmap, theGC, TRUE, thePoints, numberOfPoints );

	/* release the memory */
	xfree( thePoints );

      }
#else

      DPRINTF( "GraphicsDraw type %d not implemented\n", type );

#endif

      break;

    case GR_DRAWCURSOR:

	{

	    gint x, y, w, h;

	    /* get the 2 points of the line from the stack */
	    x = stack[1].intValue + transX;
	    y = stack[2].intValue + transY;
	    w = stack[3].intValue;
	    h = stack[4].intValue;

	    /* put color black */
	    ui_SetColor( theGC, 0x00, 0x00, 0x00 );

	    /* choose to draw XOR */
	    ui_setTheDrawFunction( theGC, DRAW_XOR );

	    /* draw the rectangle */
	    ui_DrawFilledRect( drawingPixmap, theGC, x, y, w, h );

	    /* put back the actually used drawing function */
	    ui_setTheDrawFunction( theGC, drawOp );

	}
	break;

    default:
	DPUTS( "*** ERROR *** WRONG DRAWING TYPE ***" );
	break;

    }

    /* force repaint */
    ui_GraphicsRepaint( gr, drawingPixmap, theGC );

    /* release the graphics context */
    ui_deleteGC( theGC );

    /* return nothing */
    v.obj = 0 ;
    return v;

}

/* implementation of native waba/fx/Graphics.fillRect() */
Var GraphicsFillRect( Var stack[] ) {

    return GraphicsDraw( GR_FILLRECT, stack );

}

/* implementation of native waba/fx/Graphics.drawLine() */
Var GraphicsDrawLine( Var stack[] ) {

    return GraphicsDraw( GR_DRAWLINE, stack );

}

/* implementation of native waba/fx/Graphics.fillPolygon() */
Var GraphicsFillPolygon( Var stack[] ) {

    return GraphicsDraw( GR_FILLPOLY, stack );

}

/* implementation of native waba/fx/Graphics.drawText(char chars[], int start, int count, int x, int y) */
Var GraphicsDrawChars( Var stack[] ) {

    return GraphicsDraw( GR_DRAWCHARS, stack );

}

/* implementation of native waba/fx/Graphics.drawText(String s, int x, int y) */
Var GraphicsDrawString( Var stack[] ) {

    return GraphicsDraw( GR_DRAWSTRING, stack );

}

/* implementation of native waba/fx/Graphics.drawDots() */
Var GraphicsDrawDots( Var stack[] ) {

    return GraphicsDraw( GR_DOTS, stack );

}

/* implementation of native waba/fx/Graphics.copyRect() */
Var GraphicsCopyRect( Var stack[] ) {

    return GraphicsDraw( GR_COPYRECT, stack );

}

/* implementation of native waba/fx/Graphics.drawCursor() */
Var GraphicsDrawCursor( Var stack[] ) {

    return GraphicsDraw( GR_DRAWCURSOR, stack );

}

/*************************************************************************
                               class waba/fx/SoundClip
*************************************************************************/

// var[0] = Class
// var[1] = path
// var[2] = loaded

/* implementation of native waba/fx/SoundClip.play() */
Var SoundClipPlay( Var stack[] ) {
    Var v;

    DPUTS("??? SoundClipPlay");

    /* return nothing */
    v.obj = 0 ;
    return v;

}


/*************************************************************************
                               class waba/sys/Convert
*************************************************************************/

/* implementation of native waba/sys/Convert.toIntBitwise(float f) */
Var ConvertFloatToIntBitwise( Var stack[] ) {

    return stack[0];

}

/* implementation of native waba/sys/Convert.toFloatBitwise(int i) */
Var ConvertIntToFloatBitwise( Var stack[] ) {

    return stack[0];

}

/* implementation of native waba/sys/Convert.toInt(String s) */
Var ConvertStringToInt( Var stack[] ) {
    WObject string, charArray;
    int32 i, isNeg, len, value;
    uint16 *chars;
    Var v;

    // NOTE: We do it all here instead of calling atoi() since it looks
    // like various versions of CE don't support atoi(). It's also faster
    // this way since we don't have to convert to a byte array.
    v.intValue = 0;
    string = stack[0].obj;
    if (string == 0)
	return v;
    charArray = WOBJ_StringCharArrayObj(string);
    if (charArray == 0)
	return v;
    chars = (uint16 *)WOBJ_arrayStart(charArray);
    len = WOBJ_arrayLen(charArray);
    isNeg = 0;
    if (len > 0 && chars[0] == '-')
	isNeg = 1;
    value = 0;
    for (i = isNeg; i < len; i++)
	{
	    if (chars[i] < (uint16)'0' || chars[i] > (uint16)'9')
		return v;
	    value = (value * 10) + ((int32)chars[i] - (int32)'0');
	}
    if (isNeg)
	value = -(value);
    v.intValue = value;
    return v;

}

/* implementation of native waba/sys/Convert.toString(int i) */
Var ConvertIntToString( Var stack[] ) {
    Var v;
    char buf[20];

    sprintf(buf, "%d", stack[0].intValue);

    v.obj = createString(buf);
    return v;
}

/* implementation of native waba/sys/Convert.toString(float f) */
Var ConvertFloatToString( Var stack[] ) {
    Var v;
    char buf[40];
    //	int len;

    sprintf(buf, "%f", stack[0].floatValue);

    v.obj = createString(buf);
    return v;
}

/* implementation of native waba/sys/Convert.toString(char c) */
Var ConvertCharToString( Var stack[] ) {
    Var v;
    char buf[2];

    buf[0] = (char)stack[0].intValue;
    buf[1] = 0;
    v.obj = createString(buf);
    return v;
}

/* implementation of native waba/sys/Convert.toString(boolean b) */
Var ConvertBooleanToString( Var stack[] ) {
    Var v;
    char *s;

    if (stack[0].intValue == 0)
	s = "false";
    else
	s = "true";
    v.obj = createString(s);
    return v;
}

/*************************************************************************
                               class waba/io/Catalog
As linux has no catalog, this is not implemented at this time
*************************************************************************/

Var CatalogCreate(Var stack[]) { Var v; v.obj = 0; return v; }
void CatalogDestroy(WObject cat) {}
Var CatalogIsOpen(Var stack[]) { Var v; v.intValue = 0; return v; }
Var CatalogClose(Var stack[]) { Var v; v.intValue = 0; return v; }
Var CatalogDelete(Var stack[]) { Var v; v.intValue = 0; return v; }
Var CatalogListCatalogs(Var stack[]) { Var v; v.obj = 0; return v; }
Var CatalogResizeRecord(Var stack[]) { Var v; v.intValue = 0; return v; }
Var CatalogGetRecordSize(Var stack[]) { Var v; v.intValue = -1; return v; }
Var CatalogGetRecordCount(Var stack[]) { Var v; v.intValue = -1; return v; }
Var CatalogDeleteRecord(Var stack[]) { Var v; v.intValue = 0; return v; }
Var CatalogAddRecord(Var stack[]) { Var v; v.intValue = -1; return v; }
Var CatalogSetRecordPos(Var stack[]) { Var v; v.intValue = 0; return v; }
Var CatalogRead(Var stack[]) { Var v; v.intValue = -1; return v; }
Var CatalogWrite(Var stack[]) { Var v; v.intValue = -1; return v; }
Var CatalogSkipBytes(Var stack[]) { Var v; v.intValue = -1; return v; }


/*************************************************************************
                               class waba/io/File
*************************************************************************/

char *_FileAllocStr(WObject string, const char* add)
{ // SD !
  WObject charArray;
  char *s;
  int32 len, addLen, i, j;
  int16 *chars;

  if (string == 0)
    return NULL;
  charArray = WOBJ_StringCharArrayObj(string);
  if (charArray == 0)
    return NULL;
  if (add != NULL)
    addLen = strlen(add);
  else
    addLen = 0;
  len = WOBJ_arrayLen(charArray);
  chars = (int16 *)WOBJ_arrayStart(charArray);

  // UNICODE: unsupported for now
  s = (char *)xmalloc((len + addLen + 1) * sizeof(char));
  if (s == NULL)
    return NULL;
  j = 0;
  for (i = 0; i < len; i++)
    s[j++] = (char)chars[i];
  for (i = 0; i < addLen; i++)
    s[j++] = add[i];
  s[j] = 0;
  return s;
}


void _FileFreeStr(char* path)
{ // SD !
  if (path != NULL)
    xfree((void *)path);
}


int32 _FileClose(WObject file)
{ // SD !
  int fileDesc;

  fileDesc = (int)WOBJ_FileHandle(file);
  if (fileDesc != -1)
    close(fileDesc);
  WOBJ_FileHandle(file) = (void*)-1;
  return 1;
}


Var FileCreate(Var stack[])
{ // SD !
  WObject file;
  Var v;
  int fileMode;
  int flags;
  int mustExist;
  char* path;
  int fileDesc;

  file = stack[0].obj;
  v.obj = 0;
  fileMode = WOBJ_FileMode(file);
  flags = 0;
  if (fileMode == File_READ_ONLY)
    flags |= O_RDONLY;
  else if (fileMode == File_WRITE_ONLY)
    flags |= O_WRONLY;
  else if (fileMode == File_READ_WRITE)
    flags |= O_RDWR;
  else if (fileMode == File_CREATE)
    flags |= O_CREAT|O_WRONLY|O_TRUNC;
  else
    return v;

  WOBJ_FileHandle(file) = (void *)-1;

  path = _FileAllocStr(WOBJ_FileName(file), NULL);
  if (path == NULL)
    return v;

  // except for CREATE, the file must exist
  if (fileMode != File_CREATE) {
    struct stat st;
    if (stat(path, &st) != 0) {
      _FileFreeStr(path);
      return v;
    }
  }

  fileDesc = open(path, flags);
  _FileFreeStr(path);
  WOBJ_FileHandle(file) = (void *)fileDesc;
  if (fileDesc==-1)
    return v;
}

void FileDestroy(WObject file)
{ // SD !
  _FileClose(file);
}

Var FileIsOpen(Var stack[])
{ // SD !
  WObject file;
  int  fileDesc;
  Var v;

  file = stack[0].obj;
  fileDesc = (int)WOBJ_FileHandle(file);
  v.intValue = 0;
  if (fileDesc == -1)
    return v;
  v.intValue = 1;
  return v;
}

Var FileGetLength(Var stack[])
{ // SD !
  WObject file;
  int fileDesc;
  long len;
  Var v;
  struct stat st;

  file = stack[0].obj;
  fileDesc = (int)WOBJ_FileHandle(file);
  v.intValue = 0;

  if (fileDesc == -1)
    return v;

  if (fstat(fileDesc, &st) != 0)
      return v;

  len = st.st_size;
  v.intValue = len;
  return v;
}

int _FileIsDir(char* path)
{ // SD !
  struct stat st;
  if (stat(path, &st) == 0) {
    if (S_ISDIR(st.st_mode))
      return 1;
  }
  return 0;
}


Var FileOp(Var stack[], int op)
{ // SD !
  WObject file;
  char* path;
  Var v;

  v.intValue = 0;
  file = stack[0].obj;
  path = _FileAllocStr(WOBJ_FileName(file), NULL);
  if (path == NULL)
    return v;
  switch (op) {
    case FILE_CREATE_DIR: {
      if (mkdir(path, 0) == 0)
	v.intValue = 1;
      break;
    }
    case FILE_IS_DIR: {
      v.intValue = _FileIsDir(path);
      break;
    }
    case FILE_DELETE: {
      _FileClose(file);
      if (remove(path) ==0 )
	v.intValue = 1;
      break;
    }
    case FILE_RENAME: {
      char * dstPath;

      dstPath = _FileAllocStr(stack[1].obj, NULL);
      if (dstPath != NULL) {
	_FileClose(file);
	if (rename(path, dstPath) == 0)
	  v.intValue = 1;
	_FileFreeStr(dstPath);
      }
      break;
    }
    case FILE_EXISTS: {
      struct stat st;
      if (stat(path, &st) == 0)
	v.intValue = 1;
      break;
    }
  }
  _FileFreeStr(path);
  return v;
}

Var FileCreateDir(Var stack[])
{
  return FileOp(stack, FILE_CREATE_DIR);
}

Var FileIsDir(Var stack[])
{
  return FileOp(stack, FILE_IS_DIR);
}

Var FileDeleteWaba(Var stack[])
{
  return FileOp(stack, FILE_DELETE);
}

Var FileRename(Var stack[])
{
  return FileOp(stack, FILE_RENAME);
}

Var FileExists(Var stack[])
{
  return FileOp(stack, FILE_EXISTS);
}

Var FileSeekWaba(Var stack[])
{ // SD !
  WObject file;
  off_t pos;
  int fileDesc;
  Var v;

  v.intValue = 0;
  file = stack[0].obj;
  pos = stack[1].intValue;
  fileDesc = (int)WOBJ_FileHandle(file);
  if (fileDesc == -1)
    return v;
  if (lseek(fileDesc, pos, SEEK_SET) == pos)
    v.intValue = 1;
  return v;
}

Var FileCloseWaba(Var stack[])
{ // SD !
  Var v;

  v.intValue = _FileClose(stack[0].obj);
  return v;
}

typedef struct FileListItemStruct
{
  char *fileName;
  struct FileListItemStruct *next;
} FileListItem;

Var FileListDir(Var stack[])
{ // SD ?
  WObject file, stringArray, *strings;
  char *path;
  int findDesc;
  char *fileName;
  FileListItem *list, *item;
  int i, numItems;
  Var v;

  struct        dirent  *dp;
  DIR           *dirp;
  struct        stat    statbuf;


  v.obj = 0;
  file = stack[0].obj;
  path = _FileAllocStr(WOBJ_FileName(file), NULL);
  if (path == NULL)
    return v;

  // read paths into linked list
  dirp = opendir(path);
  _FileFreeStr(path);
  if (dirp == NULL)
    return v;

  list = NULL;
  numItems = 0;

  while ((dp = readdir(dirp)) != NULL) {
    fileName = dp->d_name;
    if ((fileName[0] == '.' && fileName[1] == 0) ||
	(fileName[0] == '.' && fileName[1] == '.' && fileName[2] == 0))
      continue;
    item = (FileListItem *)xmalloc(sizeof(FileListItem));
    if (item == NULL)
      break;
    item->fileName = (char*)malloc((strlen(fileName) + 1) * sizeof(char));
    if (item->fileName == NULL) {
      free(item);
      break;
    }
    strcpy(item->fileName, dp->d_name);
    item->next = list;
    list = item;
    numItems++;
  }

  // convert linked list into string array
  stringArray = createArrayObject(1, numItems);
  if (pushObject(stringArray) == -1)
    goto freereturn;
  i = 0;  
  item = list;
  strings = (WObject *)WOBJ_arrayStart(stringArray);
  while (item) {
    // we need to recompute the start pointer each iteration
    // in case garbage collection during a string create causes
    // memory to move around
    strings[i++] = createString(item->fileName);
    item = item->next;
  }
  popObject(); // stringArray

 freereturn:
  // free linked list
  while (list) {
    item = list;
    list = list->next;
    xfree(item->fileName);
    xfree(item);
  }
  
  v.obj = stringArray;
  return v;
}

Var FileReadWriteBytes(Var stack[], int isRead)
{
  WObject file, byteArray;
  size_t start, count;
  uchar *bytes;
  int fileDesc;
  ssize_t numRW;
  Var v;

  v.intValue = -1;
  file = stack[0].obj;
  fileDesc = (int)WOBJ_FileHandle(file);
  if (fileDesc == -1)
    return v;
  byteArray = stack[1].obj;
  start = stack[2].intValue;
  count = stack[3].intValue;
  if (arrayRangeCheck(byteArray, start, count) == 0)
    return v; // array null or range invalid
  bytes = (uchar *)WOBJ_arrayStart(byteArray);
  if (isRead)
    numRW = read(fileDesc, &bytes[start], count);
  else
    numRW = write(fileDesc, &bytes[start], count);
  v.intValue = numRW;
  return v;
}

Var FileReadBytes(Var stack[])
{
  return FileReadWriteBytes(stack, 1);
}

Var FileWriteBytes(Var stack[])
{
  return FileReadWriteBytes(stack, 0);
}

/*************************************************************************
                               class waba/io/SerialPort
*************************************************************************/

/* implementation of function for closing a serial port */
int32 _SerialPortClose( WObject port ) {

    DPUTS("??? _SerialPortClose"); 

    /* return nothing */
    return 0;

}

/* implementation of native waba/io/SerialPort._nativeCreate() */
Var SerialPortCreate( Var stack[] ) {
    Var v;

    DPUTS("??? SerialPortCreate");
    
    /* return nothing */
    v.obj = 0 ;
    return v;

}

/* implementation of the function called by the garbage collector when a
   SerialPort object is destroyed                                         */
void SerialPortDestroy( WObject port ) {

    _SerialPortClose(port);

}

/* implementation of native waba/io/SerialPort.isOpen() */
Var SerialPortIsOpen( Var stack[] ) {
    Var v; 

    DPUTS("??? SerialPortIsOpen"); 

    /* return nothing */
    v.obj = 0 ;
    return v;

}

/* implementation of native waba/io/SerialPort.setReadTimeout() */
Var SerialPortSetReadTimeout( Var stack[] ) {
    Var v;

    DPUTS("??? SerialPortSetReadTimeout"); 

    /* return nothing */
    v.obj = 0 ;
    return v;

}

/* implementation of native waba/io/SerialPort.readCheck() */
Var SerialPortReadCheck( Var stack[] ) {
    Var v;

    v.intValue = -1;
    return v;

}

/* implementation of native waba/io/SerialPort.setFlowControl() */
Var SerialPortSetFlowControl( Var stack[] ) {
    Var v;

    DPUTS("??? SerialPortSetFlowControl"); 
    
    /* return nothing */
    v.obj = 0 ;
    return v;

}

/* implementation of function for reading/writing with a serial port */
Var SerialPortReadWriteBytes( Var stack[], int isRead ) {
    Var v; 

    DPUTS("??? SerialPortReadWriteBytes");

    /* return nothing */
    v.obj = 0 ;
    return v;

}

/* implementation of native waba/io/SerialPort.readBytes() */
Var SerialPortRead( Var stack[] ) {

    return SerialPortReadWriteBytes(stack, 1);

}

/* implementation of native waba/io/SerialPort.writeBytes() */
Var SerialPortWrite( Var stack[] ) {

    return SerialPortReadWriteBytes(stack, 0);

}

/* implementation of native waba/io/SerialPort.close() */
Var SerialPortClose(Var stack[]) {
    WObject port;
    Var v;

    port = stack[0].obj;
    v.intValue = _SerialPortClose(port);
    return v;

}

/*************************************************************************
                               class waba/sys/Time
*************************************************************************/

/* implementation of native waba/sys/Time._nativeCreate() */
Var TimeCreate( Var stack[] ) {
    Var v;
    WObject otime;
    struct tm* tm;
    struct timeval now;
    time_t tt;

    /* get the Time object from the stack */
    otime = stack[0].obj;
  
    /* get the actual time */
    gettimeofday( &now, NULL );
    tt = time( 0 );
    tm = localtime(&tt);
  
    /* set the variable of the Time object */
    WOBJ_TimeYear(otime) = tm->tm_year + 1900;
    WOBJ_TimeMonth(otime) = tm->tm_mon + 1;
    WOBJ_TimeDay(otime) = tm->tm_mday;
    WOBJ_TimeHour(otime) = tm->tm_hour;
    WOBJ_TimeMinute(otime) = tm->tm_min;
    WOBJ_TimeSecond(otime) = tm->tm_sec;
    WOBJ_TimeMillis(otime) = now.tv_usec/1000;

    /* return nothing */
    v.obj = 0;
    return v; 

}

/*************************************************************************
                               class waba/sys/Vm
*************************************************************************/

/* implementation of native waba/sys/Vm.getTimeStamp() */
Var VmGetTimeStamp( Var stack[] ) {
    Var v;

    v.intValue = getTimeStamp();
    return v;

}

/* implementation of native waba/sys/Vm.isColor() */
Var VmIsColor( Var stack[] ) {
    Var v;

    v.intValue = 1;
    return v;

}

/* implementation of native waba/sys/Vm.exec() */
Var VmExec( Var stack[] ) {
    Var v;

    DPUTS("??? VmExec");

    /* return nothing */
    v.obj = 0;
    return v; 

}

/* implementation of native waba/sys/Vm.sleep() */
Var VmSleep( Var stack[] ) {
    Var v; 
    DPUTS("??? VmSleep"); 

    /* return nothing */
    v.obj = 0;
    return v;

}

/* implementation of native waba/sys/Vm.getPlatform() */
Var VmGetPlatform( Var stack[] ) {
    Var v;

#ifndef PLATFORM
# error You must define PLATFORM
#endif
    v.obj = createString(PLATFORM);
    return v;

}

/* implementation of native waba/sys/Vm.setDeviceAutoOff() */
Var VmSetDeviceAutoOff( Var stack[] ) {
    Var v;

    v.intValue = 0;
    return v;

}

/* implementation of native waba/sys/Vm.getUserName() */
Var VmGetUserName( Var stack[] ) {
    Var v;
    struct passwd *pw;
    uid_t uid;
    
    uid = geteuid();
    pw = getpwuid(uid);
    if (pw) {
	v.obj = createString(pw->pw_name);
    }
    else {
	v.obj = createString("<unknown>");
    }

    return v;

}

Var VmPrint(Var stack[]) {
    WObject strString;
    UtfString str;
    Var v;

    v.intValue = 0;
    strString = stack[0].obj;
    str = stringToUtf(strString, STU_NULL_TERMINATE);

    printf("%s", str.str);
    fflush(stdout);
    return v;

}

Var VmPrintLn( Var stack[] ) {
    WObject strString;
    UtfString str;
    Var v;

    v.intValue = 0;
    strString = stack[0].obj;
    str = stringToUtf(strString, STU_NULL_TERMINATE);


    printf("%s\n", str.str);
    return v;

}

//
// PalmOS
//
// PalmOS specific functions, so these are just stubs for other platforms.
// By Isao F. Yamashita 01/24/2001

/* implementation of native waba/ui/PalmOsPref.getPalmOsPref() */
Var PalmOsPrefGetPalmOsPref( Var stack[] ) {
  Var v;

  DPUTS("??? PalmOsPrefGetPalmOsPref"); 

  v.intValue = 0;
  return v;

}

/* implementation of native waba/ui/PalmOsPref.setPalmOsPref() */
Var PalmOsPrefSetPalmOsPref(Var stack[]) {
  Var     v;

  DPUTS("??? PalmOsPrefSetPalmOsPref"); 

  v.intValue = 0;
  return v;

}


/*
   Local Variables:
   c-file-style: "smartdata"
   End:
*/

