/*
   Project: Adun

   Copyright (C) 2006 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/
#ifndef _ADRELATIONSHIP_
#define _ADRELATIONSHIP_

#include <Foundation/Foundation.h>

/**
Represent a relation ship between one or more objects.
The relationships can be:

- one to one
- one to many
- many to many

depending on which init method is used. In essence an AdRelationship instance
represents the declaration: object(s) relationship object(s) e.g. systemOne interactsWith systemTwo.
The first object(s) is (are) called the subject(s). The second object(s) is (are) called the objects.
The interpretation of the \e relationship is completly dependant on the user. AdRelationship simply
provides a way to keep track of them. 
*/

@interface AdRelationship: NSObject
{
	NSArray* validTypes;
	id subject;
	id object;
	id relationship;
	id relationshipType;
}

/**
Represents a one to one relationship between \e objectOne and \e objectTwo
*/
- (id) initWithObject: (id) objectOne relationship: (NSString*) aString object: (id) objectTwo;
/**
Represent a one to many relationship between \e objectOne and the objects in \e anArray
*/
- (id) initWithObject: (id) objectOne relationship: (NSString*) aString objects: (NSArray*) anArray;
/**
Represents a many to many relationship between the objects of \e arrayOne and \e arrayTwo
*/
- (id) initWithObject: (id) arrayOne relationship: (NSString*) aString objects: (NSArray*) arrayTwo;
/**
Designated initialiser. The object types of \e objectOne and \e objectTwo are determined by \e type.
If the object types do not agree with those implied by \type  then an NSInvalidArgumentException is 
raised. The valid types are AdOneToOneRelationship (and two objects), AdOneToManyRelationship (any object and
an NSArray), AdManyToManyRelationship (two NSArray's)
*/
- (id) initWithObject: (id) objectOne 
	relationship: (NSString*) aString 
	relationshipType: (NSString*) type
	object: (id) objectTwo;

/**
Returns the type of the relationship
*/
- (id) type;

/**
Returns the string describing the relationship
*/
- (NSString*) relationship;
/**
Returns the subject(s) in the relationship
*/
- (id) subjects;
/**
Returns the object(s) in the relationship
*/
- (id) objects;
/**
Sets the relationship between the subject(s) and object(s) to
\e aString
*/
- (void) setRelationship: (NSString*) aString;
/**
Sets the subject of the relationship to \e anObject. For use with
a one to one or one to many type relationship. Raises an NSInternalInconsistencyException
if the relationship type is AdManyToManyRelationship
*/
- (void) setSubject: (id) anObject;
/**
Sets the subjects of a relationship to the objects contained in anArray.
The relationship type must be AdManyToManyRelationship otherwise NSInvalidArgumentException
is raised.
*/
- (void) setSubjects: (NSArray*) anArray;
/**
Sets the object of the relationship to \e anObject. For use with
a one to one relationship. Raises an NSInternalInconsistencyException
if the relationship type is not AdOneToOneRelationship. 
*/
- (void) setObject: (id) anObject;
/**
Sets the objects of a relationship to the objects contained in anArray.
If the relationship type is AdOneToOneRelationship NSInvalidArgumentException
is raised.
*/
- (void) setObjects: (NSArray*) anArray;
@end


#endif
