//
//	aegis - project change supervisor
//	Copyright (C) 2003-2005 Peter Miller;
//	All rights reserved.
//
//	This program is free software; you can redistribute it and/or modify
//	it under the terms of the GNU General Public License as published by
//	the Free Software Foundation; either version 2 of the License, or
//	(at your option) any later version.
//
//	This program is distributed in the hope that it will be useful,
//	but WITHOUT ANY WARRANTY; without even the implied warranty of
//	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//	GNU General Public License for more details.
//
//	You should have received a copy of the GNU General Public License
//	along with this program; if not, write to the Free Software
//	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.
//
// MANIFEST: functions to manipulate edit_numbers
//

#include <common/ac/stdio.h>

#include <common/error.h> // for assert
#include <libaegis/change.h>
#include <libaegis/change/branch.h>
#include <libaegis/project/file/roll_forward.h>

#include <aeget/emit/edit_number.h>
#include <aeget/http.h>


static file_event_ty *
file_event_from_revision(string_ty *revision, file_event_list_ty *felp)
{
    if (!felp)
	return 0;
    for (size_t j = 0; j < felp->length; ++j)
    {
	file_event_ty *fep = felp->item + j;
	assert(fep);
	if (!fep)
	    continue;
	fstate_src_ty *src = fep->src;
	assert(src);
	if (!src)
	    continue;
	assert(src->edit);
	if (!src->edit)
	    continue;
	assert(src->edit->revision);
	if (!src->edit->revision)
	    continue;
	if (str_equal(revision, src->edit->revision))
	    return fep;
    }
    return 0;
}


static void
emit_edit_number_inner(change_ty *cp, fstate_src_ty *src_data,
    file_event_list_ty *felp = 0)
{
    if (src_data->edit_origin && src_data->edit)
    {
	//
	// We have both the original version copied, and the
	// current head revision.  Print them both, with a
	// notation implying "from the old one to the new one"
	// if they differ.  Only print one if thay are the same.
	//
	assert(src_data->edit->revision);
	assert(src_data->edit_origin->revision);
	if
	(
	    !str_equal
	    (
		src_data->edit->revision,
		src_data->edit_origin->revision
	    )
	)
	{
	    file_event_ty *fep =
		file_event_from_revision
		(
		    src_data->edit_origin->revision,
		    felp
		);
	    if (fep)
	    {
		emit_file_href(fep->cp, fep->src->file_name, 0);
		printf("%s</a>\n", src_data->edit_origin->revision->str_text);
		string_ty *vers = change_version_get(cp);
		string_ty *mod =
		    str_format("diff+unified+rhs=%s", vers->str_text);
		str_free(vers);
		emit_file_href(fep->cp, fep->src->file_name, mod->str_text);
		str_free(mod);
		printf("&rarr;</a>\n");
	    }
	    else
	    {
		printf
		(
		    "%s\n&rarr;\n",
		    src_data->edit_origin->revision->str_text
		);
	    }
	}
	emit_file_href(cp, src_data->file_name, 0);
	printf("%s", src_data->edit->revision->str_text);
	printf("</a>\n");
	return;
    }

    if (src_data->edit_origin)
    {
	//
	// The "original version" copied.
	//
	assert(src_data->edit_origin->revision);
	file_event_ty *fep =
	    file_event_from_revision(src_data->edit_origin->revision, felp);
	if (fep)
	{
	    emit_file_href(fep->cp, fep->src->file_name, 0);
	    printf("%s</a>\n", fep->src->edit->revision->str_text);
	}
	else
	    printf("%s\n", src_data->edit_origin->revision->str_text);
    }
    if (src_data->edit)
    {
	//
	// For active branches, the current
	// head revision.  For completed changes
	// and branches, the revision at aeipass.
	//
	assert(src_data->edit->revision);
	file_event_ty *fep =
	    file_event_from_revision(src_data->edit->revision, felp);
	if (fep)
	{
	    emit_file_href(fep->cp, fep->src->file_name, 0);
	    printf("%s</a>\n", fep->src->edit->revision->str_text);
	}
	else
	    printf("%4s\n", src_data->edit->revision->str_text);
    }
    if (!cp->bogus && src_data->edit_origin_new)
    {
	printf("<br>{cross ");

	assert(src_data->edit_origin_new->revision);
	file_event_ty *fep =
	    file_event_from_revision(src_data->edit_origin_new->revision, felp);
	if (fep)
	{
	    emit_file_href(fep->cp, fep->src->file_name, 0);
	    printf("%s</a>", fep->src->edit->revision->str_text);
	}
	else
	    printf("%4s", src_data->edit_origin_new->revision->str_text);
	printf("}\n");
    }
}


void
emit_edit_number(change_ty *cp, fstate_src_ty *src,
    project_file_roll_forward * hp)
{
    switch (src->action)
    {
    case file_action_remove:
    case file_action_insulate:
    case file_action_transparent:
	emit_edit_number_inner(cp, src);
	break;

    case file_action_create:
    case file_action_modify:
	if (hp && hp->is_set())
	    emit_edit_number_inner(cp, src, hp->get(src));
	else
	    emit_edit_number_inner(cp, src);
	break;
    }
}
