/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <noteobj.h>
#include <clientvfs.h>
#include <qstring.h>
#include <qsize.h>
#include <qpoint.h>
#include <qfont.h>
#include <vobject.h>
#include <kconfig.h>
#include <qdatastream.h>
#include <qcstring.h>
#include <vobjectfactory.h>
#include <queryprocessor.h>
#include <qvariant.h>

extern KConfig *GlobalConfig;

// implementation for the note object
#define IDSTRING "NoteObject: "

NoteObject::NoteObject(QString _path) : VObject(_path)
{
	// debug
	printf(IDSTRING"NoteObject constructor: syncing object...\n");
	
	// restore the object properties from the server
  sync();
	
	// debug
	printf(IDSTRING"NoteWidget contructor: done\n");
}

NoteObject::NoteObject(QString _path, QByteArray &_data) : VObject(_path)
{
	// debug
	printf(IDSTRING"NoteObject constructor: syncing object...\n");
	
	// restore the object properties from the server
	if( !_data || !_data.size() )
	  sync();
	else
	  _narrow(_data);
	
	// debug
	printf(IDSTRING"NoteWidget contructor: done\n");
}

NoteObject::~NoteObject()
{
	// nothing here
}

void NoteObject::setText(QString newText)
{
	// set the property
	_text=newText;
}

QString NoteObject::text()
{
	// return the property
	return _text;
}

void NoteObject::setSize(QSize newSize)
{
	_size=newSize;
}

QSize NoteObject::size()
{
	return _size;
}

void NoteObject::setPosition(QPoint newPosition)
{
	_position=newPosition;
}

QPoint NoteObject::position()
{
	return _position;
}

void NoteObject::setFont(QFont newFont)
{
	_font=newFont;
}

QFont NoteObject::font()
{
	return _font;
}

int NoteObject::onDisplay()
{
	return _onDisplay;
}
void NoteObject::setOnDisplay(int newOnDisplay)
{
	_onDisplay=newOnDisplay;
}

void NoteObject::setColor(QString newColor)
{
	// check for invalid colors
	if(!QStringList::split(',', "yellow,blue,red,green").grep("yellow", true).count())
	{
		// set the default color
		GlobalConfig->setGroup("Notes");
		newColor=GlobalConfig->readEntry("DefaultColor");
	}
	_color=newColor;
}

QString NoteObject::color()
{
	return _color;
}

void NoteObject::setTime(QDateTime newTime)
{
	_time=newTime;
}	

QDateTime NoteObject::time()
{
	return _time;
}

void NoteObject::setCategory(QString newCategory)
{
	_category=newCategory;
}

QString NoteObject::category()
{
	return _category;
}

QByteArray NoteObject::_stream()
{
	QByteArray tdata;
	QDataStream tstream(tdata, IO_WriteOnly);
	
	// stream the object properties
	tstream<<_size<<_position<<_text<<_font<<_color<<_time<<_category<<_onDisplay<<_extraData;
	// return the byte array
	return tdata;
}

void NoteObject::_narrow(const QByteArray &tdata)
{
	// check if the stream actually contains any data
	if(tdata.size())
	{
		// debug
		printf(IDSTRING"Reading note properties from stream...\n");
		
		QDataStream tstream(tdata, IO_ReadOnly);

		// unstream the object properties
		tstream>>_size>>_position>>_text>>_font>>_color>>_time>>_category>>_onDisplay>>_extraData;
		
		printf(IDSTRING"note %s color: %s\n", (const char *)path(), (const char *)_color);
	}
	else
	{
		// debug
		printf(IDSTRING"Setting default note properties...\n");
		
		// provide default values if needed
		GlobalConfig->setGroup("Notes");
		GlobalConfig->readFontEntry("Default font", &_font);
		GlobalConfig->readSizeEntry("Default size", &_size);
		_color=GlobalConfig->readEntry("Default color","yellow").lower();
				
		_time=QDateTime::currentDateTime();
		_text="";
		_position=QPoint(0, 0);
		_category="";
		_onDisplay=true;
	}
}

bool NoteObject::getProperty(const QString &query, Any &property)
{
	if(query=="size")
	{
		property=Any(_size);
		return true;
	}
	if(query=="position")
	{
		property=Any(_position);
		return true;
	}
	if(query=="text")
	{
		property=Any(_text);
		return true;
	}
	if(query=="font")
	{
		property=Any(_font);
		return true;
	}
	if(query=="color")
	{
		property=Any(_color);
		return true;
	}
	if(query=="time")
	{
		property=Any(_time);
		return true;
	}
	if(query=="onDisplay")
	{
		property=Any(_onDisplay);
		return true;
	}
	if(query=="category")
	{
		property=Any(_category);
		return true;
	}
	return false;
}

bool NoteObject::setProperty(const QString &query, Any &property)
{
	if(query=="size")
	{
		_size=property.toSize();
		return true;
	}
	if(query=="position")
	{
		_position=property.toPoint();
		return true;
	}
	if(query=="text")
	{
		_text=property.toString();
		return true;
	}
	if(query=="font")
	{
		_font=property.toFont();
		return true;
	}
	if(query=="color")
	{
		_color=property.toString();
		return true;
	}
	if(query=="onDisplay")
	{
		_onDisplay=property.toInt();
		return true;
	}
	if(query=="category")
	{
		_category=property.toString();
		return true;
	}
	return false;
}

QueryProcessor::QueryType NoteObject::queryType(const QString &query)
{
	if(query=="size" || query=="position" || query=="text" || query=="font" || query=="color" || query=="category")
		return QueryProcessor::ReadWriteQuery;
	
	if(query=="time")
		return QueryProcessor::ReadOnlyQuery;
	
	return QueryProcessor::InvalidQuery;
}

// implementation of the note object factory

NoteObject *NoteObjectFactory::_new(QString _path)
{
	return ClientVFS::thisInstance()->template _new<NoteObject>(_path);
}

NoteObject *NoteObjectFactory::_new(QString _path, QByteArray &_data)
{
	return ClientVFS::thisInstance()->template _new<NoteObject>(_path, _data);
}
