/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <clientvfs.h>
#include <vobject.h>
#include <vfolder.h>
#include <qstring.h>
#include <qstringlist.h>
#include <qdict.h>
#include <vobjectfactory.h>
#include <kconfig.h>
#include <brokeriface.h>
#include <vcacheobject.h>

#include <mainwindow.h>

static ClientVFS *inst;
extern KConfig *GlobalConfig;

ClientVFS::ClientVFS()
{
	useFact=false;
	autogenTemp=false;
	genVInstances=false;
	genNonRegistered=false;
	currentPath=root();
	printf("ClientVFS activated.\n");
}

ClientVFS::~ClientVFS()
{
	// nothing here
}

ClientVFS *ClientVFS::thisInstance()
{
	return inst?inst:inst=new ClientVFS();
}

bool ClientVFS::initHierarchy()
{
	// read the folder hierarchy from the configuration object
	GlobalConfig->setGroup("VFS Init");
	QStrList topLevel;
	GlobalConfig->readListEntry("TopLevel", topLevel);
	if(!topLevel.count())
	{
		// warning: no hierarchy could be retrieved
		printf("Warning: ClientVFS: no hierarchy could be retrieved from the config object, bailing out\n");
		return false;
	}
	for(QStrListIterator it(topLevel); it.current(); ++it)
	{
		QString folderName=(*it);
		if(!makeBranch(folderName, root()))
		{
			printf("Warning: ClientVFS: makeBranch() returned false when true was expected, bailing out\n");
			return false;
		}
	}
	return true;
}

bool ClientVFS::makeBranch(QString folderName, QString path)
{
	QString groupName="VFS ";
	groupName+=folderName;
	GlobalConfig->setGroup(groupName);
	
	// read the folder type
	QString type=GlobalConfig->readEntry("Type");
	if(type.isEmpty())
		type=QString::null;

	// view type associated with folder
	QString viewType=GlobalConfig->readEntry("View type");
	if(viewType.isEmpty())
		viewType=type;

	// MIME type associated with folder
	QString mimeType=GlobalConfig->readEntry("MIME type");
			
	// read the folder binding
	QString binding=GlobalConfig->readEntry("Binding");
	
	// read the server path
	QString sPath=GlobalConfig->readEntry("Server Path");

	// read the folder flags
	QString flags=GlobalConfig->readEntry("Flags");
	if(flags.isEmpty()) flags="c";
	
	int iFind=folderName.findRev('/')+1;
  path=path+"/"+folderName.right(folderName.length()-iFind);
			
	// debug message
	printf("Creating new VFolder: %s...\n", (const char *)(type+":/"+path));
	fflush(stdout);

	VFolder *tFolder=_new(path, type, viewType, mimeType, binding, sPath);
	
	if(!tFolder)
	{
		printf("Warning: folder %s could not be created, reverting to null binding\n", (const char *)path);
		tFolder=_new(path, type, viewType, mimeType, QString::null, QString::null);
	}
		
	if(tFolder) tFolder->setFlags(flags);
	
	// create VObject instances, if necessary
	if(genVInstances)
	{
		if(!createVInstances(tFolder))
		{
			printf("Warning: ClientVFS: createVInstances(VFolder *) returned false when true was expected, bailing out\n");
			return false;
		}
	}
			
	// read the subfolder list
	QStrList subfolders;
	GlobalConfig->readListEntry("Subfolders", subfolders);
	
	// process every subfolder
	for(QStrListIterator it(subfolders); it.current(); ++it)
	{
		if(!makeBranch((*it), path))
		{
			printf("Warning: ClientVFS: makeBranch() returned false when true was expected, bailing out\n");
			return false;
		}
	}
	
	return true;
}

VFolder *ClientVFS::_new(QString path, QString type, QString viewType, QString mimeType, QString binding, QString spath)
{
	// debug
	// printf("VFolder::_new(%s, %s, %s, %s)\n", (const char *)path, (const char *)type, (const char *)binding, (const char *)spath);
	
	if(folderRef(path))
	{
		// debug
		printf("Warning: VFolder *_new() was called on an already registered path\n");
		
		return 0;
	}
	
	if(binding.isEmpty() || type=="temporary" || BrokerInterface::thisInstance()->isFolder(spath))
	{
		VFolder *f=new VFolder(path);
		f->setType(type);
		f->setViewType(viewType);
		f->setMimeType(mimeType);
		f->setBinding(binding);
		f->setServerPath(spath);
		folders.insert(path, f);
		return f;
	}
	else
	{
		// debug
		// printf("clientvfs: did not create folder\n");
	}
	
	return 0;
}

VReference *ClientVFS::_fact_new(QString path, QString type)
{
	VObjectFactory *fact=factoryDict[type];
	if(!path.isEmpty() && canDefine(path) && !reference(path) && fact)
		return fact->_new(path);
	
	// debug
	printf("Warning: ClientVFS::_fact_new could not create object %s\nReason(s):\n", (const char *)path);
	if(!canDefine(path)) printf("  -- object does not exist on server\n");
	if(!fact)            printf("  -- no factory was associated with type \"%s\"\n", (const char *)type);
	if(reference(path))  printf("  -- a reference to this object already exists, try _fact_ref instead\n");
			
	return 0;
}

VReference *ClientVFS::_fact_ref(QString path, QString type)
{
	if(!path.isEmpty())
		{
			VReference *r=reference(path);
			if(r!=0) return r;
		}
	
	VObjectFactory *fact=factoryDict[type];
	if(!path.isEmpty() && canDefine(path) && fact)
		return fact->_new(path);
	
	// debug
	printf("Warning: ClientVFS::_fact_ref could not create object %s:%s\nReason(s):\n", (const char *)type, (const char *)path);
	if(!canDefine(path)) printf("  -- object does not exist on server\n");
	if(!fact)            printf("  -- no factory was associated with type \"%s\"\n", (const char *)type);
	
	return 0;
}

bool ClientVFS::createVInstances(VFolder *folder)
{
  QByteArray childrenInfo, childInfo;
  QStringList childrenList;
	VObjectFactory *fact=factoryDict[folder->type()];
	bool bRet=true;

  if( !BrokerInterface::thisInstance()->getChildrenData(folder->path(),childrenInfo) )
  {
    printf("\nNothing to create!\n");
    fflush(stdout);
    return true;
  }
  printf("\n getChildren ----- Done!");
  fflush(stdout);

  // read the list of childs
  QDataStream infoStream(childrenInfo, IO_ReadOnly);
  // Read the number of objects
  unsigned uObjectsNo=0;
  infoStream >> uObjectsNo;
  if( !uObjectsNo )
  {
    printf("\nNothing to create!\n");
    fflush(stdout);
    return true;
  }
  printf("\nThe number of children to create %u !\n",uObjectsNo);
  fflush(stdout);
  // Read the objects list
  infoStream >> childrenList;

  // get the client urls
  childrenList=reverseMap(childrenList);
  // build all the children
  for(QStringList::Iterator child=childrenList.begin(); bRet && child!=childrenList.end(); ++child)
  {
    infoStream >> childInfo;
    // build the new mail object using the childInfo
   	if(useFact && fact)
      bRet=fact->_new((*child), childInfo);
   	else if(genNonRegistered)
   		bRet=_new<VObject>((*child), childInfo);
  }

  return bRet;
}

bool ClientVFS::createVInstances(QStringList &paths)
{
  QByteArray objectsInfo, objectInfo;
  QStringList objectsList;
  bool bRet=true;

  if( !BrokerInterface::thisInstance()->getServerObjectsData(paths, objectsInfo) )
    return true;

  // read the list of childs
  QDataStream infoStream(objectsInfo, IO_ReadOnly);
  // Read the number of objects
  unsigned uObjectsNo=0;
  infoStream >> uObjectsNo;
  if( !uObjectsNo )
  {
    printf("\nNothing to create!\n");
    fflush(stdout);
    return true;
  }
  printf("\nThe number of children to create %u !\n",uObjectsNo);
  fflush(stdout);
  // Read the objects list
  infoStream >> objectsList;

  // get the client urls
  objectsList=reverseMap(objectsList);
  // build all the children
  for(QStringList::Iterator objectIt=objectsList.begin(); bRet && objectIt!=objectsList.end(); objectIt++)
  {
  	if(!(*objectIt).contains("."))
  	{
  		QString folderPath=(*objectIt).left((*objectIt).findRev("/"));
  		VFolder *folder=folderRef(folderPath);
  		
  		if(folder)
  		{
        // get the factory
      	VObjectFactory *fact=factoryDict[folder->type()];

        infoStream >> objectInfo;
        // build the new mail object using the childInfo
       	if(useFact && fact)
          bRet=fact->_new((*objectIt), objectInfo);
       	else if(genNonRegistered)
       		bRet=_new<VObject>((*objectIt), objectInfo);
      }
    }
  }
  return bRet;
}

bool ClientVFS::createVInstances()
{
	// iterate our folder collection
	QDictIterator<VFolder> it(folders);
	VFolder *tFolder;
	while((tFolder=it.current()))
	{
		if(!createVInstances(tFolder))
		{
			printf("Warning: ClientVFS: createVInstances(VFolder *) returned false when true was expected, bailing out\n");
			return false;
		}
		++it;
	}
	return true;
}

bool ClientVFS::rehashHierarchy()
{
	folders.clear();
	currentPath=root();
	pStack.clear();
	return initHierarchy();
}

bool ClientVFS::rehashVInstances()
{
	return createVInstances();
}

QDict<VFolder> ClientVFS::folderDict()
{
	return folders;
}

QString ClientVFS::parentFolder(QString path)
{
	int depth=path.contains('/');
	
	// test whether the root was given here, and if so, we keep it
	if(depth<3) return root();

	return path.left(path.findRev("/"));	
}

VFolder *ClientVFS::parentFolder(VFolder *folder)
{
	return folderRef(parentFolder(folder->path()));
}

QStringList ClientVFS::folderList(QString folder)
{
	QStringList fList;
	int depth=folder.contains('/');
	QDictIterator<VFolder> it(folders);
	while(it.current())
	{
		int sDepth=it.currentKey().contains('/');
		if(sDepth==depth+1 && it.currentKey().contains(folder))
			fList.append(it.currentKey());
		++it;
	}
	fList.sort();
	return fList;
}

void ClientVFS::autogenTemporary(bool temp)
{
	autogenTemp=temp;
}

void ClientVFS::useFactories(bool fact)
{
	useFact=fact;
}

void ClientVFS::autogenVInstances(bool vinst, bool nonreg)
{
	genVInstances=vinst;
	genNonRegistered=nonreg;
}

VFolder *ClientVFS::folderRef(QString folder)
{
	VFolder *ref=folders[folder];
	if(ref)
		return ref;
	if(autogenTemp)
	{
		if(folder.contains('/')==2 && folder.left(9)=="/Magellan")
			return _new(folder, "temporary", "", ""); // a valid non-existent folder like "/Magellan/abcd" was given
		else if(!folder.contains('/') && !folders[folder.prepend("/Magellan/")])
			return _new(folder.prepend("/Magellan/"), "temporary", "", ""); // a folder like "abcd" was given, and "/Magellan/abcd" does not exist
		else return _new(genTempFolderName(), "temporary", "", ""); // only "temporary" was given, or some wrong name
	}
	return 0;
}

VFolder *ClientVFS::folderRef(QString parent, QString name)
{
	return folderRef(parent+"/"+name);
}

VFolder *ClientVFS::folderRef(VFolder *parent, QString name)
{
	if(!parent) return folderRef(name);
	return folderRef(parent->path()+"/"+name);
}

VObject *ClientVFS::objectRef(QString object)
{
  // if the object is null
  if(object.isEmpty())
    return 0;
	
	QString folder=object.left(object.findRev('/'));
	VFolder *f=folders[folder];
	
	if(f)
		return f->objectDict()[object.mid(object.findRev('/')+1)];
	
	// the object does not exist
	return 0;
}

VObject *ClientVFS::objectRef(VFolder *parent, QString id)
{
	if(!parent || id.isEmpty())
	  return 0;
	return parent->objectDict()[id];
}

VObject *ClientVFS::objectRef(QString folder, QString id)
{
  if(folder.isEmpty() || id.isEmpty())
    return 0;
	VFolder *f=folders[folder];
	if(f) return f->objectDict()[id];
	return 0;
}

VReference *ClientVFS::reference(QString url)
{
	if(url.isEmpty()) return 0;
	
	VReference *ref=objectRef(url);
	if(ref) return ref;
	return folderRef(url);
}

void ClientVFS::registerFactory(QString type, VObjectFactory *factory)
{
	if(type=="temporary" || !type || !factory) return;
	factoryDict.insert(type, factory);
}

bool ClientVFS::cd(QString path)
{
	// test to see if this path is the root, which is not included in the folder dict
	if(path==root())
	{
		currentPath=path;
		return true;
	}
	
	// test to see if this is an absolute path
	if(folders[path])
	{
		currentPath=path;
		return true;
	}

	// test to see if this is a relative path
	QString tgtPath=currentPath+"/"+path;
	if(folders[tgtPath])
	{
		currentPath=tgtPath;
		return true;
	}
	
	// path is not absolute nor relative
	return false;
}

QStringList ClientVFS::dir(QString params)
{
	QStringList dirList(folderList(currentPath));
	if(params.contains("-l"))
		return dirList;
	
	// short dir requested, therefore strip all the paths
	QStringList sDirList;
	for(int i=0;i<dirList.count();i++)
		sDirList.append(dirList[i]);
	sDirList.sort();
	return sDirList;
}

void ClientVFS::cdUp()
{
	cd(parentFolder(currentPath));
}

void ClientVFS::cdRoot()
{
	cd(root());
}

QString ClientVFS::pwd()
{
	return currentPath;
}

QString ClientVFS::root()
{
	return "/Magellan";
}

void ClientVFS::pushPath()
{
	pStack.append(currentPath);
}

void ClientVFS::popPath()
{
	if(!pStack.count()) return;
	currentPath=*pStack.last();
	pStack.remove(pStack.fromLast());
}

QString ClientVFS::genTempFolderName()
{
	QStringList fList=folderList(root());
	int k=0;
	QString tName;
	tName.sprintf("/Magellan/Temp%d", k++);
	while(fList.find(tName)!=fList.end())
		tName.sprintf("/Magellan/Temp%d", k++);
	return tName;
}

void ClientVFS::saveStructure()
{
	QStringList fList=folderList(root());
	QStringList sList;
	QString type;
	GlobalConfig->setGroup("VFS Init");
	for(int i=0;i<fList.count();i++)
	{
		type=folders[fList[i]]->type();
		if(type!=QString::null && type!="temporary");
			sList.append(folders[fList[i]]->path());
	}
	GlobalConfig->writeEntry("TopLevel", sList);
	for(int i=0;i<fList.count();i++)
		if(folders[fList[i]]->type()!="temporary")
			saveFolderBranch(folders[fList[i]]);

	// write the configuration
	GlobalConfig->sync();
}

void ClientVFS::saveFolderBranch(VFolder *folder)
{
	saveFolder(folder);
	
	QStringList fList=folderList(folder->path());
	
	for(int i=0;i<fList.count();i++)
		if(folders[fList[i]]->type()!="temporary")
			saveFolderBranch(folders[fList[i]]);
}

void ClientVFS::saveFolder(VFolder *folder)
{
	QString group="VFS ";
	group+=folder->path();
	GlobalConfig->setGroup(group);
	GlobalConfig->writeEntry("Type", folder->type());
	GlobalConfig->writeEntry("View type", folder->viewType());
	GlobalConfig->writeEntry("MIME type", folder->mimeType());
	GlobalConfig->writeEntry("Binding", folder->binding());
	GlobalConfig->writeEntry("Server Path", folder->serverPath());
	GlobalConfig->writeEntry("Flags", folder->flags());
	
	pushPath();
	cd(folder->path());
	QStringList sList=dir();
	popPath();
	
	GlobalConfig->writeEntry("Subfolders", sList);
}

QString ClientVFS::reverseMap(QString path)
{
	if(path.isEmpty()) return QString::null;

	// check to see if this is a folder
	QDictIterator<VFolder> it(folders);
	VFolder *_folder;
	while((_folder=it.current()))
	{
		if(_folder->serverPath()==path)
			return _folder->path();
		++it;
	}
	
	// check to see if it's a simple object
	if(!path.contains("/")) return path;
	
	// maybe it's an object
	QString id=path.mid(path.findRev("/")+1); // ids are the same for the client and server
	QString sPath=path.left(path.findRev("/"));
	QString folder;
	QDictIterator<VFolder> _it(folders);
	while((_folder=_it.current()))
	{
		if(_folder->serverPath()==sPath)
			folder=_folder->path();
		++_it;
	}
	
	if(!folder.isNull())
		return folder+"/"+id;
			
	// a reverse mapping was not found
	return QString::null;
}

QStringList ClientVFS::reverseMap(QStringList list)
{
	for(int i=0;i<list.count();i++)
		list[i]=reverseMap(list[i]);
	return list;
}

QString ClientVFS::map(QString url)
{
	if(!url || url.isNull() || url.isEmpty()) return QString::null;

	// strip query
	QString query;
	int qdx=url.find("%");
	if(qdx>-1)
	{
		query=url.mid(qdx);
		url=url.left(qdx);
	}
	
	// strip subobject
	QString part;
	int idx=url.find(".");
	if(idx>-1)
	{
		part=url.mid(idx);
		url=url.left(idx);
	}
		
	// try to get the reference
	VReference *ref=reference(url);
	if(ref)
  {
		if(!ref->binding().isNull())
			return ref->serverPath()+part+query;
  }
  else
  {
    ref=reference(url.left(url.findRev('/'))); // maybe it's a not-yet-existent object
    if(ref)
			if(!ref->binding().isNull())
	      return ref->serverPath()+"/"+url.mid(url.findRev('/')+1)+part+query;
  }
	
	// no server mapping found
	return QString::null;
}

QStringList ClientVFS::map(QStringList urlList)
{
	for(int i=0;i<urlList.count();i++)
		urlList[i]=map(urlList[i]);
	return urlList;
}

bool ClientVFS::canDefine(QString url)
{
	if(BrokerInterface::thisInstance()->exists(url)) return true;
	VFolder *f=folderRef(url.left(url.findRev('/')));
	if(f)
		if(f->type()=="temporary")
			return true;
	return false;
}

void ClientVFS::updateVInstances(QStringList &paths)
{
  for(QStringList::Iterator it=paths.begin(); it!=paths.end(); it++)
    updateVInstance((*it));
}

void ClientVFS::updateVInstance(QString path)
{
	// printf("Notify: object %s was changed on server, updating\n", (const char *)path);

	QString localPath=reverseMap(path);
	VReference *obj=reference(localPath);
	if(obj)
	{
		// debug
		// printf("Notify: object found (%p), asking for update\n", obj);

		obj->updateViews();
	}
	else
	{
		// debug
		// printf("Notify: could not find object [path:%s, reverse:%s]\n", (const char *)path, (const char *)localPath);
	}
}

void ClientVFS::deleteVInstance(QString path)
{
	// debug
	// printf("Notify: object %s was deleted on server, deleting\n", (const char *)path);
	
	QString localPath=reverseMap(path);
	VReference *obj=reference(localPath);
	if(obj)
	{
		// debug
		// printf("Notify: object %s is about to be deleted\n", (const char *)localPath);

		delete obj;
	}
}

void ClientVFS::createVInstance(QString path)
{
	if(!path.contains("."))
	{
		QString localPath=reverseMap(path);

		// debug
		// printf("Notify: object %s is about to be created\n", (const char *)localPath);

		QString folderPath=localPath.left(localPath.findRev("/"));
		VFolder *folder=folderRef(folderPath);

		if(folder)
		{
			if(useFact)
			{
				_fact_ref(localPath, folder->type());
			}
			else
			{
				printf("Warning: cannot use factories, local instance was not created\n");
			}
		}
		else
		{
			printf("Warning: cannot find parent folder (hierarchy failure?), local instance not created\n");
		}
	}
	else
	{
		printf("Warning: cannot create attributes or subobjects automatically, discarding\n");
	}
}

void ClientVFS::initFolderCache(const QString &type)
{
	QDictIterator<VFolder> it(folders);
	VFolder *folder;
	while((folder=it.current()))
	{
		if(folder->type()==type)
			_new<VCacheObject>(folder->path()+"/"+"dir.cache");
		++it;
	}
}

void ClientVFS::discardFolderCache(const QString &type)
{
	QDictIterator<VFolder> it(folders);
	VFolder *folder;
	while((folder=it.current()))
	{
		if(folder->type()==type)
		{
			VCacheObject *cacheObj=(VCacheObject *)objectRef(folder, "dir.cache");
			if(cacheObj) delete cacheObj;
		}
		++it;
	}
}

// create only a virtual folder
void ClientVFS::createFolder(const QString &parent, const QString &name, const QString &type, const QString &viewType, const QString &serverPath)
{
  QString parentFolder;
  // Account -> /Magellan/Mail/Account
  if( type=="imap" )
    parentFolder=root()+"/Mail/"+parent;
  else
    parentFolder=parent;

  VFolder* vParent=folderRef(parentFolder);
  if( !vParent)
		// debug
		printf("clientvfs: warning: could not create the VFS %s folder\n", name.latin1());
  else
  {
    QString type=vParent->type();
    QString folderPath=parentFolder+"/"+name;

  	if(!serverPath.isEmpty() || viewType=="genmail")
  	{
  		// building the view type
  		QString vType;
  		QString mimeType;
  		
  		if( viewType.isNull() || viewType.isEmpty() )
  		  vType=type;
  		else
  		  vType=viewType;
  		
  		if(vType=="mail") mimeType="text/rfc822";
  		else if(vType=="genmail") mimeType="";
  		else if(vType=="notes") mimeType="application/x-notes";
  		else if(vType=="contacts") mimeType="text/directory";
  		else if(vType=="groups") mimeType="application/x-groups";
  		
  		// Create the folder
  		VFolder *newFolder=_new(folderPath, type, vType, mimeType, serverPath.isEmpty()?"":"thisInstance", serverPath);
  		if( newFolder )
  		{
  			newFolder->setFlags("c");
  			saveStructure();
  			MainWindow::ref()->createFolderView(newFolder);
  		}
  		else
  			// debug
  			printf("clientvfs: warning: could not create folder [spath:%s, type:%s]\n", (const char *)serverPath, (const char *)type);
  	}
  	else
  		// debug
  		printf("clientvfs: warning: could not create folder [spath:%s, type:%s], device expected\n", (const char *)serverPath, (const char *)type);
  }
}

VFolder *ClientVFS::createFolder(VFolder *parent, const QString &name, const QString &type, const QString &viewType)
{
	// a list of parameter for creating a new folder on the server
	QStringList paramList;
  QString folderPath=parent?parent->path()+"/"+name:"/Magellan/"+name;
	QString serverPath;
	
	// type and viewType for knowing the server folder type
	paramList.append(type);
	paramList.append(viewType);
	if( type=="imap" )
	{
  	int iFind1=0, iFind2=0;
  	
    //  folderPath=/Magellan/Mail/Account/Mailbox
    for(unsigned i=0; i<2 && iFind1!=-1; ++i)
      iFind1=folderPath.find('/', iFind1+1);
    iFind2=folderPath.find('/', iFind1+1);
    if( iFind1==-1 || iFind2==-1 )
      return 0;

	  paramList.append(folderPath.mid( iFind1+1, iFind2 - (iFind1+1) ));
	  paramList.append(folderPath.right( folderPath.length() - (iFind2+1) ));
	}
	
	// debug
	// printf("clientvfs: asking brokerinterface to create a [%s] device\n", (const char *)devType);
	
	if( type!="genmail" ) serverPath=BrokerInterface::thisInstance()->createFolder(paramList);
	else printf("clientvfs: warning: new folder [%s] has no associated device\n", (const char *)name);
	
	// For IMAP the design is different
	// The folder will be created after the IMAP server will be ready
	if( type!="imap" )
	{
  	if(!serverPath.isEmpty() || type=="genmail")
  	{
  		// building the view type
  		QString vType;
  		QString mimeType;
  		
  		if( viewType.isNull() || viewType.isEmpty() )
  		  vType=type;
  		else
  		  vType=viewType;
  		
  		if(vType=="mail") mimeType="text/rfc822";
  		else if(vType=="genmail") mimeType="";
  		else if(vType=="notes") mimeType="application/x-notes";
  		else if(vType=="contacts") mimeType="text/directory";
  		else if(vType=="groups") mimeType="application/x-groups";
  		
  		// Create the folder
  		if(VFolder *newFolder=_new(folderPath, type, vType, mimeType, serverPath.isEmpty()?"":"thisInstance", serverPath))
  		{
  			newFolder->setFlags("c");
  			saveStructure();
  			return newFolder;
  		}
  		else
  		{
  			// debug
  			printf("clientvfs: warning: could not create folder [spath:%s, type:%s]\n", (const char *)serverPath, (const char *)type);
  		}
  	}
  	else
  	{
  		// debug
  		printf("clientvfs: warning: could not create folder [spath:%s, type:%s], device expected\n", (const char *)serverPath, (const char *)type);
  	}
	}
	
	return 0;
}

void ClientVFS::deleteFolder(const QString &parent, const QString &folderName, const QString &type)
{
  QString folderPath;
  // Account -> /Magellan/Mail/Account
  if( type=="imap" )
    folderPath=root()+"/Mail/"+parent+"/"+folderName;
  else
    folderPath=parent+"/"+folderName;

  deleteVFolder(folderPath);
}

bool ClientVFS::deleteFolder(const QString &name)
{
	QStringList subFolders=folderList(name);
	
	for(int i=0;i<subFolders.count();i++)
		if(!deleteFolder(subFolders[i]))
		  return false;
	
	VFolder *folder=folderRef(name);
	if( folder )
	{
  	QString folderPath=folder->path();
  	QStringList paramList;
  	QString serverPath=folder->serverPath();
  	
  	// type and viewType for knowing the server folder type
  	paramList.append(folder->type());
  	paramList.append(folder->viewType());
  	if( serverPath.isNull() || serverPath.isEmpty() )
      paramList.append("default");
    else
      paramList.append(serverPath);
  	if( folder->type()=="imap" )
  	{
    	int iFind1=0, iFind2=0;
    	
      //  folderPath=/Magellan/Mail/Account/Mailbox
      for(unsigned i=0; i<2 && iFind1!=-1; ++i)
        iFind1=folderPath.find('/', iFind1+1);
      iFind2=folderPath.find('/', iFind1+1);
      if( iFind1==-1 || iFind2==-1 )
        return 0;

  	  paramList.append(folderPath.mid( iFind1+1, iFind2 - (iFind1+1) ));
  	  paramList.append(folderPath.right( folderPath.length() - (iFind2+1) ));
      BrokerInterface::thisInstance()->removeFolder(paramList);
  	}
  	else
  	{
  	  if( !folder->binding().isNull() && !folder->binding().isEmpty() )
  	    BrokerInterface::thisInstance()->removeFolder(paramList);
  	  deleteVFolder(folder);
  	}

  	return true;
	}
	else
	  return false;
}

void ClientVFS::deleteVFolder(const QString &name)
{
	VFolder *folder=folderRef(name);
	
	if( folder )
	{
  	delete folder;
  	folder=0;
  	folders.remove(name);
    saveStructure();
	}
}

void ClientVFS::deleteVFolder(VFolder *folder)
{
  if( folder )
  {
  	folders.remove(folder->path());
  	delete folder;
  	folder=0;
    saveStructure();
  }
}

bool ClientVFS::moveFolder(const QString &path, const QString &newPath)
{
	if(folders[path])
	{	
		QDictIterator<VFolder> it(folders);
		QStringList folderList;

		while(it.current())
		{
			folderList.append(it.currentKey());
			++it;
		}

		for(int i=0;i<folderList.count();i++)
		{
			QString oldFolderPath=folderList[i];

			if(oldFolderPath.find(path)==0)
			{
				QString newFolderPath=newPath+oldFolderPath.mid(path.length());

				VFolder *_folder=folders.take(oldFolderPath);

				_folder->folderPath=newFolderPath;
				folders.insert(newFolderPath, _folder);
			}
		}
		
		saveStructure();
		
		return true;
	}
	else
	{
		return false;
	}
}
























