/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <commframe.h>
#include <qdatastream.h>
#include <iostream.h>

CommFrame::CommFrame()
{
	reset();
}

CommFrame::CommFrame(QByteArray data)
{
	setData(data);
}

bool CommFrame::setData(QByteArray data)
{
	// reset frame
	reset();
	
	// check if we have at least the whole header
	if(data.size()<14) 
	{
		printf("Invalid frame (length<14), bailing out\n");
		return false;
	}
	
	// get a data stream
	QDataStream s(data, IO_ReadOnly);
	
	// check the Magellan signature byte
	unsigned char t,u;
	s>>t;
	if(t!='M') 
	{
		printf("Invalid frame (wrong signature byte), bailing out\n");
		return false;
	}
		
	// get the software version
	s>>t;
	u=t;
	t=t>>4;
	u=u<<4;
	u=u>>4;
	majorRevNo=t;
	minorRevNo=u;
	
	// check software version (only 1.1 is accepted right now)
	if(majorRevNo!=1||minorRevNo!=1) 
	{
		printf("Invalid frame (wrong software revision), bailing out\n");
		return false;
	}
	
	// get the frame characteristics
	s>>t;
	s>>u; // reserved for future use
	if(t&0x80) requestFrame=true;
	else replyFrame=true;
	if(t&0x40) clientFrame=true;
	else serverFrame=true;
	
	// test the controlled reply behavior
	if(!t&0x20)
	{
		if(t&0x10) requestsReply=true;
		else deniesReply=true;
	}
	if(t&0x08) hasStatus=true;
	if(t&0x04) hasData=true;
	
	// test the frame characteristics
	if(requestFrame==replyFrame
			|| requestsReply&&deniesReply
			|| serverFrame==clientFrame)
	{
		printf("Invalid frame (wrong combination of flags), bailing out\n");
		return false;
	}
		 
	// get the transaction number
	s>>transactionTag;
	
	// get the sequence number
	s>>sequenceTag;
	
	// get the command code
	s>>commandCode;
	
	// get the status, if available
	if(hasStatus)
	{
		s>>statusCode;
		s>>statusString;
	}
	if(hasData)
		s>>extraData;
	
	// we should have reached the end of the stream
	if(!s.atEnd()) 
	{
		printf("Invalid frame (could not reach the end of data stream), bailing out\n");
		return false;
	}
	
	validity=true;
	printf("Frame successfully parsed.\n");
	return true;
}

bool CommFrame::isValid()
{
	return validity;
}

void CommFrame::reset()
{
	frameLength=0;
	majorRevNo=0;
	minorRevNo=0;
	requestFrame=false;
	replyFrame=false;
	serverFrame=false;
	clientFrame=false;
	requestsReply=false;
	deniesReply=false;
	hasStatus=false;
	hasData=false;
	transactionTag=0;
	sequenceTag=0;
	commandCode=0;
	statusCode=0;
	statusString=QString::null;
	extraData.truncate(0);
	validity=false;
}

CommFrame::operator QByteArray()
{
	QByteArray a;
	QDataStream s(a, IO_WriteOnly);
	
	// write the Magellan signature byte
	s<<(unsigned char)'M';
		
	// write the software version number
	s<<(unsigned char)0x11;
	
	// compute the type byte
	unsigned char t=0;
	if(requestFrame) t=t|0x80;
	if(clientFrame) t=t|0x40;
	if(requestsReply||deniesReply) 
	{
		t=t|0x20;
		if(requestsReply) t=t|0x10;
	}
	if(hasStatus) t=t|0x08;
	if(hasData) t=t|0x04;
	
	// write the frame type (2 bytes, 10 bits reserved for future use)
	s<<(unsigned char)t;
	s<<(unsigned char)0;
	
	// write the transaction number
	s<<(unsigned int)transactionTag;
	
	// write the sequence number
	s<<(unsigned int)sequenceTag;
	
	// write the command code
	s<<(unsigned int)commandCode;
	
	// write the frame data
	if(hasStatus)
	{
		s<<(unsigned int)statusCode;
		s<<statusString;
	}
	if(hasData)
		s<<extraData;
	
	return a;
}






