/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <etab.h>
#include <qpainter.h>

ETabWidget::ETabWidget(QWidget *parent, const char *name):QWidget(parent, name)
{
}

void ETabWidget::setLabel(const QString &text)
{
	label=text;
	repaint();
}

void ETabWidget::setPixmap(const QPixmap &pix)
{
	pixmap=pix;
	repaint();
}

void ETabWidget::paintEvent(QPaintEvent *p)
{
	int textXOffset, textYOffset, pixmapXOffset, pixmapYOffset;

	// compute offsets
	if(pixmap.isNull())
	{
		textXOffset=TAB_LEFT_SPACING+TAB_SLOPE_WIDTH+TAB_CONTENT_OFFSET;
		textYOffset=(height()+fontMetrics().ascent())/2;
	}
	else
	{
		pixmapXOffset=TAB_LEFT_SPACING+TAB_SLOPE_WIDTH+TAB_CONTENT_OFFSET;
		pixmapYOffset=(height()-pixmap.height())/2;
		textXOffset=pixmapXOffset+pixmap.width()+TAB_SPACING;
		textYOffset=(height()+fontMetrics().ascent())/2;
	}
	
	// erase area
	QPainter painter(this);
//	painter.eraseRect(0, 0, width(), height());

	int bottom;
	
	// draw tab
	
	painter.setPen(colorGroup().dark());
	bottom=height()-3;
	painter.moveTo(0, bottom);
	painter.lineTo(TAB_LEFT_SPACING, bottom);
	painter.lineTo(TAB_LEFT_SPACING+TAB_SLOPE_WIDTH, 0);
	painter.lineTo(width()-TAB_LEFT_SPACING-TAB_SLOPE_WIDTH, 0);
	painter.lineTo(width()-TAB_LEFT_SPACING, bottom);
	painter.lineTo(width(), bottom);
	
	// dark margin first
	painter.setPen(colorGroup().light());
	bottom=height()-2;
	painter.moveTo(1, bottom);
	painter.lineTo(TAB_LEFT_SPACING, bottom);
	painter.lineTo(TAB_LEFT_SPACING+TAB_SLOPE_WIDTH, 1);
	painter.lineTo(width()-TAB_LEFT_SPACING-TAB_SLOPE_WIDTH, 1);
	painter.lineTo(width()-TAB_LEFT_SPACING, bottom);
	painter.lineTo(width(), bottom);
			
	// draw text
	painter.setPen(colorGroup().text());
	painter.drawText(textXOffset, textYOffset, label);
		
	// draw pixmap, if applicable
	if(!pixmap.isNull())
	{
		painter.drawPixmap(pixmapXOffset, pixmapYOffset, pixmap);
	}
}

ETab::ETab(QWidget *parent, const char *name):QWidget(parent, name)
{
	lastTabIndex=0;
	currentTabIndex=-1;
	layout=0;
}

ETab::~ETab()
{
}

int ETab::addTab(QWidget *w, const QString &label, const QPixmap &inactivePixmap, const QPixmap &activePixmap)
{
	// avoid crashes if null pointers are passed in here
	if(!w) 
		return -1;
	
	// create a new tab structure
	ETabData *tdata=new ETabData;
	tabList.append(tdata);
	tdata->index=lastTabIndex;
	tdata->label=label;
	tdata->inactivePixmap=inactivePixmap;
	tdata->activePixmap=activePixmap;
	tdata->tab=new ETabWidget(this);
	tdata->tab->setFixedHeight(TAB_HEIGHT);
	tdata->tab->setLabel(label);
	tdata->widget=w;
	
	// reparent widget if necessary
	if(w->parentWidget()!=this)
		w->reparent(this, 0, QPoint(0, 0), false);
	
	// install the event filter
	tdata->tab->installEventFilter(this);
	
	// make this tab the current tab, and display everything
	currentTabIndex=lastTabIndex++;
	rearrange();
	
	return currentTabIndex;
}

int ETab::addTab(QWidget *w, const QString &label)
{
	return addTab(w, label, QPixmap(), QPixmap());
}

int ETab::addTab(QWidget *w, const QPixmap &inactivePixmap, const QPixmap &activePixmap)
{
	return addTab(w, QString::null, inactivePixmap, activePixmap);
}

bool ETab::removeTab(int index)
{
	for(int i=0;i<tabList.count();i++)
	{
		ETabData *tdata=tabList.at(i);
		if(tdata->index==index)
		{
			delete tdata->widget;
			delete tdata->tab;
			if(currentTabIndex==index)
				currentTabIndex=0;
			tabList.remove(i);
			delete tdata;
			return true;
		}
	}
	
	return false;
}

bool ETab::removeTab(QWidget *w)
{
	for(int i=0;i<tabList.count();i++)
	{
		ETabData *tdata=tabList.at(i);
		if(tdata->widget==w)
		{
			delete tdata->widget;
			delete tdata->tab;
			if(currentTabIndex==tdata->index)
				currentTabIndex=0;
			tabList.remove(i);
			delete tdata;
			return true;
		}
	}
	
	return false;
}

int ETab::tabCount()
{
	return tabList.count();
}

int ETab::tabIndex(QWidget *w)
{
	for(int i=0;i<tabList.count();i++)
	{
		ETabData *tdata=tabList.at(i);
		if(tdata->widget==w)
			return tdata->index;
	}
	
	return -1;
}

int ETab::currentTab()
{
	return currentTabIndex;
}

QWidget *ETab::currentTabWidget()
{
	for(int i=0;i<tabList.count();i++)
	{
		if(tabList.at(i)->index==currentTabIndex)
			return tabList.at(i)->widget;
	}
	
	return 0;
}

void ETab::activateTab(int tabIndex)
{
	if(tabIndex==currentTabIndex) return;
	
	bool found=false;
	for(int i=0;i<tabList.count() && !found;i++)
	{
		if(tabList.at(i)->index==tabIndex)
			found=true;
	}
	if(!found) return;
	
	emit tabSelected(tabIndex, currentTabIndex);
	currentTabIndex=tabIndex;
	rearrange();
	emit tabActivated(tabIndex);
}

void ETab::activateTab(QWidget *widget)
{
	activateTab(tabIndex(widget));
}

void ETab::rearrange()
{
	if(layout) delete layout;
	layout=new QVBoxLayout(this);
	
	for(int i=0;i<tabList.count();i++)
	{
		ETabData *tdata=tabList.at(i);
		layout->addWidget(tdata->tab);
		tdata->tab->show();
		if(tdata->index==currentTabIndex)
		{
			layout->addWidget(tdata->widget);
			tdata->widget->show();
			if(!tdata->activePixmap.isNull())
				tdata->tab->setPixmap(tdata->activePixmap);
			else if(!tdata->inactivePixmap.isNull())
				tdata->tab->setPixmap(tdata->inactivePixmap);
		}
		else
		{
			if(!tdata->inactivePixmap.isNull())
				tdata->tab->setPixmap(tdata->inactivePixmap);
			tdata->widget->hide();
		}
	}
	
	layout->activate();
}

bool ETab::eventFilter(QObject *o, QEvent *e)
{
	for(int i=0;i<tabList.count();i++)
	{
		if(o==tabList.at(i)->tab && e->type()==QEvent::MouseButtonRelease && tabList.at(i)->index!=currentTabIndex)
		{
			emit tabSelected(tabList.at(i)->index, currentTabIndex);
			currentTabIndex=tabList.at(i)->index;
			rearrange();
			emit tabActivated(currentTabIndex);
			return true;
		}
	}
	
	return false;
}
