/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <stdio.h>

#include <qtextstream.h>
#include <qcstring.h>
#include <qstringlist.h>
#include <qstrlist.h>
#include <qvaluelist.h>

#include <messageclass.h>
#include <addressclass.h>
#include <addresslistclass.h>
#include <dateclass.h>
#include <mimepart.h>
#include <headerclass.h>
#include <mimecodec.h>

MessageClass::MessageClass(const char *c)
{
  if(!c) return;

  QCString msg(c), header=getPartHeader(msg, 0, msg.length());
  header_length=header.length();

  // process the main header
  parseHeader(header);
  processPart(msg, 0, msg.size());
}

void MessageClass::parseHeader(QCString &h)
{
  QCString field, value, hline, sender, resent, xpriority, inrep="", ref="";
  QTextStream txt(h, IO_ReadOnly);

  while(!txt.atEnd())
  {
    hline=txt.readLine();
    int pos=hline.find(":");
    if(pos!=-1)
    {
      field=hline.mid(0, pos).lower();
      value=hline.mid(pos+2);
      stripSpaces(value); // this shouldn't be necessary, since more than one space is a protocol violation - but who knows..
      stripComments(value);

      // check for known fields
      if(field=="received")
        Received.append(value);
      else if(field=="date")
        Date=DateClass(value);
      else if(field=="from")
        From=AddressClass(value);
      else if(field=="reply-to")
        Reply_To=AddressClass(value);
      else if(field=="to")
        To=AddressListClass(value);
      else if(field=="cc")
        Cc=AddressListClass(value);
      else if(field=="bcc")
        Bcc=AddressListClass(value);
      else if(field=="subject")
        Subject=MIMECodec::translate(value);
      else if(field=="message-id")
        Message_ID=value;
      else if(field=="in-reply-to")
        inrep=value;
      else if(field=="organization")
        Organization=value;
      else if(field=="status")
        Status=value;
      else if(field=="priority" || field=="x-msmail-priority" || field=="x-priority")
        Priority=value;
      else if(field=="content-type")
        Content_Type=value;
      else if(field=="content-transfer-encoding")
        Content_Transfer_Encoding=value;
      else if(field=="references")
        ref=value;
      else if(field=="x-priority")
        xpriority=value;
      else if(field=="sender")
        sender=value;
      else if(field=="resent-from")
        resent=value;
    }
  }

  // strip "<>" from Message_ID
  stripParanthesis(Message_ID);

  // process In_Reply_To and References
  QStrList tlist;

  if(!inrep.isEmpty())
  {
    tlist=chunk(inrep);
    if(tlist.count())
      In_Reply_To=tlist.at(0);
  }

  if(!ref.isEmpty())
  {
    tlist=chunk(ref);
    for(unsigned int i=0;i<tlist.count();i++)
      References.append(tlist.at(i));
  }
  if(!In_Reply_To.isEmpty() && References.count())
    while(References.remove((const char *)In_Reply_To));

  // assign sensible values to Status
  if(Status=="O" || Status=="R")
    Status="Read";
  else
    Status="New";

  // parse the "Content-Type" structured field
  QStrList tokens=split(';', Content_Type);

  if(tokens.count()) Content_Type=tokens.at(0);

  // a digest message is something really special, that's why we test it here
  if(Content_Type.lower().contains("multipart/digest"))
    digest=true;
  else
    digest=false;

  // check some redundant values
  if(!(QCString)From && !sender.isEmpty())
    From=AddressClass(sender);
  if(!(QCString)From && !resent.isEmpty())
    From=AddressClass(resent);
  if(!Priority && !xpriority.isEmpty())
    Priority=xpriority;

  // check and beautify the priority value, as some mailers use numbers
  int pr=Priority.toInt();
  if(pr>0 && pr<6)
    switch(pr)
    {
      case 5:
        Priority="Lowest";
        break;
      case 4:
        Priority="Low";
        break;
      case 3:
        Priority="Normal";
        break;
      case 2:
        Priority="High";
        break;
      case 1:
        Priority="Highest";
    }
  else
  {
    Priority=Priority.lower();
    if(Priority!="lowest" && Priority!="low" && Priority!="normal" && Priority!="high" && Priority!="highest")
      Priority="normal";
    QString k=QChar(Priority[0]);
    k=k.upper();
    Priority.replace(0, 1, (const char *)k);
  }

  // give some default values for the very important fields
  QCString def="unknown", defd="Mon, 1 Jan 1999 0:0:0 UTF";
  if(!(QCString)From)
    From=AddressClass(def);
  if(!(QCString)To)
    To=AddressListClass(def);
  if(!Subject)
    Subject="(no subject)";
  if(!Date.day_of_week)
    Date=DateClass(defd);
  if(!Content_Type)
    Content_Type="text/plain"; // this is needed, for MIME-1.0 compliance
  if(!Content_Transfer_Encoding)
    Content_Transfer_Encoding="8bit"; // so is this
}

void MessageClass::stripQuotes(QCString &s)
{
  if(s.length() && s[0]=='"')
    s.remove(0, 1);
  if(s.length() && s[s.length()-1]=='"')
    s.truncate(s.length()-1);
}

void MessageClass::stripParanthesis(QCString &s)
{
  if(s.length() && s[0]=='<')
    s.remove(0, 1);
  if(s.length() && s[s.length()-1]=='>')
    s.truncate(s.length()-1);
}

void MessageClass::stripSpaces(QCString &s)
{
  char c;
  while(s.length() && ((c=s[0])==' ' || c=='\t' || c=='\n'))
    s.remove(0, 1);
  while(s.length() && ((c=s[s.length()-1])==' ' || c=='\t' || c=='\n'))
    s.truncate(s.length()-1);
}

void MessageClass::stripComments(QCString &s)
{
  int nesting=0;
  bool qnesting=false;
  QCString t;
  for(unsigned int i=0;i<s.length();i++)
  {
    char c=s[i];
    if(c=='"' && !nesting) qnesting=!qnesting;
    if(c=='(' && !qnesting)
      nesting++;
    if(nesting==0 || qnesting) t+=c;
    if(c==')' && nesting && !qnesting)
      nesting--;
  }
  s=t;
}

QStrList MessageClass::split(char c, QCString &s)
{
  QStrList t;
  QCString token;
  for(unsigned int i=0;i<s.length();i++)
  {
    if(s[i]==c)
    {
      stripSpaces(token);
      t.append(token);
      token.truncate(0);
    }
    else
      token+=s[i];
  }
  if(!token.isEmpty())
  {
    stripSpaces(token);
    t.append(token);
  }
  return t;
}

QStrList MessageClass::chunk(QCString &c)
{
  QStrList t;
  QCString token;

  for(unsigned int i=0;i<c.length();i++)
  {
    if(c[i]=='>' && !token.isEmpty())
    {
      t.append(token);
      token.truncate(0);
    }
    else if(c[i]=='<')
      token.truncate(0);
    else
      token+=c[i];
  }

  return t;
}

QCString MessageClass::getPartHeader(QCString &s, int offset, int length)
{
  // isolate and unfold the header
  QTextIStream txt(s);
  txt.device()->at(offset);
  QCString header=(const char *)txt.readLine(), hline="a";

  while(!hline.isEmpty() && !txt.atEnd() && txt.device()->at()<offset+length)
  {
    hline=(const char *)txt.readLine();
    if(!hline.isEmpty())
    {
      if(!(QChar(hline[0]).isSpace()))
        header.append("\n");
      else
        header.append(" ");

      stripSpaces(hline);

      header.append(hline);
    }
  }
  return header;
}

void MessageClass::processPart(QCString &s, int offset, int length, bool related)
{
  QCString header=getPartHeader(s, offset, length), field, value, type, encoding="8bit", boundary, name="attachment.dat",
       cid="unknown", hline, charset="us-ascii"; // some of the default values are required for MIME 1.0 compliance
  type=digest?"message/rfc822":"text/plain";
  QCString disposition;

  QTextIStream txt(header);

  // get the part header fields
  while(!txt.atEnd())
  {
    hline=(const char *)txt.readLine();
    int pos=hline.find(':');
    if(pos!=-1)
    {
      field=hline.left(pos).lower();
      value=hline.mid(pos+2);
      stripSpaces(value); // this shouldn't be necessary, since more than one space is a protocol violation - but who knows..
      stripComments(value);

      // check for known fields
      if(field=="content-type")
        type=value;
      else if(field=="content-transfer-encoding")
        encoding=value;
      else if(field=="content-id")
        cid=value;
      else if(field=="content-disposition")
        disposition=value;
    }
  }

  // additional parsing for cid, boudary, charset or attachment name

  QStrList tokens=split(';', type);

  if(tokens.count())
    type=tokens.at(0);
	else
		type="text/plain";

	// search for additional part attributes
	for(unsigned int i=0;i<tokens.count();i++)
	{
		QCString spfield=tokens.at(i);
    if(spfield.left(8).lower()=="charset=")
    {
      charset=spfield.mid(8);
      stripSpaces(charset);
      stripQuotes(charset);
    }
    else if(spfield.left(9).lower()=="boundary=")
    {
      boundary=spfield.mid(9);
      stripSpaces(boundary);
      stripQuotes(boundary);
    }
    else if(!disposition.isEmpty() && spfield.left(5).lower()=="name=")
    {
      name=spfield.mid(5);
      stripSpaces(name);
      stripQuotes(name);
    }
	}

  tokens=split(';', cid);
  if(tokens.count())
  {
    cid=tokens.at(0);
    stripParanthesis(cid);
  }

  // get the part offset
  QTextIStream parttxt(s);
  parttxt.device()->at(offset);
  int pOffset;
  QCString pLine=(const char *)parttxt.readLine();
  while(!pLine.isEmpty())
  {
    pLine=(const char *)parttxt.readLine();
  }
  pOffset=parttxt.device()->at()-1;

  // get the sub-part list, scanning for the boundary;
  QValueList<int> partOffset, partLength;

  // check the boundary
  if(!boundary.isEmpty())
  {
    boundary.prepend("\n--");
    QCString end_boundary=boundary;
    end_boundary.append("--");

    int begin=s.find(boundary, offset), end;

    while(begin!=-1)
    {
      end=s.find(boundary, begin+boundary.length());

      if(end!=-1 && end<offset+length-(int)end_boundary.size())
      {
        // found a valid part
        // Please don't forget the new line char to the end !!!
        QTextIStream textStream(s);
        textStream.device()->at(begin);
        textStream.readLine();
        begin=textStream.device()->at();
        partOffset.append(begin);
        partLength.append(end-begin);
        begin=end;

        if(s.mid(end, end_boundary.length())==end_boundary)
          begin=-1; // we found the end
      }
      else
      {
        // if we got on this branch there is either a mangled mail or a parser failure
        begin=-1;
      }
    }
  }

  if(partOffset.count()>0)
  {
    // check for the special case "multipart/related"
    processPart(s, partOffset[0], partLength[0]);

    if(type.lower().contains("multipart/related"))
    {
      for(unsigned int i=1;i<partOffset.count();i++)
        processPart(s, partOffset[i], partLength[i], true); // these are all related parts
    }
    else
      for(unsigned int i=1;i<partOffset.count();i++)
        processPart(s, partOffset[i], partLength[i]);
  }
  else
  {
    // either one simple part, or there was no boundary (broken mail), so dump it in one part
    QCString partOffset=(const char *)QString::number(pOffset);
    QCString partLength=(const char *)QString::number(length+offset-pOffset);
    type=type.lower();

    if(digest)
    {
      // add a digest part
      if(type=="message/rfc822")
      {
        QCString p="D ";
        p.append(partOffset);
        p.append(" ");
        p.append(partLength);
        partList.append(new MimePart(p));
      }
    }
    else
    {
      if(related)
      {
        // add a related-to part
        QCString p="R ";
        p.append(type);
        p.append(" ");
        p.append(cid);
        p.append(" ");
        p.append(encoding);
        p.append(" ");
        p.append(partOffset);
        p.append(" ");
        p.append(partLength);
        partList.append(new MimePart(p));
      }
      else
      {
        // If it is an attachment
        // "inline" means it must be in the view part (not like attachment)
        // TODO: view all inline parts (not only the text parts)
        if( disposition.lower().contains("attachment") ||
            ( disposition.lower().contains("inline") && 
              !type.contains("text/plain") && 
              !type.contains("text/html")
            )
          )
        {
          // ordinary attachment
          QCString p;
          p="A ";
          p.append(type);
          p.append(" ");
          p.append(encoding);
          p.append(" ");
          p.append(partOffset);
          p.append(" ");
          p.append(partLength);
          p.append(" ");
          p.append(name);
          partList.append(new MimePart(p));
        }
        // Text
        else if( type.contains("text/plain") )
        {
          QCString p="T text/plain ";
          p.append(encoding);
          p.append(" ");
          p.append(partOffset);
          p.append(" ");
          p.append(partLength);
          p.append(" ");
          p.append(charset);
          partList.append(new MimePart(p));
        }
        // Html
        else if(type.contains("text/html"))
        {
          QCString p="T text/html ";
          p.append(encoding);
          p.append(" ");
          p.append(partOffset);
          p.append(" ");
          p.append(partLength);
          p.append(" ");
          p.append(charset);
          partList.append(new MimePart(p));
        }
        else
        {
          printf("\naethera: message with unknown type!");
          printf("\nType: %s",(const char*)type);
          printf("\nPart: %*.*s\n",length, length, &s[offset]);
          fflush(stdout);
        }
      }
    }
  }
}

void MessageClass::importHeader(HeaderClass &h)
{
	From=h.From;
	Reply_To=h.Reply_To;
	To=h.To;
	Cc=h.Cc;
	Bcc=h.Bcc;
	Subject=h.Subject;
	Message_ID=h.Message_ID;
	In_Reply_To=h.In_Reply_To;
	Organization=h.Organization;
	Status=h.Status;
	Priority=h.Priority;
	Received=h.Received;
	References=h.References;
	Date=h.Date;
	Content_Type=h.Content_Type;
	Content_Transfer_Encoding=h.Content_Transfer_Encoding;
}

MessageClass::MessageClass(HeaderClass &h)
{
	importHeader(h);
	
	// set some defaults
	Content_Type="text/plain";
	Content_Transfer_Encoding="8bit";
}

HeaderClass MessageClass::header()
{
	HeaderClass h;
	h.From=From;
	h.Reply_To=Reply_To;
	h.To=To;
	h.Cc=Cc;
	h.Bcc=Bcc;
	h.Subject=Subject;
	h.Message_ID=Message_ID;
	h.In_Reply_To=In_Reply_To;
	h.Organization=Organization;
	h.Status=Status;
	h.Priority=Priority;
	h.Received=Received;
	h.References=References;
	h.Date=Date;
	h.Content_Type=Content_Type;
	h.Content_Transfer_Encoding=Content_Transfer_Encoding;
	return h;
}
