/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS
 * IN THE SOFTWARE.
 */

#include <qlistview.h>
#include <qheader.h>
#include <qevent.h>
#include <qfiledialog.h>
#include <qfileinfo.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <clientnotifier.h>
#include <vfolder.h>
#include <clientvfs.h>
#include <foldertree.h>
#include <foldertreeitem.h>
#include <mainwindow.h>
#include <renamedialog.h>
#include <destinationdialog.h>
#include <maillistview.h>
#include <qdragobject.h>
#include <contentswidget.h>

FolderTree::FolderTree(QWidget *parent, const char *name):QListView(parent, name)
{
	header()->hide();
	connect(this, SIGNAL(currentChanged(QListViewItem *)), this, SLOT(ackCurrentChanged(QListViewItem *)));
	connect(this, SIGNAL(expanded(QListViewItem *)), this, SLOT(expandItem(QListViewItem *)));
	connect(this, SIGNAL(collapsed(QListViewItem *)), this, SLOT(collapseItem(QListViewItem *)));

	QListView::setAcceptDrops(true);
	viewport()->setAcceptDrops(true);
}

void FolderTree::ackCurrentChanged(QListViewItem *item)
{
	emit currentChanged((FolderTreeItem *)(void *)item);
}

void FolderTree::contentsDragEnterEvent(QDragEnterEvent *e)
{
	// Preliminarily accept the drag, and store the current
	// selected folder

	e->accept();

	m_dndSourceFolder = currentItem();
}

void FolderTree::contentsDragLeaveEvent(QDragLeaveEvent *)
{
	// Drag has left widget, so restore current item to
	// previously selected folder

	setCurrentItem(m_dndSourceFolder);
}

void FolderTree::contentsDragMoveEvent(QDragMoveEvent *e)
{
	// Find out which listview item (if any) the cursor is over
	QPoint p(contentsToViewport(e->pos()));
	FolderTreeItem *atPos = (FolderTreeItem *)itemAt(p);
	
	if(!atPos) {
		// Cursor isn't over a folder, so reject the drop
		e->accept(false);
	} else {
		// Give the user some visual indication of which
		// folder they're dropping into
		e->accept(true);

		blockSignals(true);
		setCurrentItem(atPos);
		blockSignals(false);
	}
}

void FolderTree::contentsDropEvent(QDropEvent *e)
{
	QPoint p(contentsToViewport(e->pos()));

	FolderTreeItem *atPos = (FolderTreeItem *)itemAt(p);

	// restore previously selected folder
	setCurrentItem(m_dndSourceFolder);

	if(!atPos)
		return;

	printf("Folder path = [%s]\n", atPos->getVFolderPath().latin1());

        QStrList pathList;

        if (QUriDrag::decode(e, pathList))
        {
                for (unsigned int i=0;i<pathList.count();i++)
                {
                        QString sourcePath=(const char *)pathList.at(i);

                        BrokerInterface::thisInstance()->move(sourcePath,
                                                              atPos->getVFolderPath());
                }
        }
}

void FolderTree::integrateRootDirectory()
{
	QFileIconProvider *iconSource=new QFileIconProvider();
	FolderTreeItem *rootDir=new FolderTreeItem(this, 0);
	QFileInfo rootInfo(QDir::rootDirPath());
	
	rootDir->setText(0, "Root");
	if(iconSource->pixmap(rootInfo))
		rootDir->setPixmap(0, *iconSource->pixmap(rootInfo)); // temporary, until we get a nice icon
	rootDir->setExpandable(rootInfo.isReadable());
	rootDir->setFilePath(QDir::rootDirPath());
	
	delete iconSource;
}

void FolderTree::integrateHomeDirectory()
{
	QFileIconProvider *iconSource=new QFileIconProvider();
	FolderTreeItem *homeDir=new FolderTreeItem(this, 0);
	QFileInfo homeInfo(QDir::homeDirPath());
	
	homeDir->setText(0, "Home");
	if(iconSource->pixmap(homeInfo))
		homeDir->setPixmap(0, *iconSource->pixmap(homeInfo)); // temporary, until we get a nice icon
	homeDir->setExpandable(homeInfo.isReadable());
	homeDir->setFilePath(QDir::homeDirPath());
	
	delete iconSource;
}

void FolderTree::expandItem(QListViewItem *item)
{
	QFileIconProvider *iconSource=new QFileIconProvider();
	QString fsPath=((FolderTreeItem *)item)->getFilePath();
	
	if(!fsPath.isEmpty())
	{
		QDir itemDir(fsPath);
		QFileInfoList *infoList=new QFileInfoList(*itemDir.entryInfoList(QDir::Dirs));
		
		for (unsigned int i = 0; i < infoList->count(); i++)
		{
			QFileInfo *folderInfo=infoList->at(i);
			QString fileName=folderInfo->fileName();
			if(fileName!="." && fileName!="..")
			{
				FolderTreeItem *folderItem=new FolderTreeItem((FolderTreeItem *)item, 0);

				folderItem->setText(0, fileName);
				if(iconSource->pixmap(*folderInfo))
					folderItem->setPixmap(0, *iconSource->pixmap(*folderInfo));
				folderItem->setExpandable(folderInfo->isReadable());
				folderItem->setFilePath(folderInfo->filePath());
			}
		}
		
		delete infoList;
	}
	
	delete iconSource;
}

void FolderTree::collapseItem(QListViewItem *item)
{
	QString fsPath=((FolderTreeItem *)item)->getFilePath();
	
	if(!fsPath.isEmpty())
	{
		FolderTreeItem *childItem=(FolderTreeItem *)item->firstChild(), *nextChildItem=childItem;
		
		while(childItem)
		{
			nextChildItem=(FolderTreeItem *)childItem->nextSibling();
			delete childItem;
			childItem=nextChildItem;
		}
	}
}

void FolderTree::initFolderManagement()
{
	folderMenu=new QPopupMenu();
	folderMenu->insertItem("&Open in new window", this, SLOT(openInNewWindow()));
	folderMenu->insertItem("&Find...");
	folderMenu->insertSeparator();
	folderMenu->insertItem("&Delete folder", this, SLOT(deleteCurrentFolder()), 0, FOLDERTREE_DELETE);
	folderMenu->insertItem("&Move folder", this, SLOT(moveCurrentFolder()), 0, FOLDERTREE_MOVE);
	folderMenu->insertItem("&New folder...", this, SLOT(createFolder()));
	folderMenu->insertItem("&Rename folder...", this, SLOT(renameCurrentFolder()), 0, FOLDERTREE_RENAME);
	folderMenu->insertSeparator();
	folderMenu->insertItem("Mark &all as read", this, SLOT(markAllRead()), 0, FOLDERTREE_MARKALLREAD);
	folderMenu->insertSeparator();
	folderMenu->insertItem("&Properties", this, SLOT(editProperties()));
	
	connect(this, SIGNAL(rightButtonPressed(QListViewItem *, const QPoint &, int)), this, SLOT(showPopup(QListViewItem *, const QPoint &, int)));
}

void FolderTree::showPopup(QListViewItem *item, const QPoint &pos, int)
{
	VFolder *folder=0;
	if(item)
		folder=static_cast<VFolder *>(static_cast<FolderTreeItem *>(item)->getVItem());
		
	if(folder && (folder->type()=="mail" || folder->type()=="imap"))
		folderMenu->setItemEnabled(FOLDERTREE_MARKALLREAD, true);
	else
		folderMenu->setItemEnabled(FOLDERTREE_MARKALLREAD, false);
	
	if(folder && actionsAllowed(folder->path()))
	{
		folderMenu->setItemEnabled(FOLDERTREE_DELETE, true);
		folderMenu->setItemEnabled(FOLDERTREE_MOVE, true);
		folderMenu->setItemEnabled(FOLDERTREE_RENAME, true);
	}
	else
	{
		folderMenu->setItemEnabled(FOLDERTREE_DELETE, false);
		folderMenu->setItemEnabled(FOLDERTREE_MOVE, false);
		folderMenu->setItemEnabled(FOLDERTREE_RENAME, false);
	}
	
	folderMenu->popup(pos);
}

void FolderTree::deleteCurrentFolder()
{
	FolderTreeItem *item=(FolderTreeItem *)selectedItem();
	if(item)
	{
		VFolder *folder=static_cast<VFolder *>(item->getVItem());
		if(folder
			  && QMessageBox::warning(MainWindow::ref(), "Aethera: Confirm delete",
					"Deleting a folder will also delete all its subfolders.\n"
					"This action cannot be undone. Are you sure?", QMessageBox::Yes, QMessageBox::No, 0)==QMessageBox::Yes)
		{
		  ContentsWidget::ref()->deleteCurrentFolder();
			ClientVFS::thisInstance()->deleteFolder(folder->path());
		}
	}
}

bool FolderTree::actionsAllowed(const QString &path)
{
	if(path=="/Magellan"
			|| path=="/Magellan/Contacts"
			|| path=="/Magellan/Groups"
			|| path=="/Magellan/Mail"
			|| path=="/Magellan/Mail/Deleted Messages"
			|| path=="/Magellan/Mail/Drafts"
			|| path=="/Magellan/Mail/Inbox"
			|| path=="/Magellan/Mail/Outbox"
			|| path=="/Magellan/Mail/Saved Messages"
			|| path=="/Magellan/Mail/Sent Mail"
			|| path=="/Magellan/Mail/Templates"
			|| path=="/Magellan/Notes") return false;
			
	return true;
}

void FolderTree::createFolder()
{
	MainWindow::ref()->newFolder();
}

void FolderTree::moveCurrentFolder()
{
	FolderTreeItem *item=(FolderTreeItem *)selectedItem();
	
	if(item)
	{
		VFolder *folder=(VFolder *)item->getVItem();
		QString path=folder->path();
		
		DestinationDialog *d=new DestinationDialog("Move", QString::null, MainWindow::ref());
		int res=d->exec();

		if(res==QDialog::Accepted)
		{
			FolderTreeItem *dItem=d->selectedItem();

			if(dItem)
			{
				// move folder
				QString newParentPath=dItem->path();
				
				if(newParentPath.find(path)==-1)
				{
					VFolder *oldFolder=(VFolder *)ClientVFS::thisInstance()->folderRef(path);
					if( oldFolder->type()=="imap" )
					{
					  printf("\nMoving an IMAP folder is not available yet!\n");
					  return;
					}
					ClientVFS::thisInstance()->moveFolder(path, newParentPath+"/"+folder->name());
					
					item->parent()->takeItem(item);
					
					VFolder *parentFolder=(VFolder *)ClientVFS::thisInstance()->folderRef(newParentPath);
					if(parentFolder)
					{
						FolderTreeItem *parentItem=(FolderTreeItem *)parentFolder->getIconView();
						parentItem->insertItem(item);
					}
					else
					{
						firstChild()->insertItem(item);
					}
				}
			}
		}

		delete d;
	}
}

void FolderTree::renameCurrentFolder()
{
	FolderTreeItem *item=(FolderTreeItem *)selectedItem();
	if(item)
	{
		VFolder *folder=(VFolder *)item->getVItem();
		QString path=folder->path();
		if(folder)
		{
			RenameDialog *renameDialog=new RenameDialog(folder->name(), MainWindow::ref());
			if(renameDialog->exec()==QDialog::Accepted && renameDialog->getName()!=folder->name())
			{
				QString newPath=path.left(path.findRev("/"))+"/"+renameDialog->getName();
        if( folder->type()=="imap" )
        {
          printf("\nRename an IMAP folder is not available yet!\n");
          return;
        }
				ClientVFS::thisInstance()->moveFolder(folder->path(), newPath);
				
				item->setText(0, folder->name());
			}
			delete renameDialog;
		}
	}
}

void FolderTree::editProperties()
{
}

void FolderTree::openInNewWindow()
{
}

void FolderTree::markAllRead()
{
	FolderTreeItem *item=(FolderTreeItem *)selectedItem();
	if(item)
	{
		VFolder *folder=(VFolder *)item->getVItem();
		MailListView *view=dynamic_cast<MailListView *>(folder->getView());
		
		if(view) view->markAllRead();
		else printf("foldertree: [markAllRead] this is not a mail view\n");
	}
	else
	{
		printf("foldertree: [markAllRead] nothing is selected\n");
	}
}


