/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <foldertreeitem.h>
#include <foldertree.h>
#include <qlistview.h>
#include <vfolder.h>
#include <qpainter.h>
#include <qpalette.h>
#include <qbitmap.h>
#include <qapplication.h>
#include <vfolderattribute.h>
#include <clientvfs.h>

#define TAG_COLOR QColor(0, 0, 255)
#define TAG_SPACING 5
#define LEFT_BEARING 2
#define RIGHT_BEARING 2

static QBitmap * verticalLine = 0;
static QBitmap * horizontalLine = 0;

static void cleanupBitmapLines()
{
    delete verticalLine;
    delete horizontalLine;
    verticalLine = 0;
    horizontalLine = 0;
}

FolderTreeItem::FolderTreeItem(FolderTree *parent, VFolder *folder):QListViewItem(parent),TemplateView(folder)
{
  if(folder)
	{
		attr=(VFolderAttribute *)ClientVFS::thisInstance()->_fact_ref(folder->path()+"/default.fattr", "vfolderattribute");
		acceptedTypes=folder->mimeType();
		m_vfolderPath=folder->path();
	}
	else
	{
		attr=0;
	}
	
	if(attr)
	{
		attr->sync();
		if(folder->viewType()=="outbox" || folder->viewType()=="drafts")
			tag=attr->items;
		else
			tag=attr->recent;
	}
	else
	{
		tag=0;
	}
}

FolderTreeItem::FolderTreeItem(FolderTreeItem *parent, VFolder *folder):QListViewItem(parent),TemplateView(folder)
{
  if(folder) 
	{
		attr=(VFolderAttribute *)ClientVFS::thisInstance()->_fact_ref(getVItem()->path()+"/default.fattr", "vfolderattribute");
		acceptedTypes=folder->mimeType();
		m_vfolderPath=folder->path();
	}
	else
	{
		attr=0;
	}
	
	if(attr)
	{
		attr->sync();
		if(folder->viewType()=="outbox" || folder->viewType()=="drafts")
			tag=attr->items;
		else
			tag=attr->recent;
	}
	else
	{
		tag=0;
	}
}

FolderTreeItem::~FolderTreeItem()
{
}

void FolderTreeItem::update()
{
	// debug
	// printf("foldertreeitem: [%s] update received\n", (const char *)path());
	
	if(attr)
	{
		// debug
		// printf("foldertreeitem: syncing attribute...\n");
		
		attr->sync();
		
		// debug
		// printf("foldertreeitem: attr [recent:%d, items:%d]\n", attr->recent, attr->items);

		int newTag=0;
		VFolder *folder=(VFolder *)getVItem();

		if(folder)
		{
			if(folder->viewType()=="outbox" || folder->viewType()=="drafts")
				newTag=attr->items;
			else
				newTag=attr->recent;
		}
		
		if(tag!=newTag)
		{
			tag=newTag;
			repaint();
		}
	}
}

void FolderTreeItem::paintBranches(QPainter *p, const QColorGroup &cg, int w, int y, int h, GUIStyle s)
{
  p->fillRect( 0, 0, w, h, cg.base() );
  QListViewItem * child = firstChild();
  int linetop = 0, linebot = 0;

  int dotoffset = (itemPos() + height() - y) %2;

  // each branch needs at most two lines, ie. four end points
  QPointArray dotlines( childCount() * 4 );
  int c = 0;

  // skip the stuff above the exposed rectangle
  while ( child && y + child->height() <= 0 ) 
  {
	  y += child->totalHeight();
	  child = child->nextSibling();
  }

  int bx = w / 2;

  // paint stuff in the magical area
  while ( child && y < h ) 
  {
	  linebot = y + child->height()/2;
	  if ( (child->isExpandable() || child->childCount()) && (child->height() > 0) ) 
    {
	     // needs a box
	     p->setPen( cg.text() );
	     p->drawRect( bx-4, linebot-4, 9, 9 );
	     p->setPen( cg.text() ); // ### windows uses black
       
		   // plus or minus
		   p->drawLine( bx - 2, linebot, bx + 2, linebot );
		   if ( !child->isOpen() )
		   p->drawLine( bx, linebot - 2, bx, linebot + 2 );
       
	     // dotlinery
	     dotlines[c++] = QPoint( bx, linetop );
	     dotlines[c++] = QPoint( bx, linebot - 5 );
	     dotlines[c++] = QPoint( bx + 5, linebot );
	     dotlines[c++] = QPoint( w, linebot );
	     linetop = linebot + 5;
	  } 
    else 
    {
	    // just dotlinery
	    dotlines[c++] = QPoint( bx+1, linebot );
	    dotlines[c++] = QPoint( w, linebot );
	  }

	  y += child->totalHeight();
	  child = child->nextSibling();
	}

  if ( child ) // there's a child, so move linebot to edge of rectangle
		linebot = h;

  if ( linetop < linebot ) 
  {
	  dotlines[c++] = QPoint( bx, linetop );
	  dotlines[c++] = QPoint( bx, linebot );
  }

  p->setPen( cg.dark() );
  
	if ( !verticalLine ) 
  {
	  // make 128*1 and 1*128 bitmaps that can be used for
	  // drawing the right sort of lines.
	  verticalLine = new QBitmap( 1, 129, TRUE );
	  horizontalLine = new QBitmap( 128, 1, TRUE );
	  QPointArray a( 64 );
	  QPainter p;
	  p.begin( verticalLine );
	  int i;
	  for( i=0; i<64; i++ )
			a.setPoint( i, 0, i*2+1 );
	  p.setPen( QColor("white") );
	  p.drawPoints( a );
	  p.end();
	  QApplication::flushX();
	  verticalLine->setMask( *verticalLine );
	  p.begin( horizontalLine );
	  for( i=0; i<64; i++ )
			a.setPoint( i, i*2+1, 0 );
	  p.setPen( QColor("white") );
	  p.drawPoints( a );
	  p.end();
	  QApplication::flushX();
	  horizontalLine->setMask( *horizontalLine );
	  qAddPostRoutine( cleanupBitmapLines );
	}
	int line; // index into dotlines
	for( line = 0; line < c; line += 2 ) 
  {
	    // assumptions here: lines are horizontal or vertical.
	    // lines always start with the numerically lowest
	    // coordinate.

	    // point ... relevant coordinate of current point
	    // end ..... same coordinate of the end of the current line
	    // other ... the other coordinate of the current point/line
	  if ( dotlines[line].y() == dotlines[line+1].y() ) 
    {
		  int end = dotlines[line+1].x();
		  int point = dotlines[line].x();
		  int other = dotlines[line].y();
		  while( point < end ) 
      {
		    int i = 128;
		    if ( i+point > end )
					i = end-point;
				p->setPen(QColor("black"));
		    p->drawPixmap( point, other, *horizontalLine, 0, 0, i, 1 );
		    point += i;
		  }
	  }
    else 
    {
		  int end = dotlines[line+1].y();
		  int point = dotlines[line].y();
		  int other = dotlines[line].x();
		  int pixmapoffset = ((point & 1) != dotoffset ) ? 1 : 0;
		  while( point < end ) 
      {
		     int i = 128;
		     if ( i+point > end )
			  	 i = end-point;
		     p->drawPixmap( other, point, *verticalLine, 0, pixmapoffset, 1, i );
		     point += i;
		  }
	  }
	}
}

void FolderTreeItem::paintCell(QPainter *p, const QColorGroup &cg, int column, int width, int align)
{
	if(!p) return;
	
	QString t=text(column);
	
  QListView *lv=listView();
	
  int r=lv?lv->itemMargin():1;
  const QPixmap *icon = pixmap(column);

  p->fillRect(0, 0, width, height(), cg.brush(QColorGroup::Base));

  int marg=lv?lv->itemMargin():1;
	
  if (isSelected()) 
	{		
		p->fillRect(r-marg, 0, itemWidth(p->fontMetrics(), lv, column), height(), cg.brush(QColorGroup::Highlight));
	  p->setPen(cg.highlightedText());
  }
	else 
	{
		p->setPen(cg.text());
  }

	r+=LEFT_BEARING;
	if(icon)
	{
		p->drawPixmap(r, (height()-icon->height())/2, *icon);
		r+=icon->width()+listView()->itemMargin();
	}
	
	if(tag)
	{
		QFont f=p->font();
		f.setBold(true);
		p->setFont(f);
	}
	else
	{
		QFont f=p->font();
		f.setBold(false);
		p->setFont(f);
	}
	
	p->drawText(r, 0, width-marg-r, height(), align|AlignVCenter, t);
	
	int toffset=r+RIGHT_BEARING+p->fontMetrics().width(t)+TAG_SPACING;
		
	if(tag)
	{
		QFont f=p->font();
		f.setBold(false);
		p->setFont(f);
		
		p->setPen(TAG_COLOR);
		QString strtag="("+QString::number(tag)+")";
		p->drawText(toffset, 0, p->fontMetrics().width(strtag), height(), AlignLeft|AlignVCenter, strtag);
	}
}

int FolderTreeItem::width(const QFontMetrics &fm, const QListView *lv, int c) const
{
	int w=itemWidth(fm, lv, c);
	if(tag)
	{
		w+=TAG_SPACING+fm.width("("+QString::number(tag)+")");
	}
	
  return w;
}

int FolderTreeItem::itemWidth(const QFontMetrics &fm, const QListView *lv, int c) const
{
	int w=LEFT_BEARING+RIGHT_BEARING;
  const QPixmap *pm=pixmap(c);
	
	if(pm)
		w+=pm->width()+lv->itemMargin();
	if(tag)
	{
		QFont f=lv->font();
		f.setBold(true);
		QFontMetrics boldfm(f);
		w+=boldfm.width(text(c))+lv->itemMargin()*2;
	}
	else
	{
		w+=fm.width(text(c))+lv->itemMargin()*2;
	}
	
  return w;
}

void FolderTreeItem::paintFocus(QPainter *p, const QColorGroup &cg, const QRect &r)
{
	QListView *lv=listView();
	
	QRect tr(r.left(), r.top(), itemWidth(lv->fontMetrics(), lv, 0), r.height());
  listView()->style().drawFocusRect(p, tr, cg, isSelected()?&cg.highlight():&cg.base(), isSelected());
}

QString FolderTreeItem::getVFolderPath()
{
	return m_vfolderPath;
}

QString FolderTreeItem::getFilePath()
{ 
	return fsPath; 
}
void FolderTreeItem::setFilePath(QString path) 
{ 
	fsPath=path;
	acceptedTypes=QFileInfo(fsPath).isWritable()?"*":"";
}

bool FolderTreeItem::isTypeAccepted(QString type) 
{ 
	// debug
	printf("foldertreeitem: checking if type is accepted [%s in %s]...\n", (const char *)type, (const char *)acceptedTypes);
	
	return (type.isEmpty())||(acceptedTypes=="*")||(acceptedTypes.contains(type)); 
}
