#include "filter.h"

#include <qstringlist.h>
#include <qregexp.h>

#include <kprocess.h>

#include "indexclass.h"
#include "messagedescriptor.h"
#include "messagedevice.h"
#include "mailfolder.h"
#include "orb.h"
#include "clientvfs.h"


const char* mailFilterLogicNames[] =
  { "ignore", "and", "unless", "or", NULL };

const char* mailFilterFunctionNames[] =
  {  "contains", "contains-not", "equals", "not-equal", "regexp",
    "not-regexp", NULL };

const char* mailFilterActionNames[] =
  { "<nothing>", "transfer", "copy", "execute", NULL };

const char* mailFilterFieldNames[] =
  { "", "Subject", "To", "CC", "To or CC", "From", "Body", NULL };



// FilterRule class implementation

FilterRule::FilterRule(QString &field, QString &function, QString &context )
  : m_field(field), m_function(function), m_context(context)
{
}

FilterRule::~FilterRule()
{
}

bool FilterRule::matchList( QStringList itemsToCheck )
{
  for ( QStringList::Iterator it = itemsToCheck.begin(); it != itemsToCheck.end(); ++it ) {

    if ( m_function == "equals" )
      if ( *it == m_context ) return true;

    if ( m_function == "not_equals" )
      if ( *it != m_context ) return true;

    if ( m_function == "contains" )
      if ( (*it).find( m_context ) != -1 ) return true;

    if ( m_function == "contains-not" )
      if ( (*it).find( m_context ) == -1 ) return true;

    if ( m_function == "regexp" )
      if ( (*it).find( QRegExp( m_context, false ) ) != -1 ) return true;

    if ( m_function == "not-regexp" )
      if ( (*it).find( QRegExp( m_context, false ) ) == -1 ) return true;

  }

  return false;

}

bool FilterRule::match( MessageDevice *message  )
{
  MessageDescriptor mDescr = message->getDescriptor();

  QStringList itemsToCheck;
  if ( m_field == "Subject")
    itemsToCheck.append( mDescr.subject );
  else
  if ( m_field == "To")
    itemsToCheck += mDescr.to;
  else
  if ( m_field == "CC")
    itemsToCheck += mDescr.cc;
  else
  if ( m_field == "To or CC")
    itemsToCheck += mDescr.to + mDescr.cc;
  else
  if ( m_field == "Body")
    itemsToCheck.append( message->text() );

  return matchList( itemsToCheck );

}


bool FilterRule::match( MessageClass *message  )
{

  QStringList itemsToCheck;
  if ( m_field == "Subject")
    itemsToCheck.append( message->Subject );
  else
  if ( m_field == "From")
    itemsToCheck.append( message->From.toQCString() );
  else
  if ( m_field == "To" || m_field == "To or CC" )
    itemsToCheck += QStringList::fromStrList( message->To.toQStrList() );
  else
  if ( m_field == "CC" || m_field == "To or CC" )
    itemsToCheck += QStringList::fromStrList( message->Cc.toQStrList() );


  return matchList( itemsToCheck );

}


// FilterAction class implementation

FilterAction::FilterAction(QString &action, QString &param )
  : m_action(action), m_param(param)
{
}

FilterAction::~FilterAction()
{
}

QString FilterAction::messageFolder() {

  if ( m_action == "transfer" ) {
    return QString( "/Magellan/Mail/"+m_param );
  }

  return QString();
}


bool FilterAction::apply( IndexClass* idx, MessageDevice* dev, bool useForwardActions )
{
  qDebug("apply filter " + m_action + " , " + m_param );


  if ( m_action == "transfer" && useForwardActions ) {
    MailFolder *folder=ObjectRequestBroker::thisInstance()
          ->mailfolderReference( ClientVFS::thisInstance()->map("/Magellan/Mail/"+m_param) );

    if (folder)
      folder->moveMessage(idx);
    return true;
  }

  if ( m_action == "copy") {
    MailFolder *folder=ObjectRequestBroker::thisInstance()
          ->mailfolderReference( ClientVFS::thisInstance()->map("/Magellan/Mail/"+m_param) );

    if (folder)
      folder->copyMessage(idx);
    return true;
  }

  if ( m_action == "execute") {

    QString command = m_param;

    MessageDescriptor &descriptor=dev->getDescriptor();

    // replace %s with subject
    command.replace( QRegExp("%subj"), descriptor.subject );
    command.replace( QRegExp("%body"), dev->text() );


    KProcess proc;
    proc << "sh" << "-c" << command;
    proc.start( KProcess::DontCare, KProcess::NoCommunication );
  }


  return false;
}


// Filter class implementation


Filter::Filter( KConfig *config, QString group, int id )
{
  rule1 = rule2 = 0;
  for ( int i=0; i<ACTIONS_COUNT; i++ )
    actions[i] = 0;

  readConfig(config, group, id );

}



Filter::~Filter()
{
  if ( rule1 )
    delete rule1;
  if ( rule2 )
    delete rule2;

  for ( int i=0; i<ACTIONS_COUNT; i++ )
    if ( actions[i] )
      delete actions[i];
}



bool Filter::match( MessageDevice *message )
{
  if (!rule1) return false;

  bool cond1 = rule1->match(message);

  if ( ruleLogic == "ignore" )
    return cond1;

  if (!rule2) return false;

  bool cond2 = rule2->match(message);
  bool cond = false;

  if ( ruleLogic == "and" )
    cond = cond1 && cond2;
  else
  if ( ruleLogic == "or" )
    cond = cond1 || cond2;
  else
  if ( ruleLogic == "unless" )
    cond = cond1 && !cond2;

  return cond;
}


bool Filter::match( MessageClass *message )
{
  if (!rule1) return false;

  bool cond1 = rule1->match(message);

  if ( ruleLogic == "ignore" )
    return cond1;

  if (!rule2) return false;

  bool cond2 = rule2->match(message);
  bool cond = false;

  if ( ruleLogic == "and" )
    cond = cond1 && cond2;
  else
  if ( ruleLogic == "or" )
    cond = cond1 || cond2;
  else
  if ( ruleLogic == "unless" )
    cond = cond1 && !cond2;

  return cond;
}


void Filter::apply( IndexClass *message, MessageDevice* dev , bool useForwardActions )
{
  for ( int i=0; i<ACTIONS_COUNT; i++ )
    if ( actions[i] )
      actions[i]->apply(message,dev,useForwardActions);
}

QString Filter::messageFolder( MessageClass *message )
{
  if ( !match(message) )
    return QString();

  QString res;
  for ( int i=0; i<ACTIONS_COUNT; i++ )
    if ( actions[i] ) {
      res = actions[i]->messageFolder();
      if ( !res.isEmpty() )
        return res;
    }

  return QString();
}


void Filter::readConfig( KConfig *config, QString group, int id )
{
  if ( rule1 )
    delete rule1;
  if ( rule2 )
    delete rule2;

  for ( int i=0; i<ACTIONS_COUNT; i++ )
    if ( actions[i] )
      delete actions[i];


  config->setGroup( group );
  QString prefix = "filter_"+QString::number(id)+"_";

  QString rule1_field    = config->readEntry( prefix+"rule1_field", "" );
  QString rule1_function = config->readEntry( prefix+"rule1_function", "contains" );
  QString rule1_context  = config->readEntry( prefix+"rule1_context", "" );

  QString rule2_field    = config->readEntry( prefix+"rule2_field", "" );
  QString rule2_function = config->readEntry( prefix+"rule2_function", "contains" );
  QString rule2_context  = config->readEntry( prefix+"rule2_context", "" );

  rule1 = new FilterRule( rule1_field, rule1_function, rule1_context );
  rule1 = new FilterRule( rule1_field, rule1_function, rule1_context );

  ruleLogic = config->readEntry( prefix+"rule_logic", "ignore" );

  for ( int i=0; i<ACTIONS_COUNT; i++ ) {
    QString action = config->readEntry( prefix+"action"+QString::number(i), "<nothing>" );
    QString param = config->readEntry( prefix+"action"+QString::number(i)+"_param", "" );

    actions[i] = new FilterAction( action, param );
  }

}




