/*******************************************************************************
 *  PROJECT: PaletteParser
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#ifndef __PP_PALETTE_H
#define __PP_PALETTE_H

#include <iostream>
#include <vector>
#include <glibmm/ustring.h>
#include "core/gcs-types.h"
#include "core/gcs-color.h"
#include "paletteparser/error.h"
#include "trim.h"

class ParseSuite;


/** PaletteParser library namespace */
namespace pp
{

    /** The Basic class of the PaletteParser library */
   class Palette
   {
      public:
         typedef gcs::ColorPtr value_type;
         typedef std::vector<value_type> color_list_type;
         typedef std::vector<value_type>::iterator iterator;
         typedef std::vector<value_type>::const_iterator const_iterator;
         typedef std::vector<value_type>::size_type size_type;

      public:
         Palette(void);
         virtual ~Palette(void);

         bool parse(std::istream& in);

         /** Write out the palette to a stream.
          *
          * The stream can be written either to a string or to a file or to
          * standard output.  For example, if palette is a Palette object, you
          * could write this to a file as follows:
          * \code
          * std::ofstream file("directory/filename.gpl");
          * file << palette;
          * \endcode
          */
         friend std::ostream& operator<<(std::ostream& out, const Palette& p);

         /** Read a palette definition from a stream.
          *
          * The stream can be any of the standard C++ streams: standard input,
          * strings, or file streams.  This function is used as you would
          * otherwise use something like a parse() function.
          * For example, reading a palette definition from a file could be done
          * as follows:
          * \code
          * std::ifstream file("directory/filename.gpl");
          * file >> palette;
          * \endcode
          * After which you could query the properties of the palette, such as
          * its name, number of columns, list of colors, etc:
          * \code
          * std::cout << "Pallete name: " << palette.name() << std::endl;
          * for (Palette::iterator it = begin(); it != end(); it++)
          * {
          *    std::cout << *it << std::endl;
          * }
          */
         friend std::istream& operator>>(std::istream& in, Palette& p);

         /**
          * Get the name of the Palette
          */
         Glib::ustring name(void) const;

         /**
          * Set the name of the Palette
          */
         void set_name(Glib::ustring name);

         /** 
          * Get the number of columns recommended for displaying the palette
          */
         size_type columns(void) const;

         /** 
          * Set the number of columns recommended for displaying the palette
          */
         void set_columns(size_type cols);

         inline iterator begin(void) { return m_colors.begin(); }
         inline const_iterator begin(void) const { return m_colors.begin(); }
         inline iterator end(void) { return m_colors.end(); }
         inline const_iterator end(void) const { return m_colors.end(); }

         inline value_type at(size_t index) { return m_colors[index]; }
         inline const value_type& operator[](size_t index) { return m_colors[index]; }


         /**
          * Get a reference to the list of colors in the palette
          */
         color_list_type& colors(void);
         const color_list_type& colors(void) const;

         /**
          * Get the number of colors in the palette
          */
         size_type size(void) const;

         static value_type parse_color(const char* line) throw(ParseError);

      protected:
         inline Glib::ustring parse_name(const char* line) throw(ParseError);
         inline size_type parse_columns(const char* line) throw(ParseError);
         inline bool parse_header(const char* line);
         void reset(void);

      private:
         bool m_headerSeen;
         Glib::ustring m_name;
         size_type m_numColumns;
         color_list_type m_colors;
         unsigned int m_currentLine;

         friend class ::ParseSuite;
   };


   Glib::ustring Palette::parse_name(const char* line) throw(ParseError)
   {
       const char* nm = "Name:";
       // Make sure the line actually starts with "Name:"
       if (strstr(line, nm) == line)
       {
           return trim(Glib::ustring(line + strlen(nm), strlen(line) -
                       strlen(nm)));
       }
       else
       {
           throw ParseError(line);
       }
   }

   Palette::size_type Palette::parse_columns(const char* line) throw(ParseError)
   {
       const char* cols = "Columns:";
       // make sure that the line actually starts with "Columns:"
       if (strstr(line, cols) == line)
       {
           return static_cast<size_type>(atoi(
                       static_cast<const char*>((line + strlen(cols)))
                       )
                   );
       }
       else
       {
           throw ParseError(line);
       }
   }


   bool Palette::parse_header(const char* line)
   {
       const char* hdr = "GIMP Palette";
       if (strncmp(line, hdr, sizeof(hdr)) == 0)
           return true;
       else
           return false;
   }

} // namespace pp

#endif // __PP_PALETTE_H
