#ifndef _PLAYER_H
#define _PLAYER_H

#include <time.h>

#include "strings.h"
#include "individual.h"
#include "location.h"
#ifdef WIN32
#include "../win32/winconnection.h"
#else
#include "connection.h"
#endif
#include "inp_handler.h"
#include "flags.h"
#include "weapon.h"
#include "config.h"
#include "level_list.h"
#include "linkedlist.h"
#include "mailer.h"

class Race;

struct quest_list {
   Strings      the_quest;  
   quest_list   *next_quest;
};

struct level_list {
   Strings      chain;
   int          lvl_num;
   level_list   *next_level;
};

struct brief_stack {
   Strings      brief_name;
   Strings      the_brief;
   brief_stack  *next_brief;
};

struct prompt_stack {
   Strings      prompt_name;
   Strings      the_prompt;
   prompt_stack  *next_prompt;
};

class Player : public Individual
{
public:

   Player();  /* constructor */
   Player(char *the_name);  /* constructor */
   Player(Connection *new_sock);  /* constructor */
   virtual ~Player(); /* destructor */

   /***************************************
    ** Player struct management functions
    ***************************************/

   Player      *get_next_player();  /* returns a pointer to the next plr */
   void        set_next_player(Player *the_player); /* sets next plr to..*/

   int         is_on();
   void        set_off();

   void        set_quitting();
   int         is_quitting();

   int         find_host_name();
   char        *get_host_addr();
   char        *get_ip_addr();

   char        *set_initname(void);

   Strings     *get_misc_string( void );

   void        reset_player( void );

   /**************************************
    ** Socket management functions
    **************************************/

   int         check_connection();        /* checks the conn for data */

   virtual int send_plr(char *the_str, ...);  /* sends a string to player */

   void        flush_player();
   void        display_prompt(void);

   int         echo_on();
   int         echo_off();

   Connection  *get_connection();
   int         swap_connection(Player *swap_player);
   void        set_connection(Connection *new_connection);

   int         check_player();
   Inp_Handler *get_input_handler(void);
   int         repeat_last_command();


   /***********************************
    ** Set player attributes functions
    ***********************************/

   int         set_pname(char *the_name);

   bool        can_see_ind(Individual *the_ind);

   void        push_brief(char *brief_name, char *the_brief);
   int         remove_brief(char *brief_name);
   char        *get_brief();
   char        *get_brief(char *brief_name);

   void        push_prompt(char *prompt_name, char *the_prompt);
   int         remove_prompt(char *prompt_name);
   char        *get_prompt();
   char        *get_prompt(char *prompt_name);
   int         set_prompt(char *the_prompt);

   char        *get_passwd();
   int         set_passwd(char *the_passwd);

   void        set_race(char *the_race);
   char        *get_race();
   char        *get_chatlines();
   void        set_chatlines(char *the_string);

   Flags       *get_comflags();
   int         set_comflags(save_flags *the_flags);
   Flags       *get_admflags();
   int         set_admflags(save_flags *the_flags); 
   Flags       *get_gameflags();
   int         set_gameflags(save_flags *the_flags);
 
   int         get_weight_capacity(void);
   long        get_exp();
   void        set_exp(long the_num); 
   void        increase_exp(long the_num);
   int         get_pager_lines();
   void        set_pager_lines(int new_val);
   time_t      get_birthday();
   void        set_birthday(time_t the_num); 

   void        increase_rank_exp(int difficulty, Entity *the_object, 
                                                  int divide_by);

   /**********************************
    ** Game functions
    **********************************/

   int       game_put_obj(MudObject *the_obj, MudObject *to_obj, int has_to, 
                          int num_of, int is_in_loc);
   int       write_inventory(FILE *the_file);
   int       read_inventory(FILE *the_file);
   int       release_all_inv(void);

   void      add_initial_obj();

   void      display_abilities();
   void      display_weapons();

   void      set_talent_funds(int new_num);
   int       get_talent_funds();

   int       change_level(char *chain_name, int lvl_num);
   int       check_levels(char *the_chain);   
   int       write_levels(FILE *the_file);
   int       read_levels(FILE *the_file);
   int       show_levels(Player *the_player);
   int       get_cur_lvl(char *the_chain);

   time_t    get_idle();

   virtual void describe(Player *the_player);

   int       write_quests(FILE *the_file);
   int       read_quests(FILE *the_file);
   int       add_quest(char *the_name);
   int       remove_quest(char *the_name);
   int       find_quest(char *the_name);
   void      display_quests();

   void      set_pronouns(Inventory *the_inv);
   void      check_pronouns(char *the_input, Strings *the_output);
   void      show_pronouns(Player *the_player);
   void      create_mail();
   int       load_mail(load_type the_mode);
   Mailer    *get_mailer();

   int        load_talent_list();
   int        display_talents();
   Area_Dbase *get_talent_list();
   int        delete_talent_list();

   virtual int is_busy();
   virtual   int get_mem_size();
   virtual   int get_mem_size_dynamic();

   /**********************************
    ** Combat functions
    **********************************/

   virtual int attack(Individual *target);


private:

   MudObject    *read_carried(FILE *the_file, token_record **the_token);

   Connection   *the_socket;    /* the socket of this player */
   int          still_on;       /* if the player is still on or not */
   int          quitting;       /* if the player is quitting */

   Player       *next_player;   /* pointer to the next player */
   prompt_stack *prompt;         /* the player's prompt */
   Strings      passwd;         /* the player's encrypted password */   
   Strings      player_race;

   Flags        command_flags;  /* flags allowing commands to execute */
   Flags        admin_flags;    /* flags for game admin functions */
   Flags        game_flags;     /* flags that are used in gameplay */

   Inp_Handler  input_handler;

   level_list   *player_levels;

   quest_list   *player_quests;

   brief_stack *cur_brief;

   long        experience;

   time_t      birthday;

   /* pronouns for where the player is */
   Strings     him;
   Strings     her;
   Strings     it;

   Mailer      *the_mail;

   int         pager_lines;

   Strings     misc_string;
   Strings     chatlines;

   int         talent_funds;
   Area_Dbase  *talent_list;
};

#endif



