/**********************************************************************
 ** Key class:  handles all functions for a key object, which locks/unlocks
 **             a door object
 **
 ** Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef KEY_C
#define KEY_C

#include "config.h"
#include "sysdep.h"
#include "moveable.h"
#include "key.h"
#include "objtype.h"
#include "flags.h"
#include "player.h"

/***********************************************************************
 ** Key (constructor) - creates the item
 **
 ** Parameters: the_name - the name of the marker
 **             the_area - the area the marker belongs to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Key::Key(char *the_name, char *the_area)
{
   /* this should be the only place that this is set */
   obj_type = OBJ_TYPE_KEY;
   
   if (the_name != NULL)
      set_name(the_name);
   
   if (the_area != NULL)
      set_area(the_area);

   size = weight = state = 0;  
}

/***********************************************************************
 ** ~Key (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Key::~Key()
{
}


/***********************************************************************
 ** write_object - writes the key to a specified file in builder
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Key::write_object(FILE *the_file, int build_format)
{

   fprintf(the_file, "\nkey %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());
   write_mudobject_attrib(the_file);
   write_item_attrib(the_file);
   write_moveable_attrib(the_file);
}

/***********************************************************************
 ** describe - describes the key to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Key::describe(Builder *the_builder)
{
   the_builder->send_bldr("\n&+GKey: \t\t&+M%s&*\n", get_name());
   the_builder->send_bldr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_builder->send_bldr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_builder->send_bldr("&+GClones: \t&+g%s&*\n", get_clones());
   the_builder->send_bldr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_builder->send_bldr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_builder->send_bldr("&+GLocation: \t&+M%s&*\n", get_location());
   the_builder->send_bldr("&+GSize: \t\t&+w%d&*\n", get_size());
   the_builder->send_bldr("&+GWeight:\t\t&+w%d&*\n", get_weight());
   the_builder->send_bldr("&+GBrief0:\n&*%s\n", get_brief(0));
   the_builder->send_bldr("&+GBrief1:\n&*%s\n", get_brief(1));
   the_builder->send_bldr("&+GDesc:&*\n%s\n", get_desc());

   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the key to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Key::describe(Player *the_player)
{
   MudObject *tmp_container;
   Strings   container_name;

   the_player->send_plr("\n&+GKey: \t\t&+M%s&*\n", get_name());
   the_player->send_plr("&+GTitle: \t\t&+w%s&*\n", get_title());
   the_player->send_plr("&+GAltnames: \t&+g%s&*\n", get_altname());
   the_player->send_plr("&+GClones: \t&+g%s&*\n", get_clones());
   the_player->send_plr("&+GSpecials: \t&+g%s&*\n", get_special_str());
   the_player->send_plr("&+GGuards: \t&+g%s&*\n", get_guards());
   the_player->send_plr("&+GStartLoc: \t&+M%s&*\n", get_location());

   tmp_container = get_contained_by();
   if (tmp_container == NULL)
      container_name = "nowhere";
   else
      container_name.sprintf("%s@%s", tmp_container->get_name(), 
                                         tmp_container->get_area());

   the_player->send_plr("&+GCurrentLoc: \t&+M%s&*\n", 
                                                 container_name.str_show());

   the_player->send_plr("&+GSize: \t\t&+w%d&*\n", get_size());
   the_player->send_plr("&+GWeight:\t\t&+w%d&*\n", get_weight());
   the_player->send_plr("&+GBrief0:\n&*%s\n", get_brief(0));
   the_player->send_plr("&+GBrief1:\n&*%s\n", get_brief(1));
   the_player->send_plr("&+GDesc:&*\n%s\n", get_desc());

   list_specials(the_player);
   the_player->send_plr("&+YSize: \t\t\t&+W%d&*\n", get_mem_size());

   the_player->send_plr("\n");
}

/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Key::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr("You can set the following attributes on a key.\n"
                  "   title, weight, size, altnames, clones, desc,\n"
                  "   itemflags, brief0, brief1, location, specials, "
                  "   and guards\n");
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "title",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_title(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "weight",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_weight(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "size",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_size(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "altnames",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_altnames(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "guards",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_guard(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "specials", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_special(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "clones",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_clones(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "desc", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_desc(the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "itemflags",
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_itemflags(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), "brief0", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 0);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "brief1", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 1);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), "location", 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_loc(the_parsed, the_builder);
   }


   the_builder->send_bldr("The attribute '%s' is not a 'key' attribute.\n",
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** copy_object - copies the key to another object of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Key::copy_object(Entity *copy_obj)
{
   Key *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_KEY)
      return 0;

   copy_from = (Key *) copy_obj;

   /******* set the mudobject attributes *****/
   copy_mudobject_attrib((MudObject *) copy_from);

   /****** set the item attributes ******/
   copy_item_attrib((Item *) copy_from);

   /****** set the moveable attributes ******/
   copy_moveable_attrib((Moveable *) copy_from);

   return 1;
}

/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

Key *Key::operator = (Key *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Key::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Key::get_mem_size_dynamic()
{
   int  size = 0;
   
   size += get_mem_size_moveable();
   size += get_mem_size_item();
   size += get_mem_size_mudobj();
   size += get_mem_size_entity();

   return size;
}


#endif









