/**********************************************************************
 ** User_Dbase class: This class manages the user database. It is the file
 **                   that stores all user information for players who are
 **                   logging on to the game
 **
 **
 **
 ** Reviewed through: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef USER_DBASE_C
#define USER_DBASE_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "user_dbase.h"
#include "config.h"
#include "player.h"
#include "flags.h"
#include "gameflags.h"
#include "indflags.h"
#include "global.h"
#include "utils.h"
#include "newfuncts.h"
#include "memchk.h"

/***********************************************************************
 ** User_Dbase (constructor) - initializes the user database
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

User_Dbase::User_Dbase()
{
}


/***********************************************************************
 ** User_Dbase (destructor) - destroys the user database
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

User_Dbase::~User_Dbase()
{
}

/***********************************************************************
 ** write_user - writes a user to the appropriate file
 **
 ** Parameters: the_player - a pointer to the player struct to write
 **             death - has the player died, if so mark it as such
 **
 ** Returns: 0 for success
 **          -1 for failure
 **
 ***********************************************************************/
int User_Dbase::write_user(Player *the_player)
{
  return write_user(the_player, 0);
}

int User_Dbase::write_user(Player *the_player, int death)
{
   Flags       *tmp_comflags;
   Flags       *tmp_admflags;
   Flags       *tmp_gameflags;
   char        user_dir;
   Strings     username;
   Strings     filename;
   FILE        *the_file = NULL;

   if (the_player == NULL)
      return -1;

#ifdef DEBUG_UDBASE
   printf("storing user\n");
#endif

   if (!STRNCASECMP("login_", the_player->get_name(), 6))
     return -1;

   user_dir = (*(the_player->get_name()));
   user_dir = (char) tolower(user_dir);   

   username = the_player->get_name();
   username.lowercase();

   filename.sprintf("%s/%s/%c/%s%s", the_config.basedir.str_show(), 
                USERDATADIR, user_dir, username.str_show(), USEREXTENTION);

   /* if they have something, open the file for writing it */
    if ((the_file = xfopen(filename.str_show(), "w+", "user_file")) == NULL)
    {
        mainstruct->log_error("Could not open player data file", "write_user");
        return -1;
    }

   tmp_comflags = the_player->get_comflags();
   tmp_admflags = the_player->get_admflags();
   tmp_gameflags = the_player->get_gameflags();
 
   fprintf(the_file, "name %s\n", the_player->get_name());
   fprintf(the_file, "^%s^\n", the_player->get_passwd());

   tmp_comflags->write_flag(the_file);
   tmp_admflags->write_flag(the_file);
   tmp_gameflags->write_flag(the_file);

   if ((the_config.conf_flags->get_flag(CF_PERMDEATH) && (death)))
      fprintf(the_file, "xxx\n"); 
   if (the_player->get_strength() != the_config.default_strength)
      fprintf(the_file, "str %d\n", the_player->get_strength());
   if (the_player->get_dexterity() != the_config.default_dexterity)
      fprintf(the_file, "dex %d\n", the_player->get_dexterity());
   if (the_player->get_constitution() != the_config.default_constitution)
      fprintf(the_file, "con %d\n", the_player->get_constitution());
   if (the_player->get_intel() != the_config.default_intel)
      fprintf(the_file, "int %d\n", the_player->get_intel());
   if (the_player->get_wisdom() != the_config.default_wisdom)
      fprintf(the_file, "wis %d\n", the_player->get_wisdom());
   if (the_player->get_charisma() != the_config.default_charisma)
      fprintf(the_file, "chr %d\n", the_player->get_charisma());
   if (the_player->get_max_endurance() != the_config.default_maxendur)
      fprintf(the_file, "maxe %d\n", the_player->get_max_endurance());
   if (the_player->get_max_magic() != the_config.default_maxmagic)
      fprintf(the_file, "maxm %d\n", the_player->get_max_magic());
   if (the_player->get_exp() != 0)
      fprintf(the_file, "exp %ld\n", the_player->get_exp());
   if (the_player->get_pager_lines() != the_config.default_pager_lines)
      fprintf(the_file, "pag %d\n", the_player->get_pager_lines());
   fprintf(the_file, "sex %d\n", the_player->get_sex());
   fprintf(the_file, "vislvl %d\n", the_player->get_vislvl());

   fprintf(the_file, "desc ^%s^\n", the_player->get_desc());
   fprintf(the_file, "bri ^%s^\n", the_player->get_brief("main"));
   fprintf(the_file, "pro ^%s^\n", the_player->get_prompt());
   fprintf(the_file, "cha ^%s^\n", the_player->get_chatlines());
   fprintf(the_file, "bir %ld\n", the_player->get_birthday());
   fprintf(the_file, "rac %s\n", the_player->get_race());
   fprintf(the_file, "loc %s\n", the_player->get_location());

   the_player->write_inventory(the_file);
   the_player->write_ranks(the_file);
   the_player->write_levels(the_file);

   if (the_config.conf_flags->get_flag(CF_USEQUESTS))
      the_player->write_quests(the_file);

   xfclose(the_file, "user_file");
   return 1;
}


/***********************************************************************
 ** read_user - retrieves and loads a user from the player database
 **
 ** Parameters: the_player - a pointer to the player struct to store it in
 **
 ** Returns: returns a 1 for success
 **          0 if not found
 **          -1 for failure
 **
 ***********************************************************************/

int User_Dbase::read_user(char *the_name, Player *the_player){
   Strings       username;
   char          user_dir;
   Strings       filename;
   FILE          *the_file = NULL;
   Flags         *tmp_comflags;
   Flags         *tmp_admflags;
   Flags         *tmp_gameflags;
   Flags         *tmp_indflags;
   token_record  *the_token;
   Strings       holder;
   int           value;
   char          *temp_desc;

   user_dir = (*(the_name));
   user_dir = (char) tolower(user_dir);   

   username = the_name;
   username.lowercase();

   filename.sprintf("%s/%s/%c/%s%s", the_config.basedir.str_show(), 
                USERDATADIR, user_dir, username.str_show(), USEREXTENTION);

   /* if we can't open their user file, they carry nothing */
   if ((the_file = xfopen(filename.str_show(), "r", "user_file")) == NULL)
   {
      return 0;
   }

   tmp_comflags = the_player->get_comflags();
   tmp_admflags = the_player->get_admflags();
   tmp_gameflags = the_player->get_gameflags();
   tmp_indflags = the_player->get_indflags();

   the_token = get_token(the_file, '\0');
   if (STRCASECMP(the_token->the_string, "name"))
   {
      holder.sprintf("Could not load user %s data file.", the_name);
      mainstruct->log_error(holder.str_show(), "read_user");
      return 0;
   }

   the_token = get_token(the_file, '\0');
   if (STRCASECMP(the_token->the_string, the_name))
   {
      holder.sprintf("File name does not match user name for file %s.", 
                                                                  the_name);
      mainstruct->log_error(holder.str_show(), "read_user");
      return 0;
   }

   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format for user %s datafile.", the_name);
      mainstruct->log_error(holder.str_show(), "read_user");
      return 0;
   }
   
   the_token = get_token(the_file, '^');
   the_player->set_passwd(the_token->the_string);

   tmp_comflags->read_flags(the_file, mainstruct->get_log());
   tmp_admflags->read_flags(the_file, mainstruct->get_log());
   tmp_gameflags->read_flags(the_file, mainstruct->get_log());

   if (tmp_gameflags->get_flag(GAMEFLAG_NIGHTVISION))
      tmp_indflags->set_flag(INDFLAG_NIGHTVISION);
   else
      tmp_indflags->clr_flag(INDFLAG_NIGHTVISION);
   
   the_token = get_token(the_file, '\0');
   while (the_token->token_type > 0)
   {
      if (!STRCASECMP(the_token->the_string, "str"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for strength attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_strength(atoi(the_token->the_string));
      }

      // If they have died and it is permanent, report back
      else if ((the_config.conf_flags->get_flag(CF_PERMDEATH)) && 
	       (!STRCASECMP(the_token->the_string, "xxx")))
	return -2;

      else if (!STRCASECMP(the_token->the_string, "dex"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for dexterity attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_dexterity(atoi(the_token->the_string));
      }
      else if (!STRCASECMP(the_token->the_string, "con"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for constitution attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_constitution(atoi(the_token->the_string));
         the_player->set_health(the_player->get_maxhealth());
      }
      else if (!STRCASECMP(the_token->the_string, "int"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for intelligence attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_intel(atoi(the_token->the_string));
      }
      else if (!STRCASECMP(the_token->the_string, "wis"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for wisdom attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_wisdom(atoi(the_token->the_string));
      }
      else if (!STRCASECMP(the_token->the_string, "chr"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for charisma attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_charisma(atoi(the_token->the_string));
      }
      else if (!STRCASECMP(the_token->the_string, "maxe"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for MaxEndurance attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_max_endurance(value = atoi(the_token->the_string));
         the_player->set_endurance(value);
      }
      else if (!STRCASECMP(the_token->the_string, "maxm"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for MaxMagic attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_max_magic(value = atoi(the_token->the_string));
         the_player->set_magic(value);
      }
      else if (!STRCASECMP(the_token->the_string, "wim"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for Wimpy attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_wimpy(atoi(the_token->the_string));
      }
      else if (!STRCASECMP(the_token->the_string, "pag"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for Pager Lines attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_pager_lines(atoi(the_token->the_string));
      }
      else if (!STRCASECMP(the_token->the_string, "sex"))
      {
          the_token = get_token(the_file, '\0');
          if (the_token->token_type != T_NUMERICAL)
          {
              holder.sprintf("Missing number for sex attribute, %s datafile.", the_name);
              mainstruct->log_error(holder.str_show(), "read_user");
              return 0;
          }
          if (tmp_gameflags->get_flag(GAMEFLAG_MALE)) {
              the_player->set_sex(SEX_MALE);
              tmp_gameflags->clr_flag(GAMEFLAG_MALE);
          } 
          else
            the_player->set_sex(atoi(the_token->the_string));
      }
      else if (!STRCASECMP(the_token->the_string, "vislvl"))
      {
          the_token = get_token(the_file, '\0');
          if (the_token->token_type != T_NUMERICAL)
          {
              holder.sprintf("Missing number for vislvl attribute, %s datafile.", the_name);
              mainstruct->log_error(holder.str_show(), "read_user");
              return 0;
          }
          the_player->set_vislvl(atoi(the_token->the_string));
      }
      else if (!STRCASECMP(the_token->the_string, "bir"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for Birthday attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_birthday(atol(the_token->the_string));
      }

      else if (!STRCASECMP(the_token->the_string, "exp"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for Experience attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_player->set_exp(atoi(the_token->the_string));
      }
      else if (!STRCASECMP(the_token->the_string, "desc"))
      {

         temp_desc = read_desc_type(the_file, mainstruct->get_log(), 
                                                 (MudObject *) the_player);
   
         if (temp_desc == NULL)
            return -1;
         the_player->set_desc(temp_desc);
         delete temp_desc;
      }

      else if (!STRCASECMP(the_token->the_string, "bri"))
      {

         temp_desc = read_desc_type(the_file, mainstruct->get_log(), 
                                                 (MudObject *) the_player);

         if (temp_desc == NULL)
            return -1;
         if (the_player->get_brief("main") == NULL)
            the_player->push_brief("main", temp_desc);

         delete temp_desc;
      }

      else if (!STRCASECMP(the_token->the_string, "pro"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_CARROT)
	 {
            holder.sprintf("Unrecognized symbol '%s' in user %s datafile.", 
                                        the_token->the_string, the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         /* get the prompt */
         the_token = get_token(the_file, '^');
         the_player->set_prompt(the_token->the_string);
      }
      else if (!STRCASECMP(the_token->the_string, "cha"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_CARROT)
	 {
            holder.sprintf("Unrecognized symbol '%s' in user %s datafile.", 
                                        the_token->the_string, the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         /* get the chatlines */
         the_token = get_token(the_file, '^');
         the_player->set_chatlines(the_token->the_string);
      }

      else if (!STRCASECMP(the_token->the_string, "rac"))
      {
         the_token = get_token(the_file, '\0');
        
         the_player->set_race(the_token->the_string);
      }
      else if (!STRCASECMP(the_token->the_string, "loc"))
      {
         the_token = get_token(the_file, '\0');

         the_player->set_loc_str(the_token->the_string);
      }
      else if (!STRCASECMP(the_token->the_string, "inv"))
      {
         the_player->read_inventory(the_file);
      }
      else if ((!STRCASECMP(the_token->the_string, "abi")) ||
               (!STRCASECMP(the_token->the_string, "rank")))
      {
         the_player->read_ranks(the_file);
      }
      else if (!STRCASECMP(the_token->the_string, "lev"))
      {
         the_player->read_levels(the_file);
      }
      else if ((!STRCASECMP(the_token->the_string, "que")) &&
               (the_config.conf_flags->get_flag(CF_USEQUESTS)))
      {
         the_player->read_quests(the_file);
      }
      else
      {
         holder.sprintf("Unrecognized symbol '%s' in user %s datafile.", 
                                        the_token->the_string, the_name);
         mainstruct->log_error(holder.str_show(), "read_user");
      }
      the_token = get_token(the_file, '\0');
   }

   tmp_gameflags->clr_flag(GAMEFLAG_ONGAME);
   tmp_gameflags->clr_flag(GAMEFLAG_BUSY);

   xfclose(the_file, "user_file");

   return 1;

}


/***********************************************************************
 ** read_user - retrieves and loads a user from the player database
 **
 ** Parameters: the_builder - a pointer to the builder struct to store it in
 **
 ** Returns: returns a 1 for success
 **          0 if not found
 **          -1 for failure
 **
 ***********************************************************************/

int User_Dbase::read_user(char *the_name, Builder *the_builder)
{
   Strings       username;
   char          user_dir;
   Strings       filename;
   FILE          *the_file = NULL;
   Flags         *tmp_comflags;
   Flags         *tmp_admflags;
   Flags         *tmp_gameflags;
   token_record  *the_token;
   Strings       holder;

   user_dir = (*(the_name));
   user_dir = (char) tolower(user_dir);   

   username = the_name;
   username.lowercase();

   filename.sprintf("%s/%s/%c/%s%s", the_config.basedir.str_show(), 
                   USERDATADIR, user_dir, username.str_show(), USEREXTENTION);

   /* if we can't open their user file, they carry nothing */
   if ((the_file = xfopen(filename.str_show(), "r", "user_file")) == NULL)
   {
      return 0;
   }

   tmp_comflags = the_builder->get_comflags();
   tmp_admflags = the_builder->get_adminflags();
   tmp_gameflags = the_builder->get_gameflags();

   the_token = get_token(the_file, '\0');
   if (STRCASECMP(the_token->the_string, "name"))
   {
      holder.sprintf("Could not load user %s data file.", the_name);
      mainstruct->log_error(holder.str_show(), "read_user");
      return 0;
   }

   the_token = get_token(the_file, '\0');
   if (STRCASECMP(the_token->the_string, the_name))
   {
      holder.sprintf("File name does not match user name for file %s.", 
                                                                  the_name);
      mainstruct->log_error(holder.str_show(), "read_user");
      return 0;
   }

   the_token = get_token(the_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format for user %s datafile.", the_name);
      mainstruct->log_error(holder.str_show(), "read_user");
      return 0;
   }
   
   the_token = get_token(the_file, '^');
   the_builder->set_passwd(the_token->the_string);

   tmp_comflags->read_flags(the_file, mainstruct->get_log());
   tmp_admflags->read_flags(the_file, mainstruct->get_log());
   tmp_gameflags->read_flags(the_file, mainstruct->get_log());

   the_token = get_token(the_file, '\0');
   while (the_token->token_type > 0)
   {
      if (!STRCASECMP(the_token->the_string, "str"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "dex"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "con"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "int"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "maxe"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "xxx"))
      {
	// Do nothing
      }
      else if (!STRCASECMP(the_token->the_string, "maxm"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "lea"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "wim"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "rac"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "bir"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "loc"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "wis"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "chr"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "sex"))
      {
          the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "vislvl"))
      {
          the_token = get_token(the_file, '\0');
          if (the_token->token_type != T_NUMERICAL)
          {
              holder.sprintf("Missing number for vislvl attribute, %s datafile.", the_name);
              mainstruct->log_error(holder.str_show(), "read_user");
              return 0;
          }
          the_builder->set_vislvl(atoi(the_token->the_string));
      }
      else if (!STRCASECMP(the_token->the_string, "pag"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_NUMERICAL)
         {
            holder.sprintf("Missing number for Pager Lines attribute, "
                           "%s datafile.", the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_builder->set_pager_lines(atoi(the_token->the_string));
      }
      else if (!STRCASECMP(the_token->the_string, "exp"))
      {
         the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "desc"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_CARROT)
	 {
            holder.sprintf("Unrecognized symbol '%s' in user %s datafile.", 
                                        the_token->the_string, the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_token = get_token(the_file, '^');
 
      }
      else if (!STRCASECMP(the_token->the_string, "bri"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_CARROT)
	 {
            holder.sprintf("Unrecognized symbol '%s' in user %s datafile.", 
                                        the_token->the_string, the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         the_token = get_token(the_file, '^'); 
      }
      else if (!STRCASECMP(the_token->the_string, "pro"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_CARROT)
	 {
            holder.sprintf("Unrecognized symbol '%s' in user %s datafile.", 
                                        the_token->the_string, the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         /* get the specials */
         the_token = get_token(the_file, '^');
      }
      else if (!STRCASECMP(the_token->the_string, "cha"))
      {
         the_token = get_token(the_file, '\0');
         if (the_token->token_type != T_CARROT)
	 {
            holder.sprintf("Unrecognized symbol '%s' in user %s datafile.", 
                                        the_token->the_string, the_name);
            mainstruct->log_error(holder.str_show(), "read_user");
            return 0;
         }
         /* get the chatlines */
         the_token = get_token(the_file, '^');
      }

      else if (!STRCASECMP(the_token->the_string, "inv"))
      {
         while ((the_token->token_type > 0) &&
                (the_token->token_type != T_POUND))
            the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "que"))
      {
         while ((the_token->token_type > 0) &&
                (the_token->token_type != T_POUND))
            the_token = get_token(the_file, '\0');
      }
      else if ((!STRCASECMP(the_token->the_string, "abi")) ||
	       (!STRCASECMP(the_token->the_string, "rank")))

      {
         while ((the_token->token_type > 0) &&
                (the_token->token_type != T_POUND))
            the_token = get_token(the_file, '\0');
      }
      else if (!STRCASECMP(the_token->the_string, "lev"))
      {
         while ((the_token->token_type > 0) &&
                (the_token->token_type != T_POUND))
            the_token = get_token(the_file, '\0');
      }
      else
      {
         holder.sprintf("Unrecognized symbol '%s' in user %s datafile.", 
                                        the_token->the_string, the_name);
         mainstruct->log_error(holder.str_show(), "read_user");
      }
      the_token = get_token(the_file, '\0');
   }

   tmp_gameflags->clr_flag(GAMEFLAG_ONGAME);
   tmp_gameflags->clr_flag(GAMEFLAG_BUSY);

   xfclose(the_file, "user_file");

   return 1;

}

#endif
