// ---------------------------------------------------------------------------
// - Localset.cpp                                                            -
// - aleph engine - local set class implementation                           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Symbol.hpp"
#include "Localset.hpp"
#include "Exception.hpp"

namespace aleph {

  // this local set quark
  static const long QUARK_THIS = String::intern (".");

  // create a new local set

  Localset::Localset (void) {
    p_ptbl = new NameTable;
    p_stbl = nilp;
    Object::iref (p_ptbl);
  }

  // create a new local set but use only the primary table

  Localset::Localset (Localset* lset) {
    if (lset == nilp) {
      p_ptbl = new NameTable;
      p_stbl = nilp;
    } else {
      p_ptbl = lset->p_ptbl;
      p_stbl = new NameTable;
      symcst (QUARK_THIS, this);
    }
    Object::iref (p_ptbl);
    Object::iref (p_stbl);
  }

  // destroy this local set

  Localset::~Localset (void) {
    // protect us
    Object::iref (this);
    // destroy everything
    Object::dref (p_ptbl);
    Object::dref (p_stbl);
  }

  // return the class name

  String Localset::repr (void) const {
    return "Localset";
  }

  void Localset::mksho (void) {
    if (p_ptbl != nilp) p_ptbl->mksho ();
  }

  // clear the secondary name table / may be the first

  void Localset::clear (void) {
    // protect us before clear
    Object::iref (this);
    if (p_stbl != nilp) {
      p_stbl->clear ();
      Object::tref  (this);
      return;
    }
    if (p_ptbl != nilp) p_ptbl->clear ();
    Object::tref (this);
  }

  // bind a new object by quark

  void Localset::bind (const long quark, Object* object) {
    if (p_stbl != nilp)
      p_stbl->add (quark, object);
    else
      p_ptbl->add (quark, object);
  }

  // return true if the quark exists in the local set

  bool Localset::exists (const long quark) const {
    if (p_stbl != nilp) return p_stbl->exists (quark);
    return p_ptbl->exists (quark);
  }

  // find an object in this local set by quark

  Object* Localset::find (const long quark) {
    // normal lookup
    Object* obj = nilp;
    if (p_stbl != nilp) {
      obj = p_stbl->get (quark);
      if (obj == nilp) obj = p_ptbl->get (quark);
    } else {
      obj = p_ptbl->get (quark);
    }
    if (obj != nilp) return obj;
    if (p_parent != nilp) return p_parent->find (quark);
    return obj;
  }

  // remove an object by quark in this localset

  void Localset::remove (const long quark) {
    Object::iref (this);
    if ((p_stbl != nilp) && (p_stbl->exists (quark) == true)) {
      p_stbl->remove (quark);
      Object::tref (this);
      return;
    }
    p_ptbl->remove (quark);
    Object::tref (this);
  }

  // set an object as a const object by quark

  Object* Localset::cdef (Runnable* robj, Nameset* nset, const long quark,
			  Object* object) {
    // try first to find the object
    Object* obj = nilp;
    if (p_stbl != nilp) {
      obj = p_stbl->get (quark);
      if (obj == nilp) obj = p_ptbl->get (quark);
    } else {
      obj = p_ptbl->get (quark);
    }
    if (obj != nilp) {
      obj->cdef (robj, nset, object);
      return object;
    }

    // the object is not found - create a symbol and bind it
    Symbol* sym = new Symbol (quark, object);
    sym->setconst (true);
    if (p_stbl != nilp)
      p_stbl->add (quark, sym);
    else
      p_ptbl->add (quark, sym);
    return object;
  }

  // set an object to this object by quark

  Object* Localset::vdef (Runnable* robj, Nameset* nset, const long quark,
			   Object* object) {
    // try first to find the object
    Object* obj = nilp;
    if (p_stbl != nilp) {
      obj = p_stbl->get (quark);
      if (obj == nilp) obj = p_ptbl->get (quark);
    } else {
      obj = p_ptbl->get (quark);
    }
    if (obj != nilp) {
      obj->vdef (robj, nset, object);
      return object;
    }

    // the object is not found - create a symbol and bind it
    Symbol* sym = new Symbol (quark, object);
    if (p_stbl != nilp)
      p_stbl->add (quark, sym);
    else
      p_ptbl->add (quark, sym);
    return object;
  }

  // evaluate an object in the current nameset by quark

  Object* Localset::eval (Runnable* robj, Nameset* nset, const long quark) {
    // try first to find the object
    Object* obj = nilp;
    if (p_stbl != nilp) {
      obj = p_stbl->get (quark);
      if (obj == nilp) obj = p_ptbl->get (quark);
    } else {
      obj = p_ptbl->get (quark);
    }
    if (obj != nilp) return obj->eval (robj, nset);
    // try in the parent
    if (p_parent != nilp) return p_parent->eval (robj, nset, quark);
    // not found
    throw Exception ("eval-error", "unbound symbol", String::qmap (quark));
  }
}
