// ---------------------------------------------------------------------------
// - Graph.cpp                                                               -
// - standard object library - graph base class implementation               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2001 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Graph.hpp"
#include "Integer.hpp"
#include "Boolean.hpp"
#include "Runnable.hpp"
#include "Exception.hpp"

namespace aleph {
  // the edge, node and graph supported quarks
  static const long QUARK_ADD       = String::intern ("add");
  static const long QUARK_EXISTS    = String::intern ("exists");
  static const long QUARK_NEDGES    = String::intern ("number-of-edges");
  static const long QUARK_NNODES    = String::intern ("number-of-nodes");
  static const long QUARK_GETNODE   = String::intern ("get-node");
  static const long QUARK_GETEDGE   = String::intern ("get-edge");
  static const long QUARK_DEGREE    = String::intern ("degree");
  static const long QUARK_INDEGREE  = String::intern ("input-degree");
  static const long QUARK_OUTDEGREE = String::intern ("output-degree");
  static const long QUARK_ADDIN     = String::intern ("add-input-edge");
  static const long QUARK_GETIN     = String::intern ("get-input-edge");
  static const long QUARK_ADDOUT    = String::intern ("add-output-edge");
  static const long QUARK_GETOUT    = String::intern ("get-output-edge");
  static const long QUARK_GETSRC    = String::intern ("get-source");
  static const long QUARK_SETSRC    = String::intern ("set-source");
  static const long QUARK_GETTRG    = String::intern ("get-target");
  static const long QUARK_SETTRG    = String::intern ("set-target");
  static const long QUARK_GETCLO    = String::intern ("get-client");
  static const long QUARK_SETCLO    = String::intern ("set-client");

  // -------------------------------------------------------------------------
  // - graph edge section                                                    -
  // -------------------------------------------------------------------------

  // create a new empty edge

  Edge::Edge (void) {
    p_src = nilp;
    p_trg = nilp;
    p_clo = nilp;
  }

  // create an edge with a client object

  Edge::Edge (Object* clo) {
    p_src = nilp;
    p_trg = nilp;
    Object::iref (p_clo = clo);
  }

  // create an edge with two nodes

  Edge::Edge (Node* src, Node* trg) {
    p_clo = nilp;
    Object::iref (p_src = src);
    Object::iref (p_trg = trg);
    if (src != nilp) src->addout (this);
    if (trg != nilp) trg->addin  (this);
  }

  // destroy this edge

  Edge::~Edge (void) {
    Object::dref (p_src);
    Object::dref (p_trg);
    Object::dref (p_clo);
  }

  // return the edge class name

  String Edge::repr (void) const {
    return "Edge";
  }

  // make this edge a shared object

  void Edge::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_src != nilp) p_src->mksho ();
    if (p_trg != nilp) p_trg->mksho ();
    if (p_clo != nilp) p_clo->mksho ();
  }

  // set the source node of this edge

  void Edge::setsrc (Node* node) {
    wrlock ();
    if (p_src != node) {
      Object::dref (p_src);
      Object::iref (p_src = node);
    }
    unlock ();
  }

  // get the source node of this edge

  Node* Edge::getsrc (void) const {
    rdlock ();
    Node* node = p_src;
    unlock ();
    return node;
  }

  // set the target node of this edge

  void Edge::settrg (Node* node) {
    wrlock ();
    if (p_trg != node) {
      Object::dref (p_trg);
      Object::iref (p_trg = node);
    }
    unlock ();
  }

  // get the target node of this edge

  Node* Edge::gettrg (void) const {
    rdlock ();
    Node* node = p_trg;
    unlock ();
    return node;
  }

  // set the edge client object

  void Edge::setclo (Object* clo) {
    wrlock ();
    if (p_clo != clo) {
      Object::dref (p_clo);
      Object::iref (p_clo = clo);
    }
    unlock ();
  }

  // get the edge client object

  Object* Edge::getclo (void) const {
    rdlock ();
    Object* clo = p_clo;
    unlock ();
    return clo;
  }

  // reset the edge

  void Edge::reset (void) {
  }

  // create a new edge in a generic way

  Object* Edge::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc == 0) return new Edge;
    if (argc == 1) return new Edge (argv->get (0));
    if (argc == 2) {
      Node* src = dynamic_cast <Node*> (argv->get (0));
      Node* trg = dynamic_cast <Node*> (argv->get (1));
      if ((src == nilp) || (trg == nilp))
	throw Exception ("type-error", "invalid object to create edge");
      return new Edge (src, trg);
    }
    throw Exception ("argument-error", "too many arguments to create edge");
  }

  // apply a edge method with a set of arguments and a quark

  Object* Edge::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETSRC) {
	rdlock ();
	Object* result = getsrc ();
	robj->post (result);
	unlock ();
	return result;
      }
      if (quark == QUARK_GETTRG) {
	rdlock ();
	Object* result = gettrg ();
	robj->post (result);
	unlock ();
	return result;
      }
      if (quark == QUARK_GETCLO) {
	rdlock ();
	Object* result = getclo ();
	robj->post (result);
	unlock ();
	return result;
      }
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_SETSRC) {
	Node* node = dynamic_cast <Node*> (argv->get (0));
	if (node == nilp) throw Exception ("type-error", "invalid object");
	setsrc (node);
	robj->post (node);
	return node;
      }
      if (quark == QUARK_SETTRG) {
	Node* node = dynamic_cast <Node*> (argv->get (0));
	if (node == nilp) throw Exception ("type-error", "invalid object");
	settrg (node);
	robj->post (node);
	return node;
      }
      if (quark == QUARK_SETCLO) {
	Object* result = argv->get (0);
	setclo (result);
	robj->post (result);
	return result;
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }

  // -------------------------------------------------------------------------
  // - graph node section                                                    -
  // -------------------------------------------------------------------------

  // create a new empty node

  Node::Node (void) {
    Object::iref (p_in  = new Vector);
    Object::iref (p_out = new Vector);
    p_clo   = nilp;
  }

  // create a node with a client object

  Node::Node (Object* clo) {
    Object::iref (p_in  = new Vector);
    Object::iref (p_out = new Vector);
    Object::iref (p_clo = clo);
  }

  // destroy this node

  Node::~Node (void) {
    Object::dref (p_in);
    Object::dref (p_out);
    Object::dref (p_clo);
  }

  // return the node class name

  String Node::repr (void) const {
    return "Node";
  }

  // make this node a shared object

  void Node::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_in  != nilp) p_in->mksho ();
    if (p_out != nilp) p_out->mksho ();
    if (p_clo != nilp) p_clo->mksho ();
  }

  // add an edge to the incoming vector

  void Node::addin (Edge* edge) {
    wrlock ();
    if (edge != nilp) {
      p_in->append (edge);
      edge->settrg (this);
    }
    unlock ();
  }

  // add an edge to the ougoing vector

  void Node::addout (Edge* edge) {
    wrlock ();
    if (edge != nilp) {
      p_out->append (edge);
      edge->setsrc (this);
    }
    unlock ();
  }

  // get the number of incoming edges (i.e its in-degree)
  long Node::indegree (void) const {
    rdlock ();
    long result = p_in->length ();
    unlock ();
    return result;
  }

  // get the number of outgoing edges (i.e its out degree)
  long Node::outdegree (void) const {
    rdlock ();
    long result = p_out->length ();
    unlock ();
    return result;
  }

  // get the degree of this node
  long Node::degree (void) const {
    rdlock ();
    long result = p_in->length () + p_out->length ();
    unlock ();
    return result;
  }

  // get an incoming edge by index
  Edge* Node::getin (const long index) const {
    rdlock ();
    try {
      Edge* result = dynamic_cast <Edge*> (p_in->get (index));
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get an outgoing edge by index
  Edge* Node::getout (const long index) const {
    rdlock ();
    try {
      Edge* result = dynamic_cast <Edge*> (p_out->get (index));
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the node client object

  void Node::setclo (Object* clo) {
    wrlock ();
    if (p_clo != clo) {
      Object::dref (p_clo);
      Object::iref (p_clo = clo);
    }
    unlock ();
  }

  // get the node client object

  Object* Node::getclo (void) const {
    rdlock ();
    Object* clo = p_clo;
    unlock ();
    return clo;
  }

  // reset the node

  void Node::reset (void) {
  }

  // create a new node in a generic way

  Object* Node::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc == 0) return new Node;
    if (argc == 1) return new Node (argv->get (0));
    throw Exception ("argument-error", "too many arguments to create node");
  }

  // apply a node method with a set of arguments and a quark

  Object* Node::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_DEGREE)    return new Integer (degree ());
      if (quark == QUARK_INDEGREE)  return new Integer (indegree ());
      if (quark == QUARK_OUTDEGREE) return new Integer (outdegree ());
      if (quark == QUARK_GETCLO) {
	rdlock ();
	Object* result = getclo ();
	robj->post (result);
	unlock ();
	return result;
      }
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_ADDIN) {
	Edge* edge = dynamic_cast <Edge*> (argv->get (0));
	if (edge == nilp) throw Exception ("type-error", "invalid object");
	addin (edge);
	robj->post (edge);
	return edge;
      }
      if (quark == QUARK_ADDOUT) {
	Edge* edge = dynamic_cast <Edge*> (argv->get (0));
	if (edge == nilp) throw Exception ("type-error", "invalid object");
	addout (edge);
	robj->post (edge);
	return edge;
      }
      if (quark == QUARK_SETCLO) {
	Object* result = argv->get (0);
	setclo (result);
	robj->post (result);
	return result;
      }
      if (quark == QUARK_GETIN) {
	long index = argv->getint (0);
	rdlock ();
	try {
	  Edge* edge = getin (index);
	  robj->post (edge);
	  unlock ();
	  return edge;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_GETOUT) {
	long index = argv->getint (0);
	rdlock ();
	try {
	  Edge* edge = getout (index);
	  robj->post (edge);
	  unlock ();
	  return edge;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }

  // -------------------------------------------------------------------------
  // - graph section                                                         -
  // -------------------------------------------------------------------------

  // create an empty graph

  Graph::Graph (void) {
    Object::iref (p_nodes = new Vector);
    Object::iref (p_edges = new Vector);
  }

  // delete this graph

  Graph::~Graph (void) {
    Object::dref (p_edges);
    Object::dref (p_nodes);
  }

  // return the graph class name

  String Graph::repr (void) const {
    return "Graph";
  }

  // make this graph a shared object

  void Graph::mksho (void) {
    if (p_shared != nilp) return;
    Object::mksho ();
    if (p_nodes != nilp) p_nodes->mksho ();
    if (p_edges != nilp) p_edges->mksho ();
  }

  // return true if the node exists in this graph
  
  bool Graph::exists (Node* node) const {
    rdlock ();
    bool result = p_nodes->exists (node);
    unlock ();
    return result;
  }

  // return true if the edge exists in this graph
  
  bool Graph::exists (Edge* edge) const {
    rdlock ();
    bool result = p_edges->exists (edge);
    unlock ();
    return result;
  }

  // add a node to this graph

  void Graph::add (Node* node) {
    wrlock ();
    if ((node == nilp) || (p_nodes->exists (node) == true)) {
      unlock ();
      return;
    }
    if (node->degree () != 0) {
      unlock ();
      throw Exception ("graph-error", "null degree violation");
    }
    p_nodes->append (node);
    unlock ();
  }

  // add an edge to this graph

  void Graph::add (Edge* edge) {
    wrlock ();
    if ((edge == nilp) || (p_edges->exists (edge) == true)) {
      unlock ();
      return;
    }
    Node* src = edge->getsrc ();
    if (exists (src) == false) p_nodes->append (src);
    Node* trg = edge->gettrg ();
    if (exists (trg) == false) p_nodes->append (trg);
    p_edges->append (edge);
    unlock ();
  }

  // return the number of nodes

  long Graph::getnnodes (void) const {
    rdlock ();
    long result = p_nodes->length ();
    unlock ();
    return result;
  }

  // return the number of edges

  long Graph::getnedges (void) const {
    rdlock ();
    long result = p_edges->length ();
    unlock ();
    return result;
  }

  // get a node by index

  Node* Graph::getnode (const long index) const {
    rdlock ();
    try {
      Node* node = dynamic_cast <Node*> (p_nodes->get (index));
      unlock ();
      return node;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get an edge by index

  Edge* Graph::getedge (const long index) const {
    rdlock ();
    try {
      Edge* edge = dynamic_cast <Edge*> (p_edges->get (index));
      unlock ();
      return edge;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // reset all nodes in this graph

  void Graph::resetnodes (void) {
    wrlock ();
    try {
      long nnodes = getnnodes ();
      for (long i = 0; i < nnodes; i++) {
	Node* node = dynamic_cast <Node*> (p_nodes->get (i));
	node->reset ();
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // reset all edges in this graph

  void Graph::resetedges (void) {
    wrlock ();
    try {
      long nedges = getnedges ();
      for (long i = 0; i < nedges; i++) {
	Edge* edge = dynamic_cast <Edge*> (p_edges->get (i));
	edge->reset ();
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // reset both nodes and edges in the graph

  void Graph::reset (void) {
    resetnodes ();
    resetedges ();
  }

  // create a new graph in a generic way

  Object* Graph::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    if (argc == 0) return new Graph;
    throw Exception ("argument-error", "too many arguments to create graph");
  }

  // apply a graph method with a set of arguments and a quark

  Object* Graph::apply (Runnable* robj, Nameset* nset, const long quark,
			Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_NEDGES) return new Integer (getnedges ());
      if (quark == QUARK_NNODES) return new Integer (getnnodes());
    }

    // dispatch 1 argument
    if (argc == 1) {
      if (quark == QUARK_ADD) {
	Edge* edge = dynamic_cast <Edge*> (argv->get (0));
	if (edge != nilp) {
	  add (edge);
	  robj->post (edge);
	  return edge;
	}
	Node* node = dynamic_cast <Node*> (argv->get (0));
	if (node != nilp) {
	  add (node);
	  robj->post (node);
	  return node;
	}
	throw Exception ("type-error", "invalid object to add to graph");
      }
      if (quark == QUARK_EXISTS) {
	Edge* edge = dynamic_cast <Edge*> (argv->get (0));
	if (edge != nilp) return new Boolean (exists (edge));
	Node* node = dynamic_cast <Node*> (argv->get (0));
	if (node != nilp) return new Boolean (exists (node));
	throw Exception ("type-error", "invalid object to check in graph");
      }
      if (quark == QUARK_GETEDGE) {
	long index = argv->getint (0);
	rdlock ();
	try {
	  Edge* edge = getedge (index);
	  robj->post (edge);
	  unlock ();
	  return edge;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_GETNODE) {
	long index = argv->getint (0);
	rdlock ();
	try {
	  Node* node = getnode (index);
	  robj->post (node);
	  unlock ();
	  return node;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
    }

    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
