/* ups_fetch.c - Calls the routines in upsfetch.c for fetching data from UPS

   Copyright (C) 2000  Henning Kulander <hennikul@ifi.uio.no>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>
#include <errno.h>

#include "support.h"
#include "ups_canvas.h"
#include "upsfetch.h"
#include "ups_fetch.h"
#include "dial.h"
#include "ups_dials.h"
#include "preferences.h"

#define VALUE_LEN 100
#define NOT_CONNECTED _("Not connected")

char ups_vars[512];
char *upsname, *upshost;
char value[VALUE_LEN];
gboolean has_lowxfer = FALSE;
gboolean has_highxfer = FALSE;
gboolean connected = FALSE;
gboolean prev_error = FALSE;
int fd=0;

gdouble dial_values[NDIALS];
gdouble batt_pct;
gboolean low, overload, online;
gchar *model;

/* Fetch a variable from the UPS using either UDP or TCP */
char *
(*get_ups_var) (char *variable);

/* Finds the variables (infotypes) supported by this UPS */
void 
find_ups_variables (void)
{
	gchar *v;
	gchar *description;
	
	info_types_clear ();
	has_lowxfer = FALSE;
	has_highxfer = FALSE;
	
        v = strtok (ups_vars, " ");
	while (v != NULL) {
		description = ups_get_description(v);
		if (description != NULL)
			info_types_add(v, description);
                if (!strcmp(v, "LOWXFER"))
			has_lowxfer = TRUE;
                if (!strcmp(v, "HIGHXFER"))
			has_highxfer = TRUE;
		
                v = strtok (NULL, " ");
	}
}

/* Connect to the UPS using UDP transfer mode */
gboolean 
connect_ups_udp (char *host)
{
	gchar *uerror, *error;
	
	connected = FALSE;


	/* Get list of avaliable variables from ups */
	if (getupsvarlist (host, ups_vars, sizeof (ups_vars)) < 0) {
		if (prev_error == FALSE) {
			uerror = upsstrerror (__FILE__, __LINE__, upserror);
			error = 
                                g_strdup_printf(_("Error getting variables"\
                                                  " from UPS: %s\n"), 
						uerror);
			 gnome_warning_dialog (error);
			 g_free (uerror);
			 g_free (error);
			 prev_error = TRUE;
		}
		gnome_appbar_set_default
			(GNOME_APPBAR (lookup_widget (global_window_get (),
                                                      "appbar")), 
			 NOT_CONNECTED);
		gnome_appbar_refresh
			(GNOME_APPBAR (lookup_widget (global_window_get (),
                                                      "appbar")));
		connected = FALSE;
		return FALSE;
	} else {
		prev_error = FALSE;
	}

	upshost = host;
	connect_ups = connect_ups_udp;
	get_ups_var = get_ups_var_udp;
	
	find_ups_variables ();
	
	connected = TRUE;
	
	return TRUE;
}

/* Connect to the UPS using TCP transfer mode */
gboolean 
connect_ups_tcp (char *host)
{
	char *ptr;
	gchar *uerror, *error;
	
	upsname = g_strdup (host);
	
	connected = FALSE;
	if (fd)
		closeupsfd (fd);

	/* Split hostname into real hostname and upsname */
        ptr = strstr (upsname, "@");
	if (ptr != NULL) {
		*ptr = 0;
		upshost = ptr + 1;
	} else	{
		upsname = NULL;
		upshost = host;
	}

	/* Make connection to host */
	fd = upsconnect (upshost);
	if (fd < 0) {
		if ( prev_error == FALSE ) {
			uerror = upsstrerror (__FILE__, __LINE__, upserror);
                        error = g_strdup_printf (_("Error connecting to "\
                                                   "ups: %s\n"), 
						 uerror);
			gnome_warning_dialog (error);
			g_free (uerror);
			g_free (error);
			prev_error = TRUE;
		}
		gnome_appbar_set_default
			(GNOME_APPBAR (lookup_widget (global_window_get (),
                                                      "appbar")), 
			 NOT_CONNECTED);
		gnome_appbar_refresh
			(GNOME_APPBAR (lookup_widget (global_window_get (),
                                                      "appbar")));
		connected = FALSE;
		return FALSE;
	 } else {
		 prev_error = FALSE;
	 }

	/* Get list ov avaliable variables from ups */
	if (getupsvarlistfd (fd, upsname, ups_vars, sizeof (ups_vars)) < 0) {
		uerror = upsstrerror (__FILE__, __LINE__, upserror);
                error = g_strdup_printf (_("Error getting variables from "\
                                           "UPS: %s\n"), 
					 uerror);
		gnome_warning_dialog (error);
		g_free (uerror);
		g_free (error);
		connected = FALSE;
		return FALSE;
	 }

	connect_ups = connect_ups_tcp;
	get_ups_var = get_ups_var_tcp;
	
	find_ups_variables ();
	
	connected = TRUE;
	
	
        printf ("Connected to UPS %s@%s\n", upsname, upshost);
	
	return TRUE;
}

/* Get a variable from the UPS using UDP transfer mode */
char *
get_ups_var_udp (char *variable)
{
	gchar *uerror;
	
	if (getupsvar (upshost, variable, value, sizeof (value)) < 0 ) {
		uerror = upsstrerror (__FILE__, __LINE__, upserror);
                g_warning (_("Error reading from ups: %s\n"), uerror);
		g_free (uerror);
		if ((upserror == UPSF_RECVTIMEOUT) || 
		    (upserror == UPSF_RECVFAILURE)) { /* try TCP mode */
			preferences_udp_set (FALSE);
			
                        g_warning(_("Trying TCP connection instead...\n"));
			
			connect_ups_tcp (upshost);
			
			start_fetching ();
		}
		return NULL ;
	} else {
		return value;
	}
}

/* Get a variable from the UPS using TCP transfer mode */
char *
get_ups_var_tcp (char *variable)
{
	GtkWidget *widget; 
	gchar *uerror;
	
	if (getupsvarfd (fd, upsname, variable, value, sizeof (value)) < 0 ) {
		uerror = upsstrerror (__FILE__, __LINE__, upserror);
                g_warning (_("Error reading from ups: %s\n"), uerror);
		g_free (uerror);
                widget = lookup_widget (global_window_get (), "appbar");
		gnome_appbar_set_default( GNOME_APPBAR (widget), 
					  NOT_CONNECTED);
		if (prev_error == FALSE)
			gnome_app_error( GNOME_APP (global_window_get ()), 
                                         _("Network error: Disconnected "\
                                           "from UPS..."));
		prev_error = TRUE;
		return NULL;
	} else {
		prev_error = FALSE;
		return value;
	}
}

gfloat 
ups_get_lowxfer (void)
{
	gfloat fvalue;
	gchar *value;
	char **endptr=0;

	if (has_lowxfer) {
                if ((value = get_ups_var ("LOWXFER"))) {
			fvalue = (gfloat)(strtod (value, endptr));
			if ( endptr )
                                g_warning (_("Error converting float "\
                                             "\"%s\": %s\n"),
					  value,strerror (errno));
			else
				return fvalue;
		}else {
			return 00.0;
		}  
	}
	return 00.0;
}

gfloat 
ups_get_highxfer (void)
{
	gfloat fvalue;
	gchar *value;
	char **endptr=0;

	if (has_highxfer) {
                if ((value = get_ups_var ("HIGHXFER"))) {
			fvalue = (gfloat)(strtod (value, endptr));
			if (endptr)
                                g_warning (_("Error converting float "\
                                             "\"%s\": %s\n"),
					  value,strerror (errno));
			else
				return fvalue;
		} else {
			return(250.0);
		}  
	}
	return 250.0;
}

extern int statlen;
int fetch_data(char *host);
    

/* Get information from UPS and update the status widgets in the 
 * window */
gboolean 
fetch_ups_data (void)
{
	char *value;
	gdouble fvalue;
	char **endptr=NULL;
	InfoDial *info_dial;
	gint i;


	statlen = 0;
	fetch_data(upshost);

	if (connected == FALSE)
		return FALSE;

	lock_ups_mutex ();

	/* Get Model name */
        if ((value = get_ups_var ("MODEL")))     {
		if (model)
			g_free (model);
		model = g_strdup (value);
	} else {
		unlock_ups_mutex ();
		return FALSE;
	}

	/* Get Status*/
        if ((value = get_ups_var ("STATUS"))) {
                if ( !g_strncasecmp (value, "ONLINE", 6)) { /*Online*/
			low = FALSE;
			overload = FALSE;
			online = TRUE;
                } else if (!g_strncasecmp (value, "ONBATT", 6)) {/*On Battery*/
			low = FALSE;
			overload = FALSE;
			online = FALSE;
                } else if (!g_strncasecmp (value, "LOWBATT", 7)) {/*Low Battery*/
			low = TRUE;
			overload = FALSE;
			online = FALSE;
                } else if (!g_strncasecmp (value, "OVERLOAD", 8)) {/*Overload*/
			low = FALSE;
			overload = TRUE;
			online = FALSE;
                } else if (!g_strncasecmp (value, "REPLACEBATT", 11)) {/*Replace Battery*/
			low = TRUE;
			overload = TRUE;
			online = TRUE;
		}
	} else {
		unlock_ups_mutex ();
		return FALSE;		
	}

	/* Get Battery Percent */
        if ( (value = get_ups_var ("BATTPCT"))) {
		fvalue = (gdouble)(strtod (value, endptr));
		if (endptr)
                        g_warning(_("Error converting float \"%s\": %s\n"),
				  value,strerror (errno));
		else
			batt_pct = fvalue;
	} else {
		unlock_ups_mutex ();
		return FALSE;
	 }
  
	/* Get values for all the dials */
	for (i = 0 ; i < NDIALS ; i++) {
		info_dial = info_dial_get (i);
		
		if (info_dial->dial != NULL) { 
			if ((value = get_ups_var (info_dial->infotype))) {
				fvalue = (gfloat)(strtod (value, endptr));
				if (endptr)
                                        g_warning (_("Error converting "\
                                                     "float \"%s\": %s\n"),
						   value,strerror (errno));
				else
					dial_values[i] = fvalue;
			} else {
				unlock_ups_mutex ();
				return FALSE;
			}
		}
	}
	
	unlock_ups_mutex ();
	
	return TRUE; 
}

/* Get information from UPS and update the status widgets in the window
 */
gboolean 
display_ups_data (GtkWidget *window)
{
	GtkWidget *widget;
	gchar *string;
	InfoDial *info_dial;
	gint i;
	
	if (connected == FALSE)
		return FALSE;

	/* Display Model name */
	if (model) {
                widget = lookup_widget (window, "appbar");
                string = g_strdup_printf ("Model: %s", model);
		gnome_appbar_set_default (GNOME_APPBAR (widget), string);
		g_free (string);
	} else {
		return FALSE;
	}

	/* Display Status */
	ups_battery_online_set (online);
	ups_battery_low_set (low);
	ups_battery_overload_set (overload);

	/* Display Battery Percent */
	ups_battery_meter_set (batt_pct);
	
	/* Display values for all the dials */
	for (i = 0 ; i < NDIALS ; i++) {
		info_dial = info_dial_get (i);
		
		if (info_dial->dial != NULL) {
			dial_value_set (info_dial->dial, dial_values[i]);
		}
	}

	return TRUE;
}
