// aptcache.cc
//
//  Copyright 1999,2000,2001,2002 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.

#include "../aptitude.h"

#include "aptcache.h"
#include "aptitudepolicy.h"
#include "undo.h"
#include "config_signal.h"
#include "apt.h"
#include "matchers.h"

#include <apt-pkg/error.h>
#include <apt-pkg/sourcelist.h>
#include <apt-pkg/pkgcachegen.h>
#include <apt-pkg/configuration.h>
#include <apt-pkg/tagfile.h>
#include <apt-pkg/fileutl.h>
#include <apt-pkg/algorithms.h>
#ifdef HAVE_LIBAPT_PKG3
#include <apt-pkg/pkgsystem.h>
#include <apt-pkg/policy.h>
#include <apt-pkg/version.h>
#endif

#include <vector>

#include <unistd.h>
#include <stdio.h>
#include <errno.h>

using namespace std;

class aptitudeDepCache::apt_undoer:public undoable
// Allows an action performed on the package cache to be undone.  My first
// thought was to just snapshot the package cache before the action and then
// copy over the state, but this works better (IMO :) ) -- the other method
// had the drawback of not telling what was actually happening, and potentially
// leaving bits of the cache in weird states.
//
// Of course, there's always the danger that this won't properly restore the
// cache state, so I'll have to revert to the original method..
{
  PkgIterator pkg;
  int prev_mode;  // One of Delete,Keep,Install
  int prev_flags; /* For Delete, tracks Purge mode; for Install, tracks 
		   * Reinstall mode..
		   */
  changed_reason prev_installremovereason;
  pkgCache::State::PkgSelectedState prev_selection_state;

  aptitudeDepCache *owner;
public:
  apt_undoer(PkgIterator _pkg, int _prev_mode, int _prev_flags,
	     changed_reason _prev_installremovereason,
	     pkgCache::State::PkgSelectedState _prev_selection_state,
	     aptitudeDepCache *_owner)
    :pkg(_pkg), prev_mode(_prev_mode), prev_flags(_prev_flags),
     prev_installremovereason(_prev_installremovereason),
     prev_selection_state(_prev_selection_state), owner(_owner)
  {
  }

  void undo()
  {
    if(prev_flags&ReInstall)
      owner->internal_mark_install(pkg, false, true, NULL, false);
    else switch(prev_mode)
      {
      case ModeDelete:
	owner->internal_mark_delete(pkg, prev_flags&Purge, prev_installremovereason==unused, NULL, false);
	break;
      case ModeKeep:
	owner->internal_mark_keep(pkg, prev_flags&AutoKept, prev_selection_state==pkgCache::State::Hold, NULL, false);
	break;
      case ModeInstall:
	owner->internal_mark_install(pkg, false, false, NULL, false);
	break;
      }

    owner->get_ext_state(pkg).installremove_reason=prev_installremovereason;
  }
};

extern SigC::Signal0<void> cache_reloaded;

class aptitudeDepCache::forget_undoer:public undoable
// Undoes a "forget_new" command
{
  vector<pkgCache::PkgIterator> packages;

  aptitudeDepCache *owner;
public:
  forget_undoer(aptitudeDepCache *_owner):owner(_owner) {}

  void add_item(pkgCache::PkgIterator item)
  {
    packages.push_back(item);
  }

  bool empty()
  {
    return packages.empty();
  }

  void undo()
  {
    for(vector<pkgCache::PkgIterator>::iterator i=packages.begin(); i!=packages.end(); i++)
      owner->get_ext_state(*i).new_package=true;

    // Hack to make all the trees rebuild themselves.
    cache_reloaded();
  }
};

#ifdef HAVE_LIBAPT_PKG3
class aptitudeDepCache::candver_undoer:public undoable
// Undoes a "set candidate version" command.
{
  pkgCache::VerIterator oldver;

  aptitudeDepCache *owner;
public:
  candver_undoer(pkgCache::VerIterator _oldver,
		 aptitudeDepCache *_owner)
    :oldver(_oldver), owner(_owner)
  {
  }

  void undo()
  {
    owner->set_candidate_version(oldver, NULL);
  }
};
#endif

#ifndef HAVE_LIBAPT_PKG3
aptitudeDepCache::aptitudeDepCache(MMap &Map, OpProgress &Prog,
				   bool do_initselections,
				   bool WithLock,
				   const char *status_fname)
  :pkgDepCache(Map, Prog), dirty(false), package_states(NULL), lock(-1),
   group_level(0), mark_and_sweep_in_progress(false)
{
  if(!_error->PendingError())
    build_selection_list(Prog, WithLock, do_initselections, status_fname);
}
#else
aptitudeDepCache::aptitudeDepCache(pkgCache *Cache, Policy *Plcy)
  :pkgDepCache(Cache, Plcy), dirty(false), package_states(NULL), lock(-1),
   group_level(0), mark_and_sweep_in_progress(false)
{
}

bool aptitudeDepCache::Init(OpProgress *Prog, bool WithLock, bool do_initselections, const char *status_fname)
{
  return build_selection_list(*Prog, WithLock, do_initselections, status_fname);
}
#endif

aptitudeDepCache::~aptitudeDepCache()
{
  delete[] package_states;
  if(lock!=-1)
    close(lock);
}

bool aptitudeDepCache::build_selection_list(OpProgress &Prog, bool WithLock,
					    bool do_initselections,
					    const char *status_fname)
{
  bool initial_open=false;
  // This will be set to true if the state file does not exist.

  if(!pkgDepCache::Init(&Prog))
    return false;

  string statedir=aptcfg->FindDir("Dir::Aptitude::state", STATEDIR);
  // Should this not go under Dir:: ?  I'm not sure..
  delete package_states;
  package_states=new aptitude_state[Head().PackageCount];
  for(unsigned int i=0; i<Head().PackageCount; i++)
    {
      package_states[i].new_package=true;
      package_states[i].selection_state=package_states[i].dselect_state=pkgCache::State::Unknown;
      package_states[i].reinstall=false;
      package_states[i].installremove_reason=manual;
    }

  if(WithLock && lock==-1)
    {
      lock=GetLock(statedir+"lock");

      if(_error->PendingError())
	{
	  if(lock!=-1)
	    close(lock);
	  lock=-1;
	  return false;
	}
    }
  // Errrrr, I think we need to do this first in case the stuff below
  // manages to trigger a mark operation.
  duplicate_cache(&backup_state);

  FileFd state_file;

  // Read in the states that we saved
  if(status_fname==NULL)
    state_file.Open(statedir+"pkgstates", FileFd::ReadOnly);
  else
    state_file.Open(status_fname, FileFd::ReadOnly);

  if(!state_file.IsOpen())
    {
      _error->Discard();
      if(errno!=ENOENT)
	_error->Warning(_("Can't open Aptitude extended state file"));
      else
	initial_open=true;
    }
  else
    {
      int file_size=state_file.Size();
      Prog.OverallProgress(0, file_size, 1, _("Reading extended state information"));

#ifndef HAVE_LIBAPT_PKG3
      pkgTagFile tagfile(state_file);
#else
      pkgTagFile tagfile(&state_file);
#endif
      pkgTagSection section;
      int amt=0;
      bool do_dselect=aptcfg->FindB(PACKAGE "::Track-Dselect-State", true);
      while(tagfile.Step(section))
	{
	  PkgIterator pkg=FindPkg(section.FindS("Package"));
	  if(!pkg.end() && !pkg.VersionList().end())
	    // Silently ignore unknown packages and packages with no actual
	    // version.
	    {
	      unsigned long tmp=0;
	      string candver;

	      aptitude_state &pkg_state=get_ext_state(pkg);

	      section.FindFlag("Unseen", tmp, 1);
	      pkg_state.new_package=(tmp==1);

	      tmp=0;
	      section.FindFlag("ReInstall", tmp, 1);
	      pkg_state.reinstall=(tmp==1);

	      pkg_state.installremove_reason=(changed_reason) section.FindI("Last-Change", manual);

	      candver=section.FindS("Version");

	      pkg_state.selection_state=(pkgCache::State::PkgSelectedState) section.FindI("State", pkgCache::State::Unknown);
	      pkg_state.dselect_state=(pkgCache::State::PkgSelectedState) section.FindI("Dselect-State", pkg->SelectedState);
	      pkg_state.candver=candver;

	      if(do_dselect && pkg->SelectedState!=pkg_state.dselect_state &&
		 do_initselections)
		{
		  MarkFromDselect(pkg);
		  pkg_state.dselect_state=(pkgCache::State::PkgSelectedState) pkg->SelectedState;
		}
	    }
	  amt+=section.size();
	  Prog.OverallProgress(amt, file_size, 1, _("Reading extended state information"));
	}
      Prog.OverallProgress(amt, file_size, 1, _("Reading extended state information"));
    }

  // Act on them
  for(pkgCache::PkgIterator i=PkgBegin(); !i.end(); i++)
    {
      StateCache &state=(*this)[i];
      aptitude_state &estate=get_ext_state(i);

      if(initial_open) // Don't make everything "new".
	estate.new_package=false;

      switch(estate.selection_state)
	{
	case pkgCache::State::Unknown:
	  if(i.CurrentVer().end())
	    estate.selection_state=pkgCache::State::DeInstall;
	  else
	    estate.selection_state=pkgCache::State::Install;
	  break;
	case pkgCache::State::Install:
	  if(!do_initselections)
	    break;

	  // FIXME: should I check this for "unknown" packages as well?
	  // Does that even make sense??
#ifdef HAVE_LIBAPT_PKG3
	  if(!estate.candver.empty() &&
	     estate.candver!=state.InstVerIter(GetCache()).VerStr())
	    {
	      for(pkgCache::VerIterator ver=i.VersionList(); !ver.end(); ++ver)
		if(ver.VerStr()==estate.candver)
		  {
		    SetCandidateVersion(ver);
		    MarkInstall(ver.ParentPkg());
		    break;
		  }
	    }
	  else
#endif
	    if(i.CurrentVer().end())
	      MarkInstall(i);
	    else
	      SetReInstall(i, estate.reinstall);
	  break;
	case pkgCache::State::Hold:
	  if(!do_initselections)
	    break;

	  MarkKeep(i, false);
	  break;
	case pkgCache::State::DeInstall:
	  if(!do_initselections)
	    break;

	  if(!i.CurrentVer().end())
	    MarkDelete(i, false);
	  break;
	case pkgCache::State::Purge:
	  if(!do_initselections)
	    break;

	  if(!i.CurrentVer().end())
	    MarkDelete(i, true);
	  break;
	}
    }

  duplicate_cache(&backup_state);

  if(aptcfg->FindB(PACKAGE "::Auto-Upgrade", true) && do_initselections)
    mark_all_upgradable(aptcfg->FindB("Auto-Install", true),
			NULL);

  duplicate_cache(&backup_state);

  return true;
}

void aptitudeDepCache::mark_all_upgradable(bool with_autoinst,
					   undo_group *undo)
{
  for(int iter=0; iter==0 || (iter==1 && with_autoinst); ++iter)
    {
      // Do this twice, only turning auto-install on the second time.
      // A reason for this is the following scenario:
      //
      // Packages A and B are installed at 1.0.  Package C is not installed.
      // Version 2.0 of each package is available.
      //
      // Version 2.0 of A depends on "C (= 2.0) | B (= 2.0)".
      //
      // Upgrading A if B is not upgraded will cause this dependency to
      // break.  Auto-install will then cheerfully fulfill it by installing
      // C.
      //
      // A real-life example of this is xemacs21, xemacs21-mule, and
      // xemacs21-nomule; aptitude would keep trying to install the mule
      // version on upgrades.
      bool do_autoinstall=(iter==1);

      for(pkgCache::PkgIterator i=PkgBegin(); !i.end(); i++)
	{
	  StateCache &state=(*this)[i];
	  aptitude_state &estate=get_ext_state(i);

	  switch(estate.selection_state)
	    {

	      // This case shouldn't really happen:
	    case pkgCache::State::Unknown:
	      if(!i.CurrentVer().end())
		estate.selection_state=pkgCache::State::DeInstall;
	      else
		{
		  estate.selection_state=pkgCache::State::Install;
		  if(state.Status>0)
		    MarkInstall(i, do_autoinstall);
		}
	      break;
	    case pkgCache::State::Install:
	      if(state.Status>0)
		MarkInstall(i, do_autoinstall);
	    default:
	      break;
	    }
	}
    }

  if(group_level==0)
    {
      mark_and_sweep(undo);

      cleanup_after_change(undo);

      duplicate_cache(&backup_state);
    }
}

// If fd is -1, just write unconditionally to the given fd.
//
//  FIXME: clean up the logic by having an internal "write to this fd"
// routine and an exported "ok, set up for the write and then clean up"
// routine.
bool aptitudeDepCache::save_selection_list(OpProgress &prog,
					   const char *status_fname)
{
  // Refuse to write to disk if nothing changed and we aren't writing
  // to an unusual file
  if(!dirty && !status_fname)
    return true;

  if(lock==-1 && !status_fname)
    return true;
  string statefile=_config->FindDir("Dir::Aptitude::state", STATEDIR)+"pkgstates";

  FileFd newstate;

  if(!status_fname)
    newstate.Open(statefile+".new", FileFd::WriteEmpty);
  else
    newstate.Open(status_fname, FileFd::WriteEmpty);

  if(!newstate.IsOpen())
    _error->Error(_("Cannot open Aptitude state file"));
  else
    {
      int num=0;
      prog.OverallProgress(0, Head().PackageCount, 1, _("Writing extended state information"));

      for(PkgIterator i=PkgBegin(); !i.end(); i++)
	if(!i.VersionList().end())
	  {
	    aptitude_state &state=get_ext_state(i);
	    char buf[400];
	    int len;

	    // HACK: only write the version-string if we are writing
	    //      to a non-default file.  ICKY.  MUST FIX.
	    string tailstr=status_fname && !state.candver.empty()
	      ?"Version: "+state.candver+"\n":"";

	    if(state.reinstall)
	      tailstr+="ReInstall: yes\n\n";
	    else
	      tailstr+="\n";

	    len=snprintf(buf,
			 400,
			 "Package: %s\nUnseen: %s\nState: %i\nDselect-State: %i\nLast-Change: %i\n%s",
			 i.Name(),
			 state.new_package?"yes":"no",
			 state.selection_state,
			 i->SelectedState,
			 state.installremove_reason,
			 tailstr.c_str());
	    if(len>=399)
	      {
		_error->Error(_("Internal buffer overflow on package \"%s\" while writing state file"), i.Name());
		newstate.Close();

		if(!status_fname)
		  unlink((statefile+".new").c_str());
		return false;
	      }
	    if(newstate.Failed() || !newstate.Write(buf, len))
	      {
		_error->Error(_("Couldn't write state file"));
		newstate.Close();

		if(!status_fname)
		  unlink((statefile+".new").c_str());
		return false;
	      }

	    num++;
	    prog.OverallProgress(num, Head().PackageCount, 1, _("Writing extended state information"));
	  }

      if(newstate.Failed())
	// This is /probably/ redundant, but paranoia never hurts.
	{
	  _error->Error(_("Error writing state file"));
	  newstate.Close();

	  if(!status_fname)
	    unlink((statefile+".new").c_str());
	  return false;
	}
      newstate.Close();
      // FIXME!  This potentially breaks badly on NFS.. (?) -- actually, it
      //       wouldn't be harmful; you'd just get gratuitous errors..
      if(!status_fname)
	{
	  if(rename((statefile+".new").c_str(), statefile.c_str())==-1)
	    {
	      _error->Errno("save_selection_list", _("couldn't replace old state file"));
	      unlink((statefile+".new").c_str());
	      return false;
	    }
	}
    }

  return true;
}

void aptitudeDepCache::forget_new(undoable **undoer)
{
  forget_undoer *undo=undoer?new forget_undoer(this):NULL;

  for(pkgCache::PkgIterator i=PkgBegin(); !i.end(); i++)
    if(package_states[i->ID].new_package)
      {
	dirty=true;
	package_states[i->ID].new_package=false;
	if(undo)
	  undo->add_item(i);
      }

  if(undoer && undo && !undo->empty())
    *undoer=undo;
  else
    delete undo;

  duplicate_cache(&backup_state);

  // Umm, is this a hack? dunno.
  cache_reloaded();
}

undoable *aptitudeDepCache::state_restorer(PkgIterator pkg, StateCache &state, aptitude_state &ext_state)
{
  return new apt_undoer(pkg, state.Mode, state.iFlags, ext_state.installremove_reason, ext_state.selection_state, this);
}

void aptitudeDepCache::cleanup_after_change(undo_group *undo, bool alter_stickies)
  // Finds any packages whose states have changed and: (a) updates the
  // selected_state if it's not already updated; (b) adds an item to the
  // undo group.
{
  for(pkgCache::PkgIterator pkg=PkgBegin(); !pkg.end(); pkg++)
    {
      if(PkgState[pkg->ID].Mode!=backup_state.PkgState[pkg->ID].Mode ||
	 package_states[pkg->ID].selection_state!=backup_state.AptitudeState[pkg->ID].selection_state ||
	 package_states[pkg->ID].reinstall!=backup_state.AptitudeState[pkg->ID].reinstall ||
	 package_states[pkg->ID].installremove_reason!=backup_state.AptitudeState[pkg->ID].installremove_reason)
	{
	  if(alter_stickies &&
	     PkgState[pkg->ID].Mode!=backup_state.PkgState[pkg->ID].Mode &&
	     package_states[pkg->ID].selection_state==backup_state.AptitudeState[pkg->ID].selection_state)
	    // Catch packages which switched without altering their Aptitude
	    // selection mode
	    {
	      switch(PkgState[pkg->ID].Mode)
		{
		case ModeDelete:
		  if(package_states[pkg->ID].selection_state!=pkgCache::State::DeInstall)
		    {
		      if(!pkg.CurrentVer().end())
			package_states[pkg->ID].installremove_reason=libapt;

		      package_states[pkg->ID].selection_state=pkgCache::State::DeInstall;
		    }
		  break;
		case ModeKeep:
		  package_states[pkg->ID].selection_state=pkgCache::State::Hold;
		  break;
		case ModeInstall:
		  if(package_states[pkg->ID].selection_state!=pkgCache::State::Install)
		    {
		      package_states[pkg->ID].selection_state=pkgCache::State::Install;
		      // Only set it to not be manual if it's a new install.
		      // (ie, automagically upgraded packages won't suddenly
		      // be autoremovable)
		      if(pkg.CurrentVer().end())
			package_states[pkg->ID].installremove_reason=libapt;
		    }
		  break;
		}
	    }

	  if(undo)
	    undo->add_item(state_restorer(pkg,
					  backup_state.PkgState[pkg->ID],
					  backup_state.AptitudeState[pkg->ID]));
	}
    }
}

void aptitudeDepCache::internal_mark_install(const PkgIterator &Pkg,
					     bool AutoInst,
					     bool ReInstall,
					     undo_group *undo,
					     bool do_mark_and_sweep)
{
  dirty=true;

  if(!ReInstall)
    pkgDepCache::MarkInstall(Pkg, AutoInst);
  else
    pkgDepCache::MarkKeep(Pkg, AutoInst);

  pkgDepCache::SetReInstall(Pkg, ReInstall);

  get_ext_state(Pkg).selection_state=pkgCache::State::Install;
  get_ext_state(Pkg).reinstall=ReInstall;
  get_ext_state(Pkg).installremove_reason=manual;

  if(group_level==0)
    {
      if(do_mark_and_sweep)
	mark_and_sweep(undo);

      cleanup_after_change(undo);

      duplicate_cache(&backup_state);
    }
}

void aptitudeDepCache::internal_mark_delete(const PkgIterator &Pkg,
					    bool Purge,
					    bool unused_delete,
					    undo_group *undo,
					    bool do_mark_and_sweep)
{
  dirty=true;

  pkgDepCache::MarkDelete(Pkg, Purge);
  pkgDepCache::SetReInstall(Pkg, false);

  get_ext_state(Pkg).selection_state=(Purge?pkgCache::State::Purge:pkgCache::State::DeInstall);
  get_ext_state(Pkg).reinstall=false;

  if(unused_delete)
    get_ext_state(Pkg).installremove_reason=unused;
  else
    get_ext_state(Pkg).installremove_reason=manual;

  // Argh argh argh.  Doing any of this when there's a mark-and-sweep
  // running causes extreme screwiness.
  if(!mark_and_sweep_in_progress && group_level==0)
    {
      if(do_mark_and_sweep)
	mark_and_sweep(undo);

      cleanup_after_change(undo);

      duplicate_cache(&backup_state);
    }
}

void aptitudeDepCache::internal_mark_keep(const PkgIterator &Pkg, bool Soft, bool SetHold, undo_group *undo, bool do_mark_and_sweep)
{
  dirty=true;

  pkgDepCache::MarkKeep(Pkg, Soft);
  pkgDepCache::SetReInstall(Pkg, false);
  get_ext_state(Pkg).reinstall=false;

  if(Pkg.CurrentVer().end())
    {
      if((*this)[Pkg].iFlags&Purge)
	get_ext_state(Pkg).selection_state=pkgCache::State::Purge;
      else
	get_ext_state(Pkg).selection_state=pkgCache::State::DeInstall;
    }
  else if(SetHold)
    get_ext_state(Pkg).selection_state=pkgCache::State::Hold;
  else
    get_ext_state(Pkg).selection_state=pkgCache::State::Install;

  get_ext_state(Pkg).installremove_reason=manual;

  if(group_level==0)
    {
      if(do_mark_and_sweep)
	mark_and_sweep(undo);

      cleanup_after_change(undo);

      duplicate_cache(&backup_state);
    }
}

#ifdef HAVE_LIBAPT_PKG3
void aptitudeDepCache::set_candidate_version(const VerIterator &ver,
					     undo_group *undo)
{
  dirty=true;

  if(!ver.end())
    {
      // Use the InstVerIter instead of GetCandidateVersion, since that seems
      // to store the currently to-be-installed version.
      VerIterator prev=(*this)[(ver.ParentPkg())].InstVerIter(GetCache());

      if(undo)
	undo->add_item(new candver_undoer(prev, this));

      if(ver!=GetCandidateVer(ver.ParentPkg()))
	get_ext_state(ver.ParentPkg()).candver=ver.VerStr();
      else
	get_ext_state(ver.ParentPkg()).candver="";

      SetCandidateVersion(ver);

      mark_and_sweep(undo);
    }
}
#endif

void aptitudeDepCache::mark_single_install(const PkgIterator &Pkg, undo_group *undo)
{
  dirty=true;

  for(PkgIterator i=PkgBegin(); !i.end(); i++)
    pkgDepCache::MarkKeep(i, true);

  pkgDepCache::MarkInstall(Pkg, true);

  if(group_level==0)
    {
      mark_and_sweep(undo);

      cleanup_after_change(undo, false);

      duplicate_cache(&backup_state);
    }
}

void aptitudeDepCache::mark_auto_installed(const PkgIterator &Pkg,
					   bool set_auto,
					   undo_group *undo)
{
  dirty=true;

  get_ext_state(Pkg).installremove_reason=set_auto?user_auto:manual;

  if(group_level==0)
    {
      mark_and_sweep(undo);

      cleanup_after_change(undo);

      duplicate_cache(&backup_state);
    }
}

bool aptitudeDepCache::all_upgrade(undo_group *undo)
{
  pkgProblemResolver fixer(this);

  if(BrokenCount()!=0)
    return false;

  for(pkgCache::PkgIterator pkg=PkgBegin(); !pkg.end(); ++pkg)
    {
      if((*this)[pkg].Install())
	fixer.Protect(pkg);

      if(get_ext_state(pkg).selection_state!=pkgCache::State::Hold &&
	 !pkg.CurrentVer().end() && !(*this)[pkg].Install())
	MarkInstall(pkg, false);
    }

  bool rval=fixer.ResolveByKeep();

  if(group_level==0)
    {
      mark_and_sweep(undo);

      cleanup_after_change(undo);

      duplicate_cache(&backup_state);
    }

  return rval;
}

bool aptitudeDepCache::try_fix_broken(pkgProblemResolver &fixer, undo_group *undo)
{
  dirty=true;
  bool founderr=false;
  if(!fixer.Resolve(true))
    founderr=true;

  if(founderr)
    _error->Error(_("Unable to correct dependencies, some packages cannot be installed"));

  if(group_level==0)
    {
      mark_and_sweep(undo);

      cleanup_after_change(undo);

      duplicate_cache(&backup_state);
    }

  return !founderr;
}

bool aptitudeDepCache::try_fix_broken(undo_group *undo)
{
#ifndef HAVE_LIBAPT_PKG3
  pkgProblemResolver fixer(*this);
#else
  pkgProblemResolver fixer(this);
#endif
  for(pkgCache::PkgIterator i=PkgBegin(); !i.end(); i++)
    {
      fixer.Clear(i);
      if(!i.CurrentVer().end() &&
	 get_ext_state(i).selection_state==pkgCache::pkgCache::State::Hold)
	fixer.Protect(i);
      else
	{
	  pkgDepCache::StateCache &state=(*this)[i];
	  if(state.InstBroken() || state.NowBroken())
	    MarkInstall(i,true);
	  else if(state.Delete())
	    fixer.Remove(i);
	}
    }

  return try_fix_broken(fixer, undo);
}

void aptitudeDepCache::MarkFromDselect(const PkgIterator &Pkg)
{
  aptitude_state &state=get_ext_state(Pkg);

  if(Pkg->SelectedState!=state.selection_state)
    {
      switch(Pkg->SelectedState)
	{
	case pkgCache::State::Unknown:
	  break;
	case pkgCache::State::Purge:
	  if( (!Pkg.CurrentVer().end()) || !((*this)[Pkg].iFlags&Purge) )
	    mark_delete(Pkg, true, false, NULL);
	  else
	    mark_keep(Pkg, false, false, NULL);
	  break;
	case pkgCache::State::DeInstall:
	  if(!Pkg.CurrentVer().end())
	    mark_delete(Pkg, false, false, NULL);
	  else
	    mark_keep(Pkg, false, false, NULL);
	  break;
	case pkgCache::State::Hold:
	  if(!Pkg.CurrentVer().end())
	    mark_keep(Pkg, false, true, NULL);
	  break;
	case pkgCache::State::Install:
	  if(Pkg.CurrentVer().end())
	    mark_install(Pkg, false, false, NULL);
	  else
	    mark_keep(Pkg, false, false, NULL);
	  break;
	}
    }
}

void aptitudeDepCache::duplicate_cache(apt_state_snapshot *target)
  // Remember: the tables in the target have to be correctly sized!
{
  if(!target->PkgState)
    target->PkgState=new StateCache[Head().PackageCount];
  if(!target->DepState)
    target->DepState=new unsigned char[Head().DependsCount];
  if(!target->AptitudeState)
    target->AptitudeState=new aptitude_state[Head().PackageCount];

  memcpy(target->PkgState, PkgState, sizeof(StateCache)*Head().PackageCount);
  memcpy(target->DepState, DepState, sizeof(char)*Head().DependsCount);
  memcpy(target->AptitudeState, package_states, sizeof(aptitude_state)*Head().PackageCount);

  target->iUsrSize=iUsrSize;
  target->iDownloadSize=iDownloadSize;
  target->iInstCount=iInstCount;
  target->iDelCount=iDelCount;
  target->iKeepCount=iKeepCount;
  target->iBrokenCount=iBrokenCount;
  target->iBadCount=iBadCount;
}

void aptitudeDepCache::begin_action_group()
{
  group_level++;
}

void aptitudeDepCache::end_action_group(undo_group *undo)
{
  assert(group_level>0);

  group_level--;

  if(group_level==0)
    {
      mark_and_sweep(undo);

      cleanup_after_change(undo);

      duplicate_cache(&backup_state);
    }
}

const aptitudeDepCache::apt_state_snapshot *aptitudeDepCache::snapshot_apt_state()
{
  apt_state_snapshot *rval=new apt_state_snapshot;
  duplicate_cache(rval);

  return rval;
}

void aptitudeDepCache::restore_apt_state(const apt_state_snapshot *snapshot)
{
  memcpy(PkgState, snapshot->PkgState, sizeof(StateCache)*Head().PackageCount);
  memcpy(DepState, snapshot->DepState, sizeof(char)*Head().DependsCount);
  memcpy(package_states, snapshot->AptitudeState, sizeof(aptitude_state)*Head().PackageCount);

  iUsrSize=snapshot->iUsrSize;
  iDownloadSize=snapshot->iDownloadSize;
  iInstCount=snapshot->iInstCount;
  iDelCount=snapshot->iDelCount;
  iKeepCount=snapshot->iKeepCount;
  iBrokenCount=snapshot->iBrokenCount;
  iBadCount=snapshot->iBadCount;
}

// Mark-and-sweep starts here.
//
// If a package version is not installed and is not going to be installed, it
// is not visited further.  Otherwise, the appropriate mark is set.
//
//  FIXME: follow Recommends and Suggests iff the appropriate stuff is set.
void aptitudeDepCache::mark_package(const PkgIterator &pkg,
				    const VerIterator &ver)
{
  if(!((ver==pkg.CurrentVer() && PkgState[pkg->ID].Keep()) ||
       (ver==PkgState[pkg->ID].InstVerIter(GetCache()) &&
	PkgState[pkg->ID].Install())))
    return;

  if(package_states[pkg->ID].marked)
    return;

  package_states[pkg->ID].marked=true;

  if(!ver.end())
    {
      for(DepIterator d=ver.DependsList(); !d.end(); ++d)
	{
	  if(d->Type==pkgCache::Dep::Depends ||
	     (aptcfg->FindB(PACKAGE "::Recommends-Important", true) &&
	      d->Type==pkgCache::Dep::Recommends) ||
	     (aptcfg->FindB(PACKAGE "::Suggests-Important", false) &&
	      d->Type==pkgCache::Dep::Suggests))
	    {
	      // Try all versions of this package.
	      for(VerIterator V=d.TargetPkg().VersionList(); !V.end(); ++V)
#ifndef HAVE_LIBAPT_PKG3
		if(pkgCheckDep(d.TargetVer(), V.VerStr(), d->CompareOp))
#else
		if(_system->VS->CheckDep(V.VerStr(), d->CompareOp, d.TargetVer()))
#endif
		  mark_package(V.ParentPkg(), V);

	      // Now try virtual packages
	      for(PrvIterator prv=d.TargetPkg().ProvidesList(); !prv.end(); ++prv)
#ifndef HAVE_LIBAPT_PKG3
		if(pkgCheckDep(d.TargetVer(), prv.ProvideVersion(), d->CompareOp))
#else
		if(_system->VS->CheckDep(prv.ProvideVersion(), d->CompareOp, d.TargetVer()))
#endif
		  mark_package(prv.OwnerPkg(), prv.OwnerVer());
	    }
	}
    }
}

// The root-set is every essential package, as well as those packages which
// have been manually chosen for installation, and are planned to be installed.
void aptitudeDepCache::mark_and_sweep(undo_group *undo)
{
  // EWW.
  if(mark_and_sweep_in_progress)
    return;

  mark_and_sweep_in_progress=true;

  std::string matchterm=aptcfg->Find(PACKAGE "::Delete-Unused-Pattern", "");
  pkg_matcher *matcher=matchterm.empty()?NULL:parse_pattern(matchterm);

  for(pkgCache::PkgIterator p=PkgBegin(); !p.end(); ++p)
    {
      package_states[p->ID].marked=false;
      package_states[p->ID].garbage=false;
    }

  for(pkgCache::PkgIterator p=PkgBegin(); !p.end(); ++p)
    {
      if(package_states[p->ID].installremove_reason==manual ||
	 (p->Flags & pkgCache::Flag::Essential) ||
	 (matcher && ((PkgState[p->ID].Keep() &&
		       !p.CurrentVer().end() &&
		       matcher->matches(p, p.CurrentVer())) ||
		      (PkgState[p->ID].Install() &&
		       matcher->matches(p, PkgState[p->ID].InstVerIter(GetCache()))))))
	{
	  if(PkgState[p->ID].Keep() && !p.CurrentVer().end())
	    mark_package(p, p.CurrentVer());
	  else if(PkgState[p->ID].Install())
	    mark_package(p, PkgState[p->ID].InstVerIter(GetCache()));
	}
    }

  // Be sure not to re-delete already deleted packages.
  for(pkgCache::PkgIterator p=PkgBegin(); !p.end(); ++p)
    if(!package_states[p->ID].marked)
      {
	package_states[p->ID].garbage=true;

	if((!p.CurrentVer().end() || PkgState[p->ID].Install()) &&
	   !PkgState[p->ID].Delete() &&
	   aptcfg->FindB(PACKAGE "::Delete-Unused", true))
	  mark_delete(p, false, true, undo);
      }

  mark_and_sweep_in_progress=false;

  delete matcher;
}

#ifndef HAVE_LIBAPT_PKG3
aptitudeCacheFile::aptitudeCacheFile():Map(NULL), Cache(NULL), Lock(NULL)
{
}
#else
aptitudeCacheFile::aptitudeCacheFile()
  :Map(NULL), Cache(NULL), DCache(NULL), have_system_lock(false), Policy(NULL)
{
}
#endif

aptitudeCacheFile::~aptitudeCacheFile()
{
  delete Cache;
  delete Map;
#ifndef HAVE_LIBAPT_PKG3
  delete Lock;
#else
  ReleaseLock();

  delete DCache;
  delete Policy;
#endif
}

#ifndef HAVE_LIBAPT_PKG3
bool aptitudeCacheFile::Open(OpProgress &Progress, bool do_initselections,
			     bool WithLock, const char *status_fname)
{
  if(WithLock)
    Lock=new pkgDpkgLock;

  if(_error->PendingError())
    return false;

  pkgSourceList List;
  if(!List.ReadMainList())
    return _error->Error(_("The list of sources could not be read."));

  if(WithLock)
    {
      pkgMakeStatusCache(List, Progress);
      if(_error->PendingError())
	return _error->Error(_("The package lists or status file could not be parsed or opened."));
      if(!_error->empty())
	_error->Warning(_("You may want to update the package lists to correct these missing files"));

      FileFd File(_config->FindFile("Dir::Cache::pkgcache"), FileFd::ReadOnly);
      if(_error->PendingError())
	return false;

      Map=new MMap(File, MMap::Public|MMap::ReadOnly);
      if(_error->PendingError())
	return false;
    }
  else
    {
      Map=pkgMakeStatusCacheMem(List, Progress);
      Progress.Done();
      if(!Map)
	return false;
    }

  Cache=new aptitudeDepCache(*Map, Progress, WithLock,
			     do_initselections, status_fname);
  Progress.Done();
  if(_error->PendingError())
    return false;

  return true;
}

bool aptitudeCacheFile::GainLock()
{
  if(Lock)
    return true;

  // Hacky but this is an obsolete library that I'm using..just do something
  // that works
  _error->Discard();

  Lock=new pkgDpkgLock;

  if(_error->PendingError())
    {
      delete Lock;
      return false;
    }
  else
    return true;
}

#else
bool aptitudeCacheFile::Open(OpProgress &Progress, bool do_initselections,
			     bool WithLock, const char *status_fname)
{
  if(WithLock)
    {
      if(!_system->Lock())
	return false;

      have_system_lock=true;
    }

  if(_error->PendingError())
    return false;

  pkgSourceList List;
  if(!List.ReadMainList())
    return _error->Error(_("The list of sources could not be read."));

  // Read the caches:
  bool Res=pkgMakeStatusCache(List, Progress, &Map, !WithLock);
  Progress.Done();

  if(!Res)
    return _error->Error(_("The package lists or status file could not be parsed or opened."));

  // Jason says this is bad.  I agree. :)
  if(!_error->empty())
    _error->Warning(_("You may want to update the package lists to correct these missing files"));

  Cache=new pkgCache(Map);
  if(_error->PendingError())
    return false;

  Policy=new aptitudePolicy(Cache);
  if(_error->PendingError())
    return false;
  if(!ReadPinFile(*Policy))
    return false;

  DCache=new aptitudeDepCache(Cache, Policy);
  if(_error->PendingError())
    return false;

  DCache->Init(&Progress, WithLock, do_initselections, status_fname);
  Progress.Done();

  if(_error->PendingError())
    return false;

  return true;
}

void aptitudeCacheFile::ReleaseLock()
{
  if(have_system_lock)
    {
      _system->UnLock();
      have_system_lock=false;
    }
}

bool aptitudeCacheFile::GainLock()
{
  if(have_system_lock)
    return true;

  if(!_system->Lock())
    return false;

  have_system_lock=true;
  return true;
}

#endif
