// aptcache.h  -*-c++-*-
//
//  Replacements for the Apt cache and dependency cache file classes.  Those
// are lacking in a couple of crucial ways (eg: persistent state), so I
// have to come up with my own set of routines.. :(
//
//  In order to track the appropriate package information, we must keep our
// own file (I think..apt doesn't have hooks for adding stuff..) containing
// that information, properly lock/unlock it, etc.
//
//  Copyright 1999,2000,2001,2002 Daniel Burrows
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; see the file COPYING.  If not, write to
//  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
//  Boston, MA 02111-1307, USA.

#ifndef APTCACHE_H
#define APTCACHE_H

#include <config.h>

#include <apt-pkg/depcache.h>

#include <sigc++/basic_signal.h>

#ifndef HAVE_LIBAPT_PKG3
#include <apt-pkg/dpkginit.h>
#endif

class undoable;
class undo_group;
class pkgProblemResolver;

class aptitudeDepCache:public pkgDepCache
{
public:
  enum changed_reason {manual, user_auto, libapt, temp_hold, unused};

  struct aptitude_state
  {
    bool new_package;
    // True if the package is a "new" package.  This is a slightly more complex
    // state than dselect's "unknown" state, as it persists until the
    // 'forget new' command is executed.

    pkgCache::State::PkgSelectedState selection_state;
    // Ok, I give in.  I think I see how to implement something slightly more
    // general, but this will do for the current release.  Has the same
    // semantics as dselect's SelectedState flag, except that Unknown isn't
    // included -- making things a lot simpler.
    //  A quick digression on why this is actually a somewhat Good Idea: apt's
    // modes don't tell us anything about a package except for what's to be
    // done with it /right now/.  The dselect selection state paradigm is
    // useful because it states what the user *wants done* with a package.
    //  The limitations of this are that it makes it hard to represent things
    // like downgrading to a previous version, following the latest version
    // in a particular release, and so on.  Those will be addressed here
    // if no-one puts them in libapt first (which would be much better, since
    // otherwise apt-get and the other apt tools will have the unpleasant 
    // effect of clobbering our state)

    pkgCache::State::PkgSelectedState dselect_state;
    //  Used in an attempt to detect changes to the selected state by dselect.
    //  Hopefully this will make aptitude a little less annoying about fighting
    // over package states.  The idea is that we keep the dselect/dpkg state
    // of the package as of Aptitude's last run in this item, and if it changes
    // we assume that the user has fiddled with the package in one of those
    // tools and flip our own state.  We'll see how it goes ;-)
    //  And yes, the Right Thing[tm] to do would be to actually use the dpkg
    // status db, /except/ that I still want to extend this and I don't have
    // a good way to write back to the status database (dpkg --set-selections
    // is actually problematic..)

    std::string candver;
    // This is a bit of a hack: if the user selects a version other than the
    // default candidate version, it will be stored here.  This isn't really
    // meant to substitute for tags, and in fact isn't especially accurate.
    // Because of that, it is NOT saved to the global state-file; it is only
    // saved when creating a temporary one for su-to-root.

    bool reinstall;
    //  True if the package in question should be reinstalled

    bool tagged;
    // Tracks Mutt-style tags.

    bool flagged;
    // Tracks the "interesting" flag for packages.

    changed_reason installremove_reason;
    // Who set the package to its current state?
    // (multiple ones since a package can be both installed and held)

    bool marked;
    // For mark-and-sweep.

    bool garbage;
    // The garbage-collector sets this on all packages it would delete.
    // This way, if you have told it not to delete unused packages,
    // you can still find out which ones are unused.
  };

  // true if something (anything) has been changed.
  bool dirty;

  // Some internal classes for undo information
  class apt_undoer;
  class forget_undoer;
  friend class apt_undoer;
  friend class forget_undoer;

#ifdef HAVE_LIBAPT_PKG3
  class candver_undoer;
  friend class candver_undoer;
#endif

  class apt_state_snapshot
  {
    StateCache *PkgState;
    unsigned char *DepState;
    aptitude_state *AptitudeState;
#ifndef HAVE_LIBAPT_PKG3
    signed long iUsrSize;
    unsigned long iDownloadSize;
#else
    double iUsrSize;
    double iDownloadSize;
#endif
    unsigned long iInstCount;
    unsigned long iDelCount;
    unsigned long iKeepCount;
    unsigned long iBrokenCount;
    unsigned long iBadCount;

  private:
    apt_state_snapshot():PkgState(NULL), DepState(NULL), AptitudeState(NULL) {}

  public:
    ~apt_state_snapshot()
    {
      delete[] PkgState;
      delete[] DepState;
      delete[] AptitudeState;
    }

    friend class aptitudeDepCache;
  };
private:
  aptitude_state *package_states;
  int lock;
  // The lock on the extra-info file.
  int group_level;
  // The current 'group level' -- how many times start_action_group has been
  // called without a matching end_action_group.

  bool mark_and_sweep_in_progress;
  // Tracks whether a mark-and-sweep is running to avoid infinite recursion.
  // FIXME: disgusting hack.

  apt_state_snapshot backup_state;
  // Stores what the cache was like just before an action was performed

  undoable *state_restorer(PkgIterator pkg, StateCache &state, aptitude_state &ext_state);
  // Returns an 'undoable' object which will restore the given package to the
  // given state via {Mark,Set}* routines

  void duplicate_cache(apt_state_snapshot *target);
  // This makes the **ASSUMPTION** that if the target's tables aren't
  // NULL, they're properly sized..

  void cleanup_after_change(undo_group *undo, bool alter_stickies=true);
  // Finds anything that magically changed and creates an undo item for it..
  // If alter_stickies is false, sticky states will be left alone.  (hack :( )

  void MarkFromDselect(const PkgIterator &Pkg);
  // Marks the package based on its current status and its dselect state,
  // adjusting its selected state as appropriate.

  // gc
  void mark_package(const PkgIterator &pkg,
		    const VerIterator &ver);
  void mark_and_sweep(undo_group *undo);

  // The following three methods exist so that undos function sensibly.
  // The problem is that (a) undo groups contain complete information about
  // the cache state, so mark-and-sweep is unnecessary, and (b) if the
  // undo groups are executed in the wrong order, the results will also
  // be wrong.  For instance: A depends on B (auto installed) and A is
  // removed.  When this is undone, B is first re-installed and marked as
  // "auto", causing it to immediately be removed again!
  void internal_mark_install(const PkgIterator &Pkg, bool AutoInst, bool ReInstall, undo_group *undo, bool do_mark_sweep);
  void internal_mark_delete(const PkgIterator &Pkg, bool Purge, bool unused_delete, undo_group *undo, bool do_mark_sweep);
  void internal_mark_keep(const PkgIterator &Pkg, bool Soft, bool SetHold, undo_group *undo, bool do_mark_sweep);
public:
#ifndef HAVE_LIBAPT_PKG3
  aptitudeDepCache(MMap &Map, OpProgress &Prog, bool WithLock,
		   bool do_initselections, const char * status_fname=NULL);

  bool IsImportantDep(pkgCache::DepIterator dep);
#else
  aptitudeDepCache(pkgCache *cache, Policy *Plcy=0);

  bool Init(OpProgress *Prog, bool WithLock,
	    bool do_initselections, const char * status_fname=NULL);

  bool is_locked() {return lock!=-1;}
#endif

  bool is_dirty() {return dirty;}

  // If do_initselections is "false", the "sticky states" will not be used
  // to initialize packages.  (important for the command-line mode)
  bool build_selection_list(OpProgress &Prog, bool WithLock,
			    bool do_initselections,
			    const char * status_fname=NULL);

  void forget_new(undoable **undoer);
  // Clears all information about which packages are 'new'.  Overwrites undoer
  // if it is not NULL

  inline aptitude_state &get_ext_state(const PkgIterator &Pkg)
  {return package_states[Pkg->ID];}

  bool save_selection_list(OpProgress &prog, const char *status_fname=NULL);
  // If the list isn't locked (or an fd isn't provided), is a NOP.

  void begin_action_group();
  void end_action_group(undo_group *undo);
  //  Start and end, respectively, a group of actions.  The benefit of using
  // these routines is EFFICIENCY -- in particular, mark_* will refrain from
  // iterating over the entire package cache while these are enabled.
  // These are meant to be called at the beginning and end of a large group
  // of successive state changes.  All action groups should really be folded
  // up before you return to the UI.
  //  Calls to these routines nest.

  void mark_install(const PkgIterator &Pkg, bool AutoInst, bool ReInstall, undo_group *undo)
  {internal_mark_install(Pkg, AutoInst, ReInstall, undo, true);}
  void mark_delete(const PkgIterator &Pkg, bool Purge, bool unused_delete, undo_group *undo)
  {internal_mark_delete(Pkg, Purge, unused_delete, undo, true);}
  void mark_keep(const PkgIterator &Pkg, bool Soft, bool SetHold, undo_group *undo)
  {internal_mark_keep(Pkg, Soft, SetHold, undo, true);}

  void set_candidate_version(const VerIterator &TargetVer, undo_group *undo);
  // These just wrap the equivalent depCache functions for the UI's benefit;
  // they mark what the user wants done with each package.
  //  If an undo group is passed in, items are added to it for each of the
  // actions taken.

  void mark_all_upgradable(bool with_autoinst, undo_group *undo);
  // Marks any upgradable packages for upgrade.

  void mark_single_install(const PkgIterator &pkg, undo_group *undo);
  // Marks this package to be install, and all other packages to be kept.
  //  The "keep" on the other packages, however, is NOT sticky and will NOT
  // be saved in the extended state file! (this is a bit of a hack..)
  // (this could be fairly easily emulated; it's a convenience routine)

  void mark_auto_installed(const PkgIterator &pkg,
			   bool set_auto,
			   undo_group *undo);
  // Marks the given package as having been autoinstalled (so it will be
  // removed automatically) or having been installed manually.

  bool all_upgrade(undo_group *undo);
  // Wrapper for pkgAllUpgrade (the engine of "apt-get upgrade")

  bool try_fix_broken(undo_group *undo);
  // Attempts to fix any broken packages, dumping the changes the problem
  // fixer creates into the given undo group.

  bool try_fix_broken(pkgProblemResolver &fixer, undo_group *undo);
  // Just runs the resolver given and catches automatic changes.
  // (this lets callers customize the information given to the resolver)

  const apt_state_snapshot *snapshot_apt_state();
  // Returns the current state of the *APT* cache (no information about
  // Aptitude states is included); this is meant to be used to implement undo,
  // detection of "wtf happened??" after the problem resolver runs, etc.
  void restore_apt_state(const apt_state_snapshot *snapshot);
  // Restores the *APT* cache to the given state.

  // Emitted when a package's marking is changed (to install, for instance).
  SigC::Signal1<void, const PkgIterator &> package_state_changed;

  // Emitted when a package's categorization is potentially changed.
  // (in particular, when package "new" states are forgotten)
  SigC::Signal0<void> package_category_changed;

  virtual ~aptitudeDepCache();
};

#ifndef HAVE_LIBAPT_PKG3
class aptitudeCacheFile
// Hack around problems in libapt.  Most of the code associated with this
// class was copied directly from libapt and reindented..
{
  MMap *Map;
  aptitudeDepCache *Cache;
  pkgDpkgLock *Lock;
public:
  // We look pretty much exactly like a pointer to a dep cache
  inline operator aptitudeDepCache &() {return *Cache;};
  inline operator aptitudeDepCache *() {return Cache;};
  inline aptitudeDepCache *operator ->() {return Cache;};
  inline aptitudeDepCache &operator *() {return *Cache;};
  inline aptitudeDepCache::StateCache &operator [](pkgCache::PkgIterator const &I) {return (*Cache)[I];};
  inline unsigned char &operator [](pkgCache::DepIterator const &I) {return (*Cache)[I];};

  inline void ReleaseLock() {if(Lock) {Lock->Close(); delete Lock; Lock=NULL;}}
  bool GainLock();

  bool Open(OpProgress &Progress, bool do_initselections, bool WithLock=true,
	    const char * status_fname=NULL);

  bool is_locked() {return Lock!=NULL;}
  
  aptitudeCacheFile();
  ~aptitudeCacheFile();
};
#else
class pkgPolicy;

class aptitudeCacheFile
// Hack around problems in libapt.  Most of the code associated with this
// class was copied directly from libapt and reindented..
{
  MMap *Map;

  pkgCache *Cache;
  aptitudeDepCache *DCache;

  bool have_system_lock;
  // hm, used to make it look like the old stuff?
public:

  pkgPolicy *Policy;

  // We look pretty much exactly like a pointer to a dep cache
  inline operator pkgCache &() {return *Cache;};
  inline operator pkgCache *() {return Cache;};
  inline operator aptitudeDepCache &() {return *DCache;};
  inline operator aptitudeDepCache *() {return DCache;};
  inline aptitudeDepCache *operator ->() {return DCache;};
  inline aptitudeDepCache &operator *() {return *DCache;};
  inline aptitudeDepCache::StateCache &operator [](pkgCache::PkgIterator const &I) {return (*DCache)[I];};
  inline unsigned char &operator [](pkgCache::DepIterator const &I) {return (*DCache)[I];};

  bool Open(OpProgress &Progress, bool do_initselections, bool WithLock=true,
	    const char * status_fname=NULL);
  bool is_locked() {return have_system_lock;} // EWW (also not quite right)

  void ReleaseLock();
  bool GainLock();

  aptitudeCacheFile();
  ~aptitudeCacheFile();
};
#endif

#endif
