# 
# archive-cached-revisions.sh: list revisions cached in whole-source form
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#       --version | -V
#       --help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
                printf "list full source trees cached in an archive\\n"
                printf "usage: archive-cached-revisions [options] [[archive/]version]\\n"
                printf "\\n"
                printf " -V --version                  print version info\\n"
                printf " -h --help                     display help\\n"
                printf "\\n"
                printf " -R --root root                specify the local archive root\\n"
                printf " -A --archive archive          specify the archive name\\n"
                printf "\\n"
		printf "List the revisions cached in whole source form in VERSION.\\n"
		printf "\\n"
		printf "Also see \"larch archive-cache-revision --help\"\\n"
                printf "\\n"
                exit 0
                ;;

      *)
                ;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

archroot=
archive=
cache_dir=

while test $# -ne 0 ; do

  case "$1" in 

    -R|--root)          shift
                        if test $# -eq 0 ; then
                          printf "archive-cached-revisions: -R and --root require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archroot="$1"
                        shift
                        ;;

    -A|--archive)       shift
                        if test $# -eq 0 ; then
                          printf "archive-cached-revisions: -A and --archive require an argument\\n" 1>&2
                          printf "try --help\\n" 1>&2
                          exit 1
                        fi
                        archive="$1"
                        shift
                        ;;

    --)			shift
    			break
			;;

    -*)                 printf "archive-cached-revisions: unrecognized option (%s)\\n" "$1" 1>&2
                        printf "try --help\\n" 1>&2
                        exit 1
                        ;;

    *)                  break
                        ;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -gt 1 ; then
  printf "usage: archive-cached-revisions [options] [[archive/]version]\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

if test $# -ne 0 ; then
  version_spec="$1"
  shift
fi

################################################################
# Sanity Check and Process Defaults
# 

if test -z "$version_spec" ; then
  version_spec="`larch tree-version`"
fi

larch valid-package-name -e archive-cached-revisions --tolerant -- "$version_spec"
archive=`larch parse-package-name -R "$archroot" -A "$archive" --arch "$version_spec"`
version_spec=`larch parse-package-name --non-arch "$version_spec"`
category=`larch parse-package-name --basename "$version_spec"`
branch=`larch parse-package-name --package "$version_spec"`

################################################################
# Ensure that We Have an Archive Connection 
# 

if test "$WITHARCHIVE" != "$archive" ; then
  exec larch with-archive -A "$archive"  larch archive-cached-revisions "$archive/$version_spec"
fi

################################################################
# Fix This
# 

if larch valid-package-name --vsn "$version_spec" ; then
  version="$version_spec"
else
  version="`larch versions --reverse \"$archive/$version_spec\" | head -1`"
  if test -z "$version" ; then
    printf "\\n" 1>&2
    printf "archive-cached-revisions: no versions on branch.\\n" 1>&2
    printf "  archive: %s\\n" "$archive" 1>&2
    printf "  branch: %s\\n" "$version_spec" 1>&2
    printf "\\n" 1>&2
    exit 1
  fi
fi

################################################################
# Get the List
# 


for rvsn in `larch revisions "$archive/$version"` ; do

  wftp-home
  wftp-cd "$category/$branch/$version/$rvsn"
  for f in `wftp-ls` ; do
    if test "$f" = "$version--$rvsn.tar.gz" ; then
      printf "%s\\n" "$version--$rvsn"
    fi
  done

done

# tag: Tom Lord Mon Dec 10 22:09:53 2001 (archives/archive-cached-revisions.sh)
#
