# show-config.sh: show the revision frontier of a configuration
# 
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 
command_line="$*"

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "show the revision frontier of a configuration\\n"
		printf "usage: show-config [options] config-name\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " --silent                      no output (except odd errors)\\n"
		printf " --quiet                       brief output\\n"
		printf " --report                      default output\\n"
		printf " --verbose                     maximal output\\n"
		printf " --debug                       debugging output\\n"
		printf "\\n"
		printf " -d --dir DIR                  cd to DIR first\\n"
		printf " --config-dir CFG              get config from directory CFG\\n"
		printf "\\n"
		printf "Print the revision frontier specified by the named configuration\\n"
		printf "in the CFG directory (or \"configs\" subdirectory\\n"
		printf "at the root of the project tree containing DIR).\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

dir=.
config_dir=

while test $# -ne 0 ; do

  case "$1" in 

    -d|--dir)		shift
    			if test $# -eq 0 ; then
			  printf "show-config: -d and --dir require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			dir=$1
			shift
			;;

    --config-dir)       shift
			if test $# -eq 0 ; then
			  printf "show-config: --config-dir requires an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			config_dir="$1"
			shift
			;;


    --)			shift
    			break
			;;

    -*)			printf "show-config: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: show-config [options] config-name\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

name="$1"
shift

################################################################
# Sanity Check and Process Defaults
# 

cd "$dir"
dir="`pwd`"

wdroot="`larch tree-root --accurate`"
cd "$wdroot"

config_file="`larch indicated-config-file -e show-config --config-dir \"$config_dir\" --dir \"$wdroot\" --exists \"$name\"`"

################################################################
# Compute the New Configuration
# 

cd "$wdroot"


cat "$config_file" \
| larch file-syntax-filter --sh-comments --blank-lines --trailing-spaces \
| sort -k 1 \
| awk -v here="$wdroot" \
       '{
          print "cd \"" here "\"" ; 
	  print "printf \"%-30s\\t\" \"" $1 "\""  ;
	  print "cd \"" $1 "\"" ; 
 	  print "x=\"`larch log-ls --full --reverse \\\"" $2 "\\\"`\"" ; 
	  print "if test -z \"$x\" ; then"
	  print "  printf \"show-config: no log entries for %s\\\\n\" \"" $2 "\" 1>&2"
	  print "  exit 1"
	  print "else"
	  print "  larch log-ls --full --reverse \"" $2 "\" | head -1"
	  print "fi"
        }' \
| $ARCH_SHELL -e


# tag: Tom Lord Tue Jan 29 14:02:20 2002 (configurations/show-config.sh)
#
