# 
# parse-package-name.sh - parse a package name
################################################################
# Copyright (C) 2001, 2002 Tom Lord
# 
# See the file "COPYING" for further information about
# the copyright and warranty status of this work.
# 

set -e 

################################################################
# special options
# 
# Some options are special:
# 
#	--version | -V
#	--help | -h
# 
if test $# -ne 0 ; then

  for opt in "$@" ; do
    case $opt in

      --version|-V) exec larch --version
                    ;;


      --help|-h)
		printf "parse a package name\\n"
		printf "usage: parse-package-name [options] [--] [archive/]package\\n"
		printf "\\n"
		printf " -V --version                  print version info\\n"
		printf " -h --help                     display help\\n"
		printf "\\n"
		printf " -R --root root                specify the local archive root\\n"
		printf " -A --archive archive          specify the default archive name\\n"
		printf "\\n"
		printf " -a --arch                     print the archive name\\n"
		printf " --non-arch                    print the non-archive part of the name\\n"
		printf " -b --basename                 print the basename\\n"
		printf " -B --branch                   print the branch label\\n"
		printf " -p --package                  print basename and branch label\\n"
		printf " -v --vsn                      print the version number\\n"
		printf " --package-version             print baename, branch and version number\\n"
		printf " -l --lvl (or --patch-level)   print the patch-level\\n"
		printf "\\n"
		printf "Options -b, -B, -p, -v, and -l cancel each other.\\n"
		printf "\\n"
		printf "For -b, -v, and -l, just the id string is printed\\n"
		printf "without \"--\".\\n"
		printf "\\n"
		printf "For -p, the output is \"<basename>--<branch>\".\\n"
		printf "\\n"
		exit 0
      		;;

      *)
		;;
    esac
  done
fi

################################################################
# Ordinary Options
# 
# 

type=package
validation=

while test $# -ne 0 ; do

  case "$1" in 
    --)			shift
    			break
			;;
			
    -R|--root)		shift
    			if test $# -eq 0 ; then
			  printf "parse-package-name: -R and --root require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archroot="$1"
			shift
			;;

    -A|--archive)	shift
    			if test $# -eq 0 ; then
			  printf "parse-package-name: -A and --archive require an argument\\n" 1>&2
			  printf "try --help\\n" 1>&2
			  exit 1
			fi
			archive="$1"
			shift
			;;

    -a|--arch)		type=archive
			validation=-b
			shift
			;;

    --non-arch)		type=non-archive
    			validation=-b
			shift
			;;

    -b|--basename)	type=basename
			validation=-b
			shift
			;;

    -B|--branch)	type=branch
			validation=
			shift
			;;

    -p|--package)	type=package
			validation=
			shift
			;;

    -v|--vsn)   	type=version
			validation=-v
			shift
			;;

    --package-version) 	type=package-version
			validation=-v
			shift
			;;

    -l|--lvl|--patch-level)	
			type=patch-level
			validation=-l
			shift
			;;

    -*)			printf "parse-package-name: unrecognized option (%s)\\n" "$1" 1>&2
			printf "try --help\\n" 1>&2
			exit 1
			;;

    *)			break
    			;;
  esac

done



################################################################
# Ordinary Arguments
# 

if test $# -ne 1 ; then
  printf "usage: parse-package-name [options] [--] [archive/]package\\n" 1>&2
  printf "try --help\\n" 1>&2
  exit 1
fi

package="$1"

arch=${package%%/*}
package=${package#*/}

if test "$arch" = "$package" ; then
  arch=
fi

################################################################
# Parse Package Name
# 

larch valid-package-name -e "parse-package-name"  -t $validation -- "$package"

if test ! -z "$arch" ; then
  larch valid-archive-name -e parse-package-name -- "$arch"
fi

case $type in 

  archive)		if test ! -z "$arch" ; then
  			  printf "%s\\n" "$arch"
			else
			  larch my-default-archive -R "$archroot" -A "$archive"
			fi
	  		;;

  non-archive)		printf "%s\\n" "$package"
  			;;

  basename)		printf "%s\\n" "$package" | sed -e "s/--.*//"
			;;

  branch)		printf "%s\\n" "$package" | sed -e "s/--[0-9].*//" -e "/--/!d" -e "s/\\(.*\\)--\\(.*\\)/\\2/"
			;;

  package)		printf "%s\\n" "$package" | sed -e "s/--[0-9].*//"
			;;	

  version)		printf "%s\\n" "$package" | sed -e "s/.*--\\([0-9][0-9]*\\.[0-9][0-9]*\\)\\(--.*\\)\\{0,1\\}/\\1/" 
			;;

  package-version)	printf "%s\\n" "$package" | sed -e "s/\\(.*--[0-9][0-9]*\\.[0-9][0-9]*\\)\\(--.*\\)\\{0,1\\}/\\1/" 
			;;

  patch-level)		printf "%s\\n" "$package" | sed -e "s/.*--//"
			;;
esac


