/*
 * Copyright (c) 2000 QoSient, LLC
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted, 
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear
 * in supporting documentation, and that the name of QoSient not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.  
 * 
 * QOSIENT, LLC DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 * SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS, IN NO EVENT SHALL QOSIENT, LLC BE LIABLE FOR ANY
 * SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER
 * RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF
 * CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF OR IN
 * CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifndef ArgusClient_h
#define ArgusClient_h
#endif


#include <stdio.h>
#include <string.h>
#include <sys/time.h>


#if defined(HAVE_SOLARIS)
#include <sys/types.h>
#include <sys/socket.h>
#endif

#include <netinet/in.h>

#include <compat.h>
#include <ctype.h>

#include <pcap.h>

#include <argus_def.h>
#include <argus_out.h>
#include <argus_parse.h>
#include <argus_filter.h>

#define RA_MODELTAGSTR		"Model"
#define RA_FLOWTAGSTR		"Flow"

#define RA_MODELIST		1
#define RA_FLOWLIST		2

#define RA_FLOWPOLICYFIELDNUM	9
#define RA_MODELPOLICYFIELDNUM	7
  
#define RA_LABELSTRING		0
#define RA_POLICYID		1
#define RA_POLICYSRCADDR	2
#define RA_POLICYDSTADDR	3
#define RA_POLICYPROTO		4
#define RA_POLICYSRCPORT	5
#define RA_POLICYDSTPORT	6
#define RA_POLICYMODELST	7
#define RA_POLICYTIMEOUT	8

#define RA_MODIFIED		1

#define RA_CON			1
#define RA_DONE			2

#define RA_CIDR_TYPE		1
#define RA_NETCLASS_TYPE	2


#define RA_HASHTABLESIZE	65536
#define RA_SVCPASSED		0x0100
#define RA_SVCFAILED		0x0200
#define RA_SVCINCOMPLETE        0x0400
#define RA_SVCTEST		(RA_SVCFAILED|RA_SVCPASSED|RA_SVCINCOMPLETE)


#define ARGUS_FAR_SRCADDR_MODIFIED      0x0100
#define ARGUS_FAR_DSTADDR_MODIFIED      0x0200
#define ARGUS_FAR_PROTO_MODIFIED        0x0400
#define ARGUS_FAR_SRCPORT_MODIFIED      0x0800
#define ARGUS_FAR_DSTPORT_MODIFIED      0x1000
#define ARGUS_FAR_TPVAL_MODIFIED        0x2000

#define ARGUS_MAX_SORT_ALG	32
#define MAX_SORT_ALG_TYPES	19

#define RASORTSTARTTIME		0
#define RASORTLASTTIME		1
#define RASORTDURATION		2
#define RASORTSRCADDR		3
#define RASORTDSTADDR		4
#define RASORTPROTOCOL		5
#define RASORTSRCPORT		6
#define RASORTDSTPORT		7
#define RASORTSRCTOS		8
#define RASORTDSTTOS		9
#define RASORTSRCTTL		10
#define RASORTDSTTTL		11
#define RASORTBYTECOUNT		12
#define RASORTSRCBYTECOUNT	13
#define RASORTDSTBYTECOUNT	14
#define RASORTPKTSCOUNT		15
#define RASORTSRCPKTSCOUNT	16
#define RASORTDSTPKTSCOUNT      17
#define RASORTLOAD              18

struct ArgusServiceRecord {
   u_int status;
   struct ArgusRecord *argus;
};

struct ArgusHoldingBuffer {
   int size, count;
   struct ArgusListStruct **array;
};

struct ArgusQueueHeader {
   struct ArgusQueueHeader *prv, *nxt;
   struct RaQueueStruct *queue;
   int status;
   struct timeval lasttime, logtime;
};

struct RaHashTableStruct {
   int size, count;
   struct RaHashTableHeader **array;
};

struct RaQueueStruct {
   int count;
   struct RaHashTableStruct *htable;
   struct ArgusQueueHeader **array;
   struct ArgusQueueHeader *start;
};

struct ArgusTimeStatsStruct {
   int n;
   long long sumtime;
   double sumsqrd;
};

struct ArgusRecordData {
   int status, farhdrstatus;
   struct ArgusFarHeaderStruct *farhdrs[32];
   struct ArgusFlow flow;
   struct ArgusRecord *argus;
   struct ArgusAGRStruct agr;
   struct ArgusTimeStatsStruct act, idle;
};

#define ARGUS_TIMESERIES                16

struct ArgusRecordStore {
   struct ArgusQueueHeader qhdr;
   struct RaHashTableStruct *htable;
   struct RaHashTableHeader *rahtblhdr;
   struct RaQueueStruct *queue;
   int  ArgusTimeout, ArgusIdleTimeout;
   int  status, sortag;
   char *displaystr;
   struct ArgusRecordData **data;
};

struct RaCIDRAddr {
   u_int type, addr, mask;
};

struct RaPolicyStruct {
   u_int RaEntryType, RaPolicyId;
   struct RaCIDRAddr src, dst;
   u_char proto;
   u_short sport, dport;
   u_int RaModelId, ArgusTimeout, ArgusIdleTimeout;
   char *str;
};


struct RaHashTableHeader {
   struct RaHashTableHeader *nxt, *prv;
   struct RaHashTableStruct *htable;
   unsigned short hash;
   struct ArgusFlow flow;
   struct ArgusRecordStore *storeobj;
};



struct ArgusListObjectStruct {
   struct ArgusListObjectStruct *nxt, *prv;
   void *obj;
};


struct ArgusListStruct {
   struct ArgusListObjectStruct *start;
   unsigned int count;
};
 
void RaProcessRecord (struct ArgusRecord *);
void RaProcessSrvRecord (struct ArgusServiceRecord *);
 

#ifdef ArgusClient

char *appOptstring = NULL;

int RaPrintDate = 1;
int RaPrintTime = 1;
int RaPrintDuration = 0;
int RaPrintIndicator = 1;
int RaPrintTransactions = 0;
int RaPrintMACAddress = 0;
int RaPrintProto = 1;
int RaPrintSrcAddr = 1;
int RaPrintSrcPort = 1;
int RaPrintDir = 1;
int RaPrintDstAddr = 1;
int RaPrintDstPort = 1;
int RaPrintCounts = 1;
int RaPrintSrcPackets = 1;
int RaPrintDstPackets = 1;
int RaPrintBytes = 1;
int RaPrintSrcBytes = 1;
int RaPrintDstBytes = 1;
int RaPrintJitter = 0;
int RaPrintLoad = 0;
int RaPrintUserData = 0;
int RaPrintStatus = 1;

int RaCumulativeMerge = 1;

int RaFlowModelFile = 0;
int RaAllocHashTableHeaders = 0;
int RaAllocArgusRecordStore = 0;
int RaAllocArgusRecord      = 0;

int RaHistoTimeSeries = 1;

struct ArgusFlow RaArgusFlowBuf, *RaArgusFlow = &RaArgusFlowBuf;
long long RaThisActiveDuration = 0;
int RaThisActiveIndex = 0;

struct RaCIDRAddr RaCIDRBuffer, *RaCIDRPtr = &RaCIDRBuffer;

int RaThisFlowNum = 0;
int RaThisModelNum = 0;
int RaParseError = 0;

struct RaHashTableStruct RaHashTable;

struct RaPolicyStruct  *RaFlowArray[1024];
struct RaPolicyStruct *RaModelArray[1024];

struct RaPolicyStruct *RaFlowModelPolicy = NULL;

struct RaQueueStruct *RaModelerQueue = NULL;
struct RaQueueStruct *RaAvailableQueue = NULL;
struct RaQueueStruct *RaFailedQueue = NULL;

int RaSortIndex = 0;
void RaSortQueue (struct RaQueueStruct *);
char *RaSortAlgorithmStrings[ARGUS_MAX_SORT_ALG];

int (*RaSortAlgorithm)(struct ArgusRecord *, struct ArgusRecord *) = NULL;
int (*RaSortAlgorithms[ARGUS_MAX_SORT_ALG])(struct ArgusRecord *, struct ArgusRecord *);

extern char *RaSortAlgorithmStrings[];

char *RaSortKeyWords[MAX_SORT_ALG_TYPES] = {
   "startime",
   "lasttime",
   "duration",
   "srcaddr",
   "dstaddr",
   "proto",
   "sport",
   "dport",
   "stos",
   "dtos",
   "sttl",
   "dttl",
   "bytes",
   "srcbytes",
   "dstbytes",
   "packets",
   "srcpackets",
   "dstpackets",
   "load",
};

int RaSortStartTime (struct ArgusRecord *, struct ArgusRecord *);
int RaSortLastTime (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDuration (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcAddr (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstAddr (struct ArgusRecord *, struct ArgusRecord *);
int RaSortProtocol (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcPort (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstPort (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcTos (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstTos (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcTtl (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstTtl (struct ArgusRecord *, struct ArgusRecord *);
int RaSortByteCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortLoad (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcByteCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstByteCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortPktsCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortSrcPktsCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortDstPktsCount (struct ArgusRecord *, struct ArgusRecord *);
int RaSortLoad (struct ArgusRecord *, struct ArgusRecord *);
 
int (*RaSortFunction)(struct ArgusRecord  *, struct ArgusRecord *) = RaSortByteCount;
int RaSortRoutine (const void *, const void *);

int (*RaSortAlgorithmTable[MAX_SORT_ALG_TYPES])(struct ArgusRecord *, struct ArgusRecord *) = {
   RaSortStartTime,
   RaSortLastTime,
   RaSortDuration,
   RaSortSrcAddr,
   RaSortDstAddr,
   RaSortProtocol,
   RaSortSrcPort,
   RaSortDstPort,
   RaSortSrcTos,
   RaSortDstTos,
   RaSortSrcTtl,
   RaSortDstTtl,
   RaSortByteCount,
   RaSortSrcByteCount,
   RaSortDstByteCount,
   RaSortPktsCount,
   RaSortSrcPktsCount,
   RaSortDstPktsCount,
   RaSortLoad
};

char *RaPolicyError [] = {
   "Malformed String",
   NULL,
};

char RaLabelStr[1024], *RaLabel = NULL;
char *RaDefaultFlowLabel = "     SrcAddr  Sport  Dir         DstAddr  Dport ";
char *RaMatrixFlowLabel  = "     SrcAddr         Dir         DstAddr        ";
char *RaTopNFlowLabel    = "     SrcAddr           ";
char *RacOptionLabel = " SrcPkt   Dstpkt    SrcBytes     DstBytes   ";
char *RAcOptionLabel = " SrcPkt   Dstpkt    SAppBytes    DAppBytes  ";
char *RaLoadOptionLabel = "      SrcApp bps      DstApp bps  ";
char *RamOptionLabel = "    SrcMacAddr       DstMacAddr       ";
char *RaROptionLabel = " SrcPkt   Dstpkt    Response Information    ";
char *RaTOptionLabel  = " Stos Dtos Sttl Dttl   SMinIdle    SMeanIdl    SMaxIdle     DMinIdle    DMeanIdl    DMaxIdle ";

int RaPolicyMatch (struct ArgusRecord *, struct RaPolicyStruct *);
int RaCIDRAddrMatches (u_int, struct RaCIDRAddr *);
int RaProtoMatches (u_short, int);
int RaPortMatches (u_short p1, u_short p2);
unsigned int RaIndexArgusRecord (struct ArgusRecord *, struct ArgusFarHeaderStruct **);
struct RaPolicyStruct *RaFlowModelOverRides (struct ArgusRecord *);
void RaModifyFlow(struct RaPolicyStruct *, struct ArgusRecord *);
void RaModifyDefaultFlow(struct ArgusRecord *);
struct RaHashTableHeader *RaFindHashObject (struct RaHashTableStruct *);
struct RaHashTableHeader *RaAddHashEntry (struct RaHashTableStruct *, struct ArgusRecordStore *);
void RaRemoveHashEntry (struct RaHashTableStruct *, struct RaHashTableHeader *);
struct RaQueueStruct *RaNewQueue (void);
void RaDeleteQueue (struct RaQueueStruct *);
int RaGetQueueCount(struct RaQueueStruct *);
void RaPushQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
int RaAddToQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
struct ArgusQueueHeader *RaPopQueue(struct RaQueueStruct *);
struct ArgusQueueHeader *RaRemoveFromQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
void RaEmptyQueue(struct RaQueueStruct *);
struct ArgusListStruct *ArgusNewList(void);
void ArgusDeleteList(struct ArgusListStruct *);
int ArgusListEmpty (struct ArgusListStruct *);
int ArgusGetListCount (struct ArgusListStruct *);
void *ArgusFrontList (struct ArgusListStruct *);
void ArgusPushFrontList(struct ArgusListStruct *, void *);
void ArgusPopFrontList (struct ArgusListStruct *);
void ArgusPushBackList (struct ArgusListStruct *, void *);
void ArgusPopBackList  (struct ArgusListStruct *);
int RaCheckTimeout(struct ArgusRecordStore *obj, struct ArgusRecord *);
void RaEmptyQueue(struct RaQueueStruct *);
struct ArgusRecordStore *RaFindArgusRecord (struct RaHashTableStruct *, struct ArgusRecord *);
struct ArgusRecordStore *RaFindRevArgusRecord(struct RaHashTableStruct *, struct ArgusRecord *);
struct ArgusRecordStore *RaNewArgusStore (struct ArgusRecord *);

void RaDeleteArgusStore(struct ArgusRecordStore *);
void RaDeleteArgusData(struct ArgusRecordData *);

struct ArgusRecordData *RaNewArgusData (struct ArgusRecord *);
void RaRemoveArgusRecord (struct ArgusRecord *);
void RaMergeArgusRecord (struct ArgusRecord *, struct ArgusRecordData *);
void RaUpdateArgusStore(struct ArgusRecord *, struct ArgusRecordStore *);
void RaTimeoutArgusStore (struct ArgusRecordStore *);
int RaCreatePolicyEntry (int, char *);
int RaReadFlowModelFile (char **);
struct RaPolicyStruct *RaParsePolicyEntry (char *);
struct RaCIDRAddr *RaParseCIDRAddr (char *);

struct ArgusRecordStore *RaThisArgusStore = NULL;
struct ArgusHoldingBuffer *RaHoldingBuffer = NULL;

extern struct ARGUS_INPUT *ArgusInput;
extern struct ArgusRecord argusOriginal;
extern int major_version, minor_version;

extern char *ArgusProgramName;
extern char *ArgusProgramOptions;

extern u_int ArgusThisFarStatus;
extern struct ArgusFarHeaderStruct *ArgusThisFarHdrs[];

extern int ArgusGrepSource, ArgusGrepDestination;

extern long long tcp_dst_bytes, tcp_src_bytes;
extern long long udp_dst_bytes, udp_src_bytes;
extern long long icmp_dst_bytes, icmp_src_bytes;
extern long long ip_dst_bytes, ip_src_bytes;

extern void ArgusDebug (int, char *, ...);

void ArgusClientInit(void);
void RaParseComplete (int);
void ArgusClientTimeout (void);
void parse_arg (int, char**);
void usage (void);

void RaProcessRecord (struct ArgusRecord *);

struct ArgusRecord * RaCopyArgusRecord (struct ArgusRecord *);
long long RaGetActiveDuration (struct ArgusRecord *);
long long RaGetuSecDuration (struct ArgusRecord *);

void process_man (struct ArgusRecord *);
void process_frag (struct ArgusRecord *);
void process_tcp (struct ArgusRecord *);
void process_icmp (struct ArgusRecord *);
void process_udp (struct ArgusRecord *);
void process_ip (struct ArgusRecord *);
void process_arp (struct ArgusRecord *);
void process_non_ip (struct ArgusRecord *);

extern char *get_man_string (struct ArgusRecord *);
extern char *get_tcp_string (struct ArgusRecord *);
extern char *get_ip_string (struct ArgusRecord *);
extern char *get_arp_string (struct ArgusRecord *);
extern char *get_icmp_string (struct ArgusRecord *);
extern char *get_nonip_string (struct ArgusRecord *);

extern char *getname(u_char *);
extern char *ArgusGetError (void);
extern void ArgusLog (int, char *, ...);
extern void RaSendArgusRecord(struct ArgusRecordStore *);

char *RaGenerateLabel(struct ArgusRecord *);

#else

extern int RaPrintDate;
extern int RaPrintTime;
extern int RaPrintDuration;
extern int RaPrintIndicator;
extern int RaPrintTransactions;
extern int RaPrintMACAddress;
extern int RaPrintProto;
extern int RaPrintSrcAddr;
extern int RaPrintSrcPort;
extern int RaPrintDir;
extern int RaPrintDstAddr;
extern int RaPrintDstPort;
extern int RaPrintCounts;
extern int RaPrintSrcPackets;
extern int RaPrintDstPackets;
extern int RaPrintBytes;
extern int RaPrintSrcBytes;
extern int RaPrintDstBytes;
extern int RaPrintJitter;
extern int RaPrintLoad;
extern int RaPrintUserData;

extern char *appOptstring;
extern int RaPrintStatus;

extern int RaCumulativeMerge;

extern int RaFlowModelFile;
extern int RaAllocHashTableHeaders;
extern int RaAllocArgusRecordStore;
extern int RaAllocArgusRecord;

extern int RaHistoTimeSeries;

extern struct ArgusFlow RaArgusFlowBuf, *RaArgusFlow;
extern long long RaThisActiveDuration;
extern int RaThisActiveIndex;

extern struct RaCIDRAddr RaCIDRBuffer, *RaCIDRPtr;

extern int RaThisFlowNum;
extern int RaThisModelNum;
extern int RaParseError;

extern struct RaHashTableStruct RaHashTable;

extern struct RaPolicyStruct  *RaFlowArray[1024];
extern struct RaPolicyStruct *RaModelArray[1024];

extern struct RaPolicyStruct *RaFlowModelPolicy;

extern struct RaQueueStruct *RaModelerQueue;
extern struct RaQueueStruct *RaAvailableQueue;
extern struct RaQueueStruct *RaFailedQueue;

extern int RaSortIndex;
extern void RaSortQueue (struct RaQueueStruct *);
extern char *RaSortAlgorithmStrings[ARGUS_MAX_SORT_ALG];
extern int (*RaSortAlgorithm)(struct ArgusRecord *, struct ArgusRecord *);
extern int (*RaSortAlgorithms[ARGUS_MAX_SORT_ALG])(struct ArgusRecord *, struct ArgusRecord *);

extern char *RaSortAlgorithmStrings[];
extern char *RaSortKeyWords[MAX_SORT_ALG_TYPES];

extern char *ArgusProgramName;
extern char *ArgusProgramOptions;

extern void ArgusDebug (int, char *, ...);

extern void ArgusClientInit(void);
extern void RaParseComplete (int);
extern void ArgusClientTimeout (void);
extern void parse_arg (int, char**);
extern void usage (void);

extern struct ArgusRecord * RaCopyArgusRecord (struct ArgusRecord *);
extern long long RaGetActiveDuration (struct ArgusRecord *);
extern long long RaGetuSecDuration (struct ArgusRecord *);

extern struct ArgusRecordStore *RaThisArgusStore;
extern struct ArgusHoldingBuffer *RaHoldingBuffer;

extern char *RaPolicyError[];

extern char RaLabelStr[], *RaLabel;

extern int RaPolicyMatch (struct ArgusRecord *, struct RaPolicyStruct *);
extern int RaCIDRAddrMatches (u_int, struct RaCIDRAddr *);
extern int RaProtoMatches (u_short, int);
extern int RaPortMatches (u_short p1, u_short p2);
extern unsigned int RaIndexArgusRecord (struct ArgusRecord *, struct ArgusFarHeaderStruct **);
extern struct RaPolicyStruct *RaFlowModelOverRides (struct ArgusRecord *);
extern void RaModifyFlow(struct RaPolicyStruct *, struct ArgusRecord *);
extern void RaModifyDefaultFlow(struct ArgusRecord *);
extern struct RaHashTableHeader *RaFindHashObject (struct RaHashTableStruct *);
extern struct RaHashTableHeader *RaAddHashEntry (struct RaHashTableStruct *, struct ArgusRecordStore *);
extern void RaRemoveHashEntry (struct RaHashTableStruct *, struct RaHashTableHeader *);
extern struct RaQueueStruct *RaNewQueue (void);
extern void RaDeleteQueue (struct RaQueueStruct *);
extern int RaGetQueueCount(struct RaQueueStruct *);
extern void RaPushQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
extern int RaAddToQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
extern struct ArgusQueueHeader *RaPopQueue(struct RaQueueStruct *);
extern struct ArgusQueueHeader *RaRemoveFromQueue(struct RaQueueStruct *, struct ArgusQueueHeader *);
extern void RaEmptyQueue(struct RaQueueStruct *);
extern struct ArgusListStruct *ArgusNewList(void);
extern void ArgusDeleteList(struct ArgusListStruct *);
extern int ArgusListEmpty (struct ArgusListStruct *);
extern int ArgusGetListCount (struct ArgusListStruct *);
extern void *ArgusFrontList (struct ArgusListStruct *);
extern void ArgusPushFrontList(struct ArgusListStruct *, void *);
extern void ArgusPopFrontList (struct ArgusListStruct *);
extern void ArgusPushBackList (struct ArgusListStruct *, void *);
extern void ArgusPopBackList  (struct ArgusListStruct *);
extern int RaCheckTimeout(struct ArgusRecordStore *obj, struct ArgusRecord *);
extern void RaEmptyQueue(struct RaQueueStruct *);
extern struct ArgusRecordStore *RaFindArgusRecord (struct RaHashTableStruct *, struct ArgusRecord *);
extern struct ArgusRecordStore *RaFindRevArgusRecord(struct RaHashTableStruct *, struct ArgusRecord *);
extern struct ArgusRecordStore *RaNewArgusStore (struct ArgusRecord *);
extern void RaDeleteArgusStore(struct ArgusRecordStore *);
extern struct ArgusRecordData *RaNewArgusData (struct ArgusRecord *);
extern void RaRemoveArgusRecord (struct ArgusRecord *);
extern void RaMergeArgusRecord (struct ArgusRecord *, struct ArgusRecordData *);
extern void RaUpdateArgusStore(struct ArgusRecord *, struct ArgusRecordStore *);
extern void RaTimeoutArgusStore (struct ArgusRecordStore *);
extern int RaCreatePolicyEntry (int, char *);
extern int RaReadFlowModelFile (char **);
extern struct RaPolicyStruct *RaParsePolicyEntry (char *);
extern struct RaCIDRAddr *RaParseCIDRAddr (char *);

extern void RaProcessRecord (struct ArgusRecord *);
extern void process_man (struct ArgusRecord *);
extern void process_frag (struct ArgusRecord *);
extern void process_tcp (struct ArgusRecord *);
extern void process_icmp (struct ArgusRecord *);
extern void process_udp (struct ArgusRecord *);
extern void process_ip (struct ArgusRecord *);
extern void process_arp (struct ArgusRecord *);
extern void process_non_ip (struct ArgusRecord *);

extern char *getname(u_char *);
extern char *ArgusGetError (void);
extern void ArgusLog (int, char *, ...);

extern int RaSortStartTime (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortLastTime (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDuration (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcAddr (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstAddr (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortProtocol (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcPort (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstPort (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcTos (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstTos (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcTtl (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstTtl (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortByteCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortLoad (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcByteCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstByteCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortPktsCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortSrcPktsCount (struct ArgusRecord *, struct ArgusRecord *);
extern int RaSortDstPktsCount (struct ArgusRecord *, struct ArgusRecord *);

extern int (*RaSortFunction)(struct ArgusRecord  *, struct ArgusRecord *);
extern int RaSortRoutine (const void *, const void *);
extern int (*RaSortAlgorithmTable[MAX_SORT_ALG_TYPES])(struct ArgusRecord *, struct ArgusRecord *);

extern char *RaGenerateLabel(struct ArgusRecord *);

void RaSendArgusRecord(struct ArgusRecordStore *);
#endif




