/* $Id: ArkGraph.cpp,v 1.12 2003/02/04 20:18:14 zongo Exp $
** 
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2000 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <Ark/ArkSystem.h>
#include <Ark/ArkGraph.h>

#include <algorithm>
#include <math.h>


namespace Ark
{

   Graph::Graph() 
   {
   }

   Graph::~Graph(){}

   void
   Graph::Read (const String &file)
   {
      AutoReadStream ars(file);
      Lexer lexer(file, ars.Get());

      Read(lexer);
   }

   void
   Graph::Read (Lexer &lexer)
   {
      lexer.CheckToken("Positions");
      lexer.CheckToken("=");
      lexer.CheckToken("{");

      while(1)
      {
	 if(lexer.GetToken() == "}")
	    break;
	 
	 lexer.UngetToken();
	    
	 Vector3 v;
	 if(!lexer.ReadScalarVector(&v.X, 3))
	 {
	    m_Positions.resize(0);
	    return;
	 }
	 m_Positions.push_back(v);
      }

      size_t numnodes = m_Positions.size();

      lexer.CheckToken("}");
      lexer.CheckToken(";");

      lexer.CheckToken("Edges");
      lexer.CheckToken("=");
      lexer.CheckToken("{");

      m_Edges.resize(numnodes);
      for (size_t i = 0; i < numnodes; ++i)
      {
	 lexer.CheckToken("{");
	 m_Edges[i].resize(0);

	 while (lexer.GetToken() != "}")
	 {  
	    lexer.UngetToken();

	    int tmp = 0;
	    if (!lexer.ReadInteger(&tmp) || tmp > (int)numnodes)
	    {
	       lexer.Error("bad node identifier.");
	       return;
	    }
	    
	    Edge ed = {tmp,
		       (unsigned short)
		       ((m_Positions[i]-m_Positions[tmp]).
			GetMagnitude()*100.0f)};
	    m_Edges[i].push_back(ed);

	    String tok = lexer.GetToken(Lexer::SYMBOL);	 
	    if(tok == "}") break;
	    else if (tok == ",") continue;
	    else
	    {
	       lexer.Error("unexpected token");
	       return;
	    }
	 }
      }

      m_MinCostPerMeter = 100;

      lexer.CheckToken("}");

      // Test consistency
      for (size_t i = 0; i < numnodes; ++i)
      {
	 for (size_t j = 0; j < m_Edges[i].size(); ++j)
	 {
	    NodeId suc = m_Edges[i][j].m_Succ;
	    bool found = false;

	    for (size_t k = 0; k < m_Edges[suc].size(); ++k)
	    {
	       if(m_Edges[suc][k].m_Succ == i)
	       {
		  found = true;
		  break;
	       }
	    }

	    if (!found)
	       Sys()->Warning("In '%s': Inconsistent graph. '%d' has an arrow "
                              "to '%d', but the countrary isn't true\n",
			      lexer.Name().c_str(), i, suc);
	 }
      }
      
   }

   void
   Graph::GetSuccessors(NodeId a, EdgeList const ** succ) const
   {
      assert (a < m_Edges.size());
      assert (succ != NULL);
      *succ = &(m_Edges[a]);
   }

   Graph::Cost
   Graph::EstimateCost(NodeId a, NodeId b) const
   {
      Vector3 v1 = m_Positions[a], v2 = m_Positions[b];
      return m_MinCostPerMeter *
               short(fabs(v1.X - v2.X)+ fabs(v1.Y - v2.Y) + fabs(v1.Z - v2.Z));
   }

//////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////
//////////////////////////////////////////////////////////////////////


   static Pathfinder::Comp comp;

   Pathfinder::Pathfinder(const Graph *graph)
   {
      m_Graph = graph;
      m_Mark.resize(m_Graph->m_Positions.size());
   }

   Pathfinder::~Pathfinder() {}


   // I'm using a priority queue implemented as a heap.  STL has some nice
   // heap manipulation functions.  (Look at the source to `priority_queue'
   // for details.)  I didn't use priority_queue because later on, I need
   // to traverse the entire data structure to update certain elements; the
   // abstraction layer on priority_queue wouldn't let me do that.
   static inline void get_first(Pathfinder::PFNodes &v, 
				Pathfinder::PFNode &n)
   {
      n = v.front();
      pop_heap( v.begin(), v.end(), comp );
      v.pop_back();
   }

   Pathfinder::PFNodes::iterator
   Pathfinder::find_in_open(Graph::NodeId n)
   {
      // Only search for this node if we know it's in the OPEN set
      if(in_open(n) ) 
      {
	 for(PFNodes::iterator i = m_Open.begin(); i != m_Open.end(); ++i )
	 {
	   if( (*i).m_Node == n )
	      return i;
	 }
      }

      return m_Open.end();
   }

   void
   Pathfinder::FindPath(Graph::NodeId a, Graph::NodeId b,
			Graph::Path &path)
   {
      m_EndNode = b;

      PFNode N;
      N.m_Node = a;
      N.m_g = 0;
      N.m_h = m_Graph->EstimateCost(a,b);
      m_Open.push_back(N);
      m_Mark[a].m_f = N.m_g + N.m_h;
      m_Mark[a].m_g = N.m_g;

      
      // * Things in OPEN are in the open container (which is a heap),
      //   and also their mark[...].f value is nonnegative.
      // * Things in CLOSED are in the visited container (which is unordered),
      //   and also their mark[...].direction value is not DirNone.
      
      // While there are still nodes to visit, visit them!
      while( !m_Open.empty() )
      {
	 get_first (m_Open, N);
	 m_Mark[N.m_Node].m_f = -1;
	 m_Visited.push_back(N);
	 
	 // If we're at the goal, then exit
	 if (N.m_Node == b)
            break;

	 // FIXME: Test to see if we're locked
	 
	 const Graph::EdgeList *edg;
	 m_Graph->GetSuccessors(N.m_Node, &edg);

	 Graph::EdgeList::const_iterator it;
	 for(it = edg->begin(); it != edg->end(); ++it)
	 {
	    PFNode N2;
	    Graph::NodeId hn = it->m_Succ;
	    N2.m_Node = hn;
	    N2.m_g = N.m_g + it->m_Cost;
	    N2.m_h = m_Graph->EstimateCost(N2.m_Node, b);


	     // If this spot (hn) hasn't been visited, its mark is DirNone
            if(m_Mark[hn].m_Previous == NULL_NODE)
            {
                // The space is not marked
                m_Mark[hn].m_Previous = N.m_Node;
                m_Mark[hn].m_f = N2.m_g+N2.m_h;
                m_Mark[hn].m_g = N2.m_g;

		m_Open.push_back(N2);
		push_heap(m_Open.begin(), m_Open.end(), comp);
            }
	    else
	    {
	       // We know it's in OPEN or CLOSED...
	       if(in_open(N2.m_Node))
	       {
		  // It's in OPEN, so figure out whether g is better
		  if( N2.m_g < m_Mark[hn].m_g )
		  {
		     // Search for hn in open
		     PFNodes::iterator find1 = find_in_open( hn );
		     assert(find1 != m_Open.end());
		     
		     // Replace *find1's gval with N2.gval in the list&map
		     m_Mark[hn].m_Previous = N.m_Node;
		     m_Mark[hn].m_g = N2.m_g;
		     m_Mark[hn].m_f = N2.m_g+N2.m_h;
		     (*find1).m_g = N2.m_g;
		  
		     push_heap(m_Open.begin(), find1+1, comp);
		     
		  // This next step is not needed for most games FIXME: is it ?
		     // propagate_down( *find1 );
		  }
	       }
	    }	    
	 }
      }

      Graph::Path rpath;
      Graph::NodeId cur = b;

      while (cur != a && cur != NULL_NODE)
      {
	 rpath.push_back(cur);

	 if (cur != NULL_NODE)
	    cur = m_Mark[cur].m_Previous;
      }
      rpath.push_back(a);

      Graph::Path::reverse_iterator it;
      path.resize(0);
      for (it = rpath.rbegin(); it != rpath.rend(); it++)
	 path.push_back(*it);

      // Erase the mark array, for all items in open or visited
      for(PFNodes::iterator o = m_Open.begin(); o != m_Open.end(); ++o )
	 m_Mark[(*o).m_Node] = Marking();

      for(PFNodes::iterator v = m_Visited.begin(); v != m_Visited.end(); ++v )
	 m_Mark[(*v).m_Node] = Marking();

      m_Open.resize(0);
   }

}
