/* $Id: ArkVariable.h,v 1.3 2002/10/11 01:10:06 zongo Exp $
**
** Ark - Libraries, Tools & Programs for MMORPG developpements.
** Copyright (C) 1999-2002 The Contributors of the Ark Project
** Please see the file "AUTHORS" for a list of contributors
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#ifndef ARK_VARIABLE_H
#define ARK_VARIABLE_H


#include <Ark/Ark.h>
#include <Ark/ArkTypes.h>
#include <Ark/ArkString.h>


namespace Ark
{

   //  =======================================================================
   /** The Variable class has been designed to hold variables from config
    * files. It does have a name (??), and a value which can be an integer, a scalar
    * (floating-point) or a string.
    */
   //  =======================================================================
   class ARK_DLL_API Variable
   {
      public:
	 virtual ~Variable() {}
	 
	 /**
	  * Returns the value of the variable as an integer.
	  */
	 virtual int GetAsInteger() const = 0;
	 
	 /**
	  * Returns the value of the variable as an scalar (floating point value).
	  */
	 virtual scalar GetAsScalar() const = 0;
	 
	 /**
	  * Returns the value of the variable as a string.
	  */
	 virtual String GetAsString() const = 0;

	 /**
	  * Returns the value of the variable as a writable string
	  * (ie if it's a string, it's Quote()d before it's returned).
	  */
	 virtual String GetWritableString() const = 0;
   };

   class IntegerVariable : public Variable
   {
       private:
	   int m_Value;
	   
      public:
	 /// Creates a new empty variable.
	 IntegerVariable(int value) : m_Value(value) {}
	 
	 /**
	  * Returns the value of the variable as an integer.
	  */
	 virtual int GetAsInteger() const { return m_Value; }
	 
	 /**
	  * Returns the value of the variable as an scalar (floating point value).
	  */
	 virtual scalar GetAsScalar() const { return static_cast< scalar >( m_Value ); }
	 
	 /**
	  * Returns the value of the variable as a string.
	  */
	 virtual String GetAsString() const;

	 /**
	  * Returns the value of the variable as a writable string
	  * (ie if it's a string, it's Quote()d before it's returned).
	  */
	 virtual String GetWritableString() const
	 {
	    return GetAsString();
	 }
	 
   };

   class ARK_DLL_API ScalarVariable : public Variable
   {
       private:
	   scalar m_Value;
	   
      public:
	 /// Creates a new empty variable.
	 ScalarVariable(scalar value) : m_Value(value) {}
	 
	 /**
	  * Returns the value of the variable as an integer.
	  */
	 virtual int GetAsInteger() const { return static_cast< int >( m_Value ); }
	 
	 /**
	  * Returns the value of the variable as an scalar (floating point value).
	  */
	 virtual scalar GetAsScalar() const { return m_Value; }
	 
	 /**
	  * Returns the value of the variable as a string.
	  */
	 virtual String GetAsString() const;
	 
         /**
	  * Returns the value of the variable as a writable string
	  * (ie if it's a string, it's Quote()d before it's returned).
	  */
	 virtual String GetWritableString() const
	 {
	    return GetAsString();
	 }
	 
   };

   class ARK_DLL_API StringVariable : public Variable
   {
       private:
	   String m_Value;
	   
      public:
	 /// Creates a new empty variable.
	 StringVariable(const String& value) : m_Value(value) {}
	 
	 /**
	  * Returns the value of the variable as an integer.
	  */
	 virtual int GetAsInteger() const;
	 
	 /**
	  * Returns the value of the variable as an scalar (floating point value).
	  */
	 virtual scalar GetAsScalar() const;
	 
	 /**
	  * Returns the value of the variable as a string.
	  */
	 virtual String GetAsString() const { return m_Value; }
	 
	 /**
	  * Returns the value of the variable as a string.
	  *
	  * This version is faster, requiring only to return the result by reference.
	  */
	 const String& GetString() const { return m_Value; }

         /**
	  * Returns the value of the variable as a writable string
	  * (ie if it's a string, it's Quote()d before it's returned).
	  */
	 virtual String GetWritableString() const
	 {
	    return QuoteString(GetString());
	 }
	 
	 
   };

// namespace Ark
}

#endif
