/*
 * Copyright (c) 1995 - 2002, 2004 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#include <nnpfs/nnpfs_locl.h>
#include <nnpfs/nnpfs_common.h>
#include <nnpfs/nnpfs_fs.h>
#include <nnpfs/nnpfs_deb.h>

RCSID("$Id: nnpfs_node-osf.c,v 1.32 2004/02/15 21:36:55 lha Exp $");

extern struct vnodeops *nnpfs_vnodeop_p;
extern struct vfs_ubcops nnpfs_ubcops;

/*
 * Create a new nnpfs_node and make a VN_HOLD()!
 *
 * Also prevents creation of duplicates. This happens
 * whenever there are more than one name to a file,
 * "." and ".." are common cases.
 */

int
nnpfs_new_node(struct nnpfs *nnpfsp,
	     struct nnpfs_msg_node *node,
	     struct nnpfs_node **xpp,
	     struct proc *p)
{
    int do_vget = 0;
    int do_ins = 0;
    struct nnpfs_node *result;

    NNPFSDEB(XDEBNODE, ("nnpfs_new_node %d.%d.%d.%d\n",
		      node->handle.a,
		      node->handle.b,
		      node->handle.c,
		      node->handle.d));

    /* Does not allow duplicates */
    result = nnpfs_node_find(&nnpfsp->nodehead, &node->handle);

    if (result == 0) {
	int error;
	struct vnode *v;

	error = getnewvnode(VT_AFS, &nnpfs_vnodeops, &v);
	    NNPFSDEB(XDEBVNOPS,
		   ("nnpfs_new_node: v = %lx v->v_usecount = %d\n", 
		    (unsigned long)v, v?v->v_usecount:0));

	if (error) {
	    NNPFSDEB(XDEBVNOPS,
		   ("NNPFS PANIC! nnpfs_new_node: could not allocate node"));
	    return error;
	}
	do_ins++; /* do insmntque below */
	result = VNODE_TO_XNODE(v);
	bzero(result, sizeof(*result));
	result->vn = v;

	result->anonrights = node->anonrights;

	result->handle = node->handle;
	result->flags = 0;
	result->tokens = 0;
	result->offset = 0;

	nnpfsp->nnodes++;
    } else {
	/* Node is already cached */
	do_vget = 1;
    }

    /* Init other fields */
    nnpfs_attr2vattr(&node->attr, &result->attr, 1);
    result->vn->v_type = result->attr.va_type;
    NNPFS_TOKEN_SET(result, NNPFS_ATTR_R, NNPFS_ATTR_MASK);
    bcopy(node->id, result->id, sizeof(result->id));
    bcopy(node->rights, result->rights, sizeof(result->rights));

    if(do_ins)
	insmntque(result->vn, NNPFS_TO_VFS(nnpfsp), &nnpfs_ubcops); /* not done by 
								 getnewvnode */
    /*
     * We need to postpone this until here because (on FreeBSD) vget
     * tries to install a pager on the vnode and for that it wants to
     * retrieve the size with getattr.
     */

    if(do_vget) {
	vget(XNODE_TO_VNODE(result)); /* XXX unlock? */
    }
    *xpp = result;
    NNPFSDEB(XDEBNODE, ("return: nnpfs_new_node\n"));
    return 0;
}

void
nnpfs_free_node(struct nnpfs *nnpfsp, struct nnpfs_node *node)
{
    NNPFSDEB(XDEBNODE, ("nnpfs_free_node starting: node = %lx\n",
		      (unsigned long)node));

    NNPQUEUE_REMOVE(node, &nnpfsp->freehead, nn_free);
    nnpfs_remove_node(&nnpfsp->nodehead, node);

    if (DATA_FROM_XNODE(node)) {
	vrele(DATA_FROM_XNODE(node));
	DATA_FROM_XNODE(node) = NULL;
    }
    nnpfsp->nnodes--;

    NNPFSDEB(XDEBNODE, ("nnpfs_free_node done\n"));
}

int
nnpfs_free_all_nodes(struct nnpfs *nnpfsp, int flags, int unmountp)
{
    int error = 0;
    struct mount *mp = NNPFS_TO_VFS(nnpfsp);

    if (mp == NULL) {
	NNPFSDEB(XDEBNODE, ("nnpfs_free_all_nodes already freed\n"));
	return 0;
    }

    NNPFSDEB(XDEBNODE, ("nnpfs_free_all_nodes starting\n"));

    nnpfs_dnlc_purge_mp(mp);

    NNPFSDEB(XDEBNODE, ("nnpfs_free_all_nodes now removing root\n"));

    if (nnpfsp->root) {
	struct vnode *vp = XNODE_TO_VNODE(nnpfsp->root);
	NNPFSDEB(XDEBNODE, ("root->v_usecount = %d\n", vp->v_usecount));
	VN_LOCK(vp);
	vgone(vp, VX_NOSLEEP, NULL);
	VN_UNLOCK(vp);
	vrele(vp);
	nnpfsp->root = 0;
    }

    NNPFSDEB(XDEBNODE, ("nnpfs_free_all_nodes root removed\n"));

    NNPFSDEB(XDEBNODE, ("nnpfs_free_all_nodes now killing all "
			"remaining nodes\n"));

    error = vflush(mp, NULL, flags);

    if (error) {
	NNPFSDEB(XDEBNODE, ("nnpfs_free_all_nodes: vflush() error == %d\n",
			    error));
	return error;
    }

    NNPFSDEB(XDEBNODE, ("nnpfs_free_all_nodes done\n"));
    return error;
}

void
vattr2nnpfs_attr(const struct vattr *va, struct nnpfs_attr *xa)
{
    bzero(xa, sizeof(*xa));
    if (va->va_mode != (u_short)VNOVAL)
	XA_SET_MODE(xa, va->va_mode);
    if (va->va_nlink != VNOVAL)
	XA_SET_NLINK(xa, va->va_nlink);
    if (va->va_size != VNOVAL)
	XA_SET_SIZE(xa, va->va_size);
    if (va->va_uid != (uid_t)VNOVAL)
	XA_SET_UID(xa, va->va_uid);
    if (va->va_gid != (gid_t)VNOVAL)
	XA_SET_GID(xa, va->va_gid);
    if (va->va_atime.tv_sec != VNOVAL)
	XA_SET_ATIME(xa, va->va_atime.tv_sec);
    if (va->va_mtime.tv_sec != VNOVAL)
	XA_SET_MTIME(xa, va->va_mtime.tv_sec);
    if (va->va_ctime.tv_sec != VNOVAL)
	XA_SET_CTIME(xa, va->va_ctime.tv_sec);
    if (va->va_fileid != VNOVAL)
	XA_SET_FILEID(xa, va->va_fileid);
    switch (va->va_type) {
    case VNON:
	xa->xa_type = NNPFS_FILE_NON;
	break;
    case VREG:
	xa->xa_type = NNPFS_FILE_REG;
	break;
    case VDIR:
	xa->xa_type = NNPFS_FILE_DIR;
	break;
    case VBLK:
	xa->xa_type = NNPFS_FILE_BLK;
	break;
    case VCHR:
	xa->xa_type = NNPFS_FILE_CHR;
	break;
    case VLNK:
	xa->xa_type = NNPFS_FILE_LNK;
	break;
    case VSOCK:
	xa->xa_type = NNPFS_FILE_SOCK;
	break;
    case VFIFO:
	xa->xa_type = NNPFS_FILE_FIFO;
	break;
    case VBAD:
	xa->xa_type = NNPFS_FILE_BAD;
	break;
    default:
	panic("nnpfs_attr2attr: bad value");
    }
}

#define SET_TIMEVAL(X, S, N) do { (X)->tv_sec = (S); (X)->tv_usec = (N) / 1000; } while(0)

void
nnpfs_attr2vattr(const struct nnpfs_attr *xa, struct vattr *va, int clear_node)
{
    if (clear_node)
	VATTR_NULL(va);
    if (XA_VALID_MODE(xa))
	va->va_mode = xa->xa_mode;
    if (XA_VALID_NLINK(xa))
	va->va_nlink = xa->xa_nlink;
    if (XA_VALID_SIZE(xa)) {
	va->va_size = xa->xa_size;
	va->va_bytes = va->va_size;
    }
    if (XA_VALID_UID(xa))
	va->va_uid = xa->xa_uid;
    if (XA_VALID_GID(xa))
	va->va_gid = xa->xa_gid;
    if (XA_VALID_ATIME(xa)) {
	SET_TIMEVAL(&va->va_atime, xa->xa_atime, 0);
    }
    if (XA_VALID_MTIME(xa)) {
	SET_TIMEVAL(&va->va_mtime, xa->xa_mtime, 0);
    }
    if (XA_VALID_CTIME(xa)) {
	SET_TIMEVAL(&va->va_ctime, xa->xa_ctime, 0);
    }
    if (XA_VALID_FILEID(xa)) {
	va->va_fileid = xa->xa_fileid;
    }
    if (XA_VALID_TYPE(xa)) {
	switch (xa->xa_type) {
	case NNPFS_FILE_NON:
	    va->va_type = VNON;
	    break;
	case NNPFS_FILE_REG:
	    va->va_type = VREG;
	    break;
	case NNPFS_FILE_DIR:
	    va->va_type = VDIR;
	    break;
	case NNPFS_FILE_BLK:
	    va->va_type = VBLK;
	    break;
	case NNPFS_FILE_CHR:
	    va->va_type = VCHR;
	    break;
	case NNPFS_FILE_LNK:
	    va->va_type = VLNK;
	    break;
	case NNPFS_FILE_SOCK:
	    va->va_type = VSOCK;
	    break;
	case NNPFS_FILE_FIFO:
	    va->va_type = VFIFO;
	    break;
	case NNPFS_FILE_BAD:
	    va->va_type = VBAD;
	    break;
	default:
	    panic("nnpfs_attr2vattr: bad value");
	}
    }
    va->va_flags = 0;
    va->va_blocksize = 8192;
}

/*
 * A single entry DNLC for systems for handling long names that don't
 * get put into the system DNLC.
 */

struct long_entry {
    struct vnode *dvp, *vp;
    char name[MAXNAMLEN + 1];
    size_t len;
    u_long dvpid, vpid;
};

static struct long_entry tbl;

/*
 * Nuke the `tbl'
 */

static void
tbl_clear (void)
{
    tbl.dvp = tbl.vp = NULL;
    tbl.name[0] = '\0';
    tbl.len = 0;
    tbl.dvpid = tbl.vpid = 0;
}

/*
 * Set the entry in the `tbl'
 */

static void
tbl_enter (size_t len, const char *name, struct vnode *dvp, struct vnode *vp)
{
    tbl.len = len;
    bcopy(name, tbl.name, len);
    tbl.dvp = dvp;
    tbl.vp = vp;
    tbl.dvpid = dvp->v_id;
    tbl.vpid = vp->v_id;
}

/*
 * Lookup in tbl
 */

static int
tbl_lookup (nnpfs_componentname *cnp, struct vnode *dvp,
	    struct vnode **res)
{
    size_t len = cnp->cn_namelen;
    const char *name = cnp->cn_nameptr;

    if (tbl.dvp == dvp
	&& tbl.len == len
	&& strncmp(tbl.name, name, len) == 0
	&& tbl.dvpid == tbl.dvp->v_id
	&& tbl.vpid == tbl.vp->v_id) {
	*res = tbl.vp;
	CACHE_LOOKUP_REF(*res);
#if	UNIX_LOCKS
	cnp->ni_vpid = (*res)->v_id;
#endif
	return -1;
    } else
	return 0;
}

/*
 * Store a nameidata in the DNLC
 */

int
nnpfs_dnlc_enter(struct vnode *dvp,
	       nnpfs_componentname *ndp,
	       struct vnode *vp)
{
    NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_enter(%lx, %lx, %lx)\n",
		      (unsigned long)dvp,
		      (unsigned long)ndp,
		      (unsigned long)vp));
    NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_enter: v_id = %ld\n", dvp->v_id));

    NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_enter: calling cache_enter:"
		      "dvp = %lx, vp = %lx, ndp = (%s, %ld, %lu, %lx, %lx)\n",
		      (unsigned long)dvp,
		      (unsigned long)vp, ndp->ni_ptr, ndp->ni_namelen,
		      ndp->cn_hash, ndp->ni_ptr, ndp->ni_next));

    if (ndp->ni_namelen <= NCHNAMLEN)
    {
	/*
	 * This is to make sure there's no negative entry already in the dnlc
	 */
	short save_nameiop;
	struct vnode *dummy;
	int ret;

	ndp->ni_dvp = dvp;

	save_nameiop = ndp->ni_nameiop;
	ndp->ni_nameiop = CREATE;
	ndp->ni_makeentry = 1;

	if ((ret = cache_lookup(ndp)) != 0) {
	    printf ("NNPFS PANIC WARNING! nnpfs_dnlc_enter: "
		    "%s already in cache, ret = %d\n",
		    ndp->ni_ptr, ret);
	}

	if (ndp->ni_vp == NULL && vp != NULL)
	    panic ("nnpfs_dnlc_enter: bad result from cache_lookup");

	ndp->ni_nameiop = save_nameiop;

	NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_enter: entering cache_enter\n"));
	cache_enter(ndp);
	NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_enter: leaving cache_enter\n"));
    }

    if (vp != NULL)
	tbl_enter (ndp->ni_namelen, ndp->ni_ptr, dvp, vp);

    return 0;
}

static void
nnpfs_cnp_init (struct nameidata *ndp,
	      const char *name, struct vnode *vp, struct vnode *dvp,
	      struct proc *proc, struct ucred *cred,
	      int nameiop)
{
    int i;

    ndinit(ndp,NULL);

    ndp->ni_ptr = (char *)name;
    ndp->ni_namelen = strlen(name);
    ndp->ni_next = ndp->ni_ptr + ndp->ni_namelen;
    ndp->ni_vp = vp;
    ndp->ni_dvp = dvp;
    ndp->ni_hash = 0;

    if (ndp->ni_namelen <= NCHNAMLEN) {


#ifdef HAVE_FOUR_ARGUMENT_VFS_NAME_HASH
	vfs_name_hash(ndp->ni_ptr, ndp->ni_dent.d_name,
		      dvp, &ndp->ni_hash);
#elif HAVE_THREE_ARGUMENT_VFS_NAME_HASH
	vfs_name_hash(ndp->ni_ptr, ndp->ni_dent.d_name,
		      &ndp->ni_hash);
#else
	for (i = 0; i < ndp->ni_namelen; ++i)
	    ndp->ni_hash += name[i] * (i + 1);
#endif
    }

    ndp->ni_nameiop = nameiop;
    ndp->ni_cred = cred;
}

/*
 * Store a (dvp, name, vp) in the DNLC
 */

int
nnpfs_dnlc_enter_name(struct vnode *dvp,
		    const char *name,
		    struct vnode *vp)
{
    struct nameidata nd;

    NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_enter_name(%lx, \"%s\", %lx)\n",
		      (unsigned long)dvp, name,
		      (unsigned long)vp));

    nnpfs_cnp_init (&nd, name, vp, dvp, NULL, NULL, CREATE);
    return nnpfs_dnlc_enter (dvp, &nd, vp);
}

 /*
  * do the last (and locking protocol) portion of xnlc_lookup
  *
  * return:
  * -1 for succesful
  * 0  for failed
  */

static int
nnpfs_dnlc_lock(struct vnode *dvp,
	      nnpfs_componentname *cnp,
	      struct vnode **res)
{
    int error = 0, vpid;

#if	UNIX_LOCKS
    vpid = cnp->ni_vpid;
#else
    vpid = (*res)->v_id;
#endif

    if (dvp == *res) {		/* "." */
	VREF(dvp);
	abort_vget(dvp);
	if (vpid != (*res)->v_id) {
	    error = 1;
	    vrele(*res);
	}
    } else {
	VN_LOCK(*res);
	error = vget_cache_nowait(*res);
	if (!error) {
	    if (vpid != (*res)->v_id) {
		error = 1;
		VN_UNLOCK(*res);
		vrele(*res);
	    } else
		VN_UNLOCK(*res);
	} else
	    VN_UNLOCK(*res);
    }

    if(error) {
	*res = NULL;
	return 0;
    } else
	return -1;
}

/*
 * Lookup (dvp, cnp) in the DNLC and return the result in `res'.
 * Return -1 if succesful, 0 if not and ENOENT if we're sure it doesn't exist.
 */

int
nnpfs_dnlc_lookup(struct vnode *dvp,
		nnpfs_componentname *cnp,
		struct vnode **res)
{
    int error;

    NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_lookup(%lx, \"%s\")\n",
		      (unsigned long)dvp, cnp->cn_nameptr));
    
    NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_lookup: v_id = %ld\n", dvp->v_id));
    
    NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_lookup: calling cache_lookup:"
		      "dvp = %lx, cnp = (%s, %ld, %lu)\n",
		      (unsigned long)dvp,
		      cnp->cn_nameptr, cnp->cn_namelen, cnp->cn_hash));

    cnp->ni_makeentry = 1;

    cnp->ni_dvp = dvp;
    error = cache_lookup(cnp);

#if 0
    *res = cnp->ni_vp;
#endif

    NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_lookup: cache_lookup returned. "
		      "error = %d, *res = %lx res->v_usecount = %d\n", error,
		      (unsigned long)*res, *res?(*res)->v_usecount:0));

    if (error == 0)
	error = tbl_lookup (cnp, dvp, res);

    NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_lookup: tbl_lookup returned. "
		      "error = %d, *res = %lx res->v_usecount = %d\n", error,
		      (unsigned long)*res, *res?(*res)->v_usecount:0));


    if (error != -1) {
	*res = NULL;
	return error;
    }

    return nnpfs_dnlc_lock (dvp, cnp, res);
}

/*
 *
 */

int
nnpfs_dnlc_lookup_name (struct vnode *dvp,
		      const char *name,
		      struct vnode **res)
{
    struct nameidata nd;

    nnpfs_cnp_init (&nd, name, NULL, dvp, NULL, NULL, LOOKUP);
    return nnpfs_dnlc_lookup (dvp, &nd, res);
}


/*
 * Remove one entry from the DNLC
 */

void
nnpfs_dnlc_purge (struct vnode *vp)
{
    NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_purge\n"));

    if (tbl.dvp == vp || tbl.vp == vp)
	tbl_clear ();

    cache_purge(vp);
}

/*
 * Remove all entries belong to `mp' from the DNLC
 */

void
nnpfs_dnlc_purge_mp (struct mount *mp)
{
    NNPFSDEB(XDEBDNLC, ("nnpfs_dnlc_purge_mp()\n"));

    tbl_clear ();
#ifdef HAVE_KERNEL_CACHE_PURGEVFS
    cache_purgevfs(mp);
#else
    /* XXX */
#endif /* HAVE_KERNEL_CACHE_PURGEVFS */
}

/*
 * Returns 1 if pag has any rights set in the node
 */

int
nnpfs_has_pag(const struct nnpfs_node *xn, nnpfs_pag_t pag)
{
    int i;

    for (i = 0; i < MAXRIGHTS; i++)
	if (xn->id[i] == pag)
	    return 1;

    return 0;
}
