/*
 * Copyright (c) 1995 - 2004 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * Alternatively, this software may be distributed under the terms of the
 * GNU General Public License ("GPL").
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

#ifndef _nnpfs_h
#define _nnpfs_h

#include <nnpfs/nnpfs_common.h>
#include <nnpfs/nnpfs_node.h>
#include <linux/types.h>

/*
 * Queues of nnpfs_links hold outbound messages and processes sleeping
 * for replies. The last field is used to return error to sleepers and
 * to keep record of memory to be deallocated when messages have been
 * delivered or dropped.
 */
struct nnpfs_link {
    struct nnpfs_link *prev, *next;
    struct nnpfs_message_header *message;
    u_int error_or_size;	/* error on sleepq and size on messageq */
    u_int woken;
    wait_queue_head_t wait_queue;
};  

/*
 * Filesystem struct.
 */
struct nnpfs {
    u_int status;		/* Inited, opened or mounted */
#define NNPFS_MOUNTED	0x1
#define NNPFS_DEVOPEN	0x2
#define NNPFS_ROOTINSTALLED 0x4
    struct super_block *sb;
    struct nnpfs_node *root;
    u_int nnodes;
    
    struct semaphore node_sem;		/* protects node_list */
    struct nnpfs_nodelist_head node_head;

    struct semaphore inactive_sem;      /* protects inactive_list */
    struct list_head inactive_list;

    struct semaphore channel_sem; /* protects messageq, sleepq, nsequence */
    struct nnpfs_link messageq;	/* Messages not yet read */
    struct nnpfs_link sleepq;	/* Waiting for reply message */
    u_int nsequence;
    wait_queue_head_t wait_queue;
    struct nnpfs_message_header *message_buffer;
};

#ifdef LINUX2_5
#define VFS_TO_NNPFS(v)      ((struct nnpfs *) ((v)->s_fs_info))
#define VFS_SET_NNPFS(v, n)  ((v)->s_fs_info = (n))
#else
#define VFS_TO_NNPFS(v)      ((struct nnpfs *) ((v)->u.generic_sbp))
#define VFS_SET_NNPFS(v, n)  ((v)->u.generic_sbp = (n))
#endif

#define NNPFS_TO_VFS(x)      ((x)->sb)

#define NNPFS_FROM_VNODE(vp) VFS_TO_NNPFS((vp)->i_sb)
#define NNPFS_FROM_XNODE(xp) ((xp)->nnpfsp)

extern struct nnpfs nnpfs[];

extern struct vnodeops nnpfs_vnodeops;

int
nnpfs_node_find(struct nnpfs *nnpfsp, nnpfs_handle *handlep, 
		struct nnpfs_node **node);
int
new_nnpfs_node(struct nnpfs *nnpfsp, struct nnpfs_msg_node *node,
	       struct inode *inode, struct nnpfs_node **result);
void nnpfs_free_node (struct nnpfs_node *);
void clear_nnpfs_node (struct nnpfs_node *);
void free_all_nnpfs_nodes (struct nnpfs *nnpfsp);
void print_nodes (struct nnpfs *nnpfsp);

int nnpfs_has_pag(const struct nnpfs_node *xn, nnpfs_pag_t);

void nnpfs_invalid_xnode(struct nnpfs_node *xnode);
void nnpfs_force_invalid_xnode(struct nnpfs_node *xnode);
int  nnpfs_node_users(struct nnpfs_node *xnode);
void nnpfs_attr2inode(const struct nnpfs_attr *, struct inode *, int);
void vattr2nnpfs_attr(const struct iattr *iattr, struct nnpfs_attr *attr);
int nnpfs_setattr (struct dentry *inode, struct iattr *sb);
#endif /* _nnpfs_h */
