/* 
 * Copyright (c) 2008 Andrea Di Pasquale <spikey.it@gmail.com>
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice(s), this list of conditions and the following disclaimer as
 *    the first lines of this file unmodified other than the possible
 *    addition of one or more copyright notices.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice(s), this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER(S) ``AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH
 * DAMAGE.
 *
 * $ArpON: arpon.c,v 1.90 10/14/2008 14:18:42 Andrea Di Pasquale Exp $
 */

#include <sys/ioctl.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <sys/resource.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#ifndef LINUX
#include <net/bpf.h>
#endif
#include <net/if.h>
#ifdef LINUX
#include <net/if_arp.h>
#endif
#ifdef NETBSD
#include <net/if_ether.h>
#elif !defined OPENBSD
#include <net/ethernet.h>
#endif
#include <netinet/in.h>
#ifdef LINUX 
#include <netinet/ether.h>
#endif
#ifdef OPENBSD
#include <netinet/if_ether.h>
#endif

#include <pthread.h>
#include <pcap.h>
#ifdef DEBIAN
#include <dumbnet.h>
#else
#include <dnet.h>
#endif
#include <libnet.h>

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <stdarg.h>
#include <unistd.h>
#include <errno.h>
#include <time.h>
#include <getopt.h>
#include <fcntl.h>

#include "include/queue.h"

#define	CUR_VERSION	("1.90")
#define	WEB_SITE	("http://arpon.sourceforge.net")

#define	ROOT_USER       (0x0)

#if defined NETBSD || defined OPENBSD || defined LINUX
#define	octet	ether_addr_octet
#endif	/* struct ether_addr */

#define	MAC_ADDR_LEN		(0x12)
#define	MAC_ADDR_BROADCAST	("ff:ff:ff:ff:ff:ff")
#define	INET4_ADDR_LEN		(0x10)

#define	ERROR(str)	aprintf(stderr, "\n  ERROR: %s:%d @ %s.\n\n", __FILE__, __LINE__, str)
#define	PRINT_TIME(arg) aprintf(stdout, arg,							\
				tm_cur.tm_mon, tm_cur.tm_mday, tm_cur.tm_year + 1900,		\
				tm_cur.tm_hour, tm_cur.tm_min, tm_cur.tm_sec, tm_cur.tm_zone);	

enum device_t		{ DEVICE_MANUAL, DEVICE_AUTO, DEVICE_LIST };
enum netmask_class_t 	{ NETMASK_CLASS_A = 16777215, NETMASK_CLASS_B = 65535, NETMASK_CLASS_C = 255 };
enum ping_t 		{ PING_HOST, PING_BROADCAST }; 
enum ping_reply_t 	{ PING_REPLY_FALSE = -1, PING_REPLY_TRUE = 0 };
enum arp_cache_t	{ ARP_CACHE_ADD, ARP_CACHE_DEL, ARP_CACHE_LIST };
enum cache_t 		{ ARP_CACHE, SARPI, SARPI_REFRESH, DARPI };

struct device {
	struct ether_addr	dev_mac;
	struct in_addr 		dev_inet4;
	struct in_addr		dev_netmask;
	char 			dev[IF_NAMESIZE];
	int 			dev_offset;	/* offset for ethernet/wireless datalink */
};

struct arp_header {	
	struct arp_hdr		ah_header;	/* Arp header */
	struct arp_ethip 	ah_addresses;	/* Ethernet header */
};

struct arp_cache {	
	TAILQ_ENTRY(arp_cache) entries;
	struct ether_addr	ac_mac;
	struct in_addr		ac_ip;
};

struct darpi_cache {
	TAILQ_ENTRY(darpi_cache) entries;
	struct in_addr		dc_ip;	/* Entry Inet4 address */
};

static int	 task_mode_cpu_priority(int, int, int);
static int	 task_mode_daemon(void);
static FILE	*task_mode_pid_open(void);
static void	 task_mode_pid_close(FILE *);
static int	 task_mode_log(void);
static FILE 	*task_mode_log_open(void);
static void	 task_mode_log_close(FILE *);
static int	 device_manager(enum device_t, char *);
static void	 device_set_name(char *);
static int	 device_check_datalink(char *);
static int	 device_del_promisc(void);
static int	 device_get_mac_address(void);
static int	 device_get_inet4_addresses(enum device_t);
static void	 device_info_print(void);
static void 	 arp_ping_set_timeout(char *);
static int	 arp_ping_host(char *);
static int	 arp_ping_broadcast(void);
static pcap_t 	*arp_ping_open_packet(void);
static void	 arp_ping_close_packet(pcap_t *);
static int	 arp_ping_recv_packet(enum ping_t, pcap_t *, unsigned char *);
static void	 arp_ping_recv_packet_signal(int sig);
#ifdef LINUX
static void	*arp_ping_recv_packet_thread(void *arg);
#endif
static void	 arp_ping_read_packet(unsigned char *, const struct pcap_pkthdr *, const unsigned char *);
static int	 arp_ping_send_packet(enum ping_t, struct ether_addr *, struct in_addr *);
static int	 arp_sniff_packets(void);
static void	 arp_sniff_signal(int);
static void 	*arp_sniff_thread(void *arg);
static pcap_t 	*arp_sniff_open_packets(void);
static void	 arp_sniff_close_packets(pcap_t * pcap);
static void	 arp_sniff_read_packets(unsigned char *, const struct pcap_pkthdr *, const unsigned char *);
static int	 arp_cache_manager(enum arp_cache_t, char *);
static int	 arp_cache_add(enum cache_t, arp_t *, char *);
static int	 arp_cache_del(enum cache_t, arp_t *, char *);
static int	 arp_cache_list(arp_t *);
static int	 arp_cache_list_create(const struct arp_entry *, void *);
static void	 arp_cache_list_print(void);
static void	 arp_cache_list_destroy(void);
static void	 sarpi_set_timeout(char *);
static int	 sarpi_manager(void);
static arp_t 	*sarpi_arp_cache_open(void);
static void	 sarpi_arp_cache_close(arp_t *);
static void	 sarpi_arp_cache_list(void);
static void	*sarpi_arp_cache_refresh_thread(void *);
static int	 sarpi_realtime(arp_t *);
static void	 sarpi_realtime_signal(int);
static void	*sarpi_realtime_thread(void *arg);
static pcap_t 	*sarpi_realtime_open_packets(void);
static void	 sarpi_realtime_close_packets(pcap_t *);
static void	 sarpi_realtime_read_packets(unsigned char *, const struct pcap_pkthdr *, const unsigned char *);
static void	 darpi_set_timeout(char *);
static int	 darpi_manager(void);
static arp_t 	*darpi_arp_cache_open(void);
static void	 darpi_arp_cache_close(arp_t *);
static int	 darpi_arp_cache_delete_entries(void);
static int	 darpi_arp_cache_add_entry(struct in_addr *, struct ether_addr *);
static int	 darpi_arp_cache_del_entry(char *);
static int	 darpi_realtime();
static void	 darpi_realtime_signal(int);
static void	*darpi_realtime_thread(void *arg);
static pcap_t 	*darpi_realtime_open_packets(void);
static void	 darpi_realtime_close_packets(pcap_t *);
static void	 darpi_realtime_read_packets(unsigned char *, const struct pcap_pkthdr *, const unsigned char *);
static int	 darpi_realtime_send_packet(struct ether_addr *, struct in_addr *);
static int	 darpi_cache_list_create(struct in_addr *);
static void	*darpi_cache_list_create_check_thread(void *);
static int	 darpi_cache_list_search(char *, struct ether_addr *, struct in_addr *);
static void	 darpi_cache_list_destroy(void);
static void	 aprintf(FILE *, char *, ...);
static void	 license(void);
static void	 version(void);
static void	 help(void);
static void	 main_signal(int);

static enum ping_t ping_type;	/* Ping to host or broadcast */
static enum ping_reply_t ping_check_reply = PING_REPLY_FALSE;	/* Check Arp reply if it's true or false */

static struct device dev;
static struct arp_header *arp_packet;
static struct arp_cache *arp_cache_begin, *arp_cache_next, *arp_cache_pos;
static TAILQ_HEAD(, arp_cache) arp_cache_head;
static struct darpi_cache *darpi_cache_begin, *darpi_cache_next, *darpi_cache_pos;
static TAILQ_HEAD(, darpi_cache) darpi_cache_head;

static pcap_t *pcap_sniff;
static pid_t pid_main;
static pthread_t thread[2];	/* thread handler */
static pthread_attr_t join_attr, detached_attr;
static pthread_rwlock_t rlock, wlock;

#ifdef LINUX
static char ping_host[INET4_ADDR_LEN];
#endif
static char *pid_file = "/var/run/arpon.pid";
static char *log_file = "/var/log/arpon.log";

static int cpu_priority = 0;		/* Nice value for CPU priority */
static int log_mode = -1;		/* Log mode off */
static int ping_timeout = 500;		/* Arp ping timeout, default 500 ms */
static int ping_broadcast_count = 1;	/* Active hosts count */
static int arp_sniff_request_count = 0;	/* Arp request count in passive sniffer mode */
static int arp_sniff_reply_count = 0;	/* Arp reply count in passive sniffer mode */
static int sarpi_timeout = 10;		/* Default Arp cache refresh timeout, 10 minuts */
static int darpi_timeout = 500;		/* Default DARPI cache entry timeout, 500 ms */

/**********************
 * Task mode handler: *
 **********************/

/* 
 * Sets CPU priority for who 
 */
static int 
task_mode_cpu_priority(int which, int who, int prio)
{
	
	if (cpu_priority != 0) {
		if (setpriority(which, who, prio) < 0) {
			ERROR(strerror(errno));
			return (-1);
		}
	}

	return (0);
}

/* 
 * Demonize the process. 
 */
static int
task_mode_daemon()
{
	struct stat stats;
	struct tm tm_cur;
	FILE *pid_fstream;	
	time_t time_cur;
	int fd;

	if ((pid_fstream = task_mode_pid_open()) == NULL)
		return (-1);

	if (stat(pid_file, &stats) < 0) {
		aprintf(stderr, "\n  ERROR: %s\n\n", strerror(errno));
		return (-1);
	}

	task_mode_pid_close(pid_fstream);

	if (! S_ISREG(stats.st_mode)) {
		aprintf(stderr, "\n  ERROR: %s is not a regular file.\n\n", pid_file);
		return (-1);
	}

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("\n  [%02d/%02d/%04d - %02d:%02d:%02d %s] "); 

	aprintf(stdout, "Task is forking to background, using %s pid file...\n\n", pid_file);

	switch (fork()) { 
	case (-1):
		ERROR(strerror(errno));
		return (-1);
		
	case (0):
		break;
        	
	default:
		exit(EXIT_SUCCESS);	
	}

	if (setsid() < 0) {
		ERROR(strerror(errno));
		return (-1);
	}

	/* PID Scheduling */	
	if (task_mode_cpu_priority(PRIO_PROCESS, getpid(), cpu_priority) < 0)
		return (-1);

	/* Write arpon.pid */
	if ((pid_fstream = task_mode_pid_open()) == NULL) {
		kill(pid_main, SIGTERM);
		exit(EXIT_FAILURE);
	}
	
	fprintf(pid_fstream, "%d\n", (int)getpid());
	fflush(pid_fstream);
	task_mode_pid_close(pid_fstream);
	
	if ((fd = open("/dev/null", O_RDWR, 0)) < 0) {
		ERROR(strerror(errno));
		return (-1);
	}

	dup2(fd, STDIN_FILENO);
	dup2(fd, STDOUT_FILENO);
		
	if (fd > 2)
		close(fd);
	
	return (0);
}

/* 
 * Open pid file 
 */
static FILE *
task_mode_pid_open()
{
	FILE *pid_fstream;

	if ((pid_fstream = fopen(pid_file, "w+")) == NULL) {
		aprintf(stderr, "\n  ERROR: %s: %s\n\n", pid_file, strerror(errno));
		return (NULL);
	}

	return (pid_fstream);
}

/* 
 * Close pid file 
 */
static void
task_mode_pid_close(FILE *pid_fstream)
{

	if (pid_fstream != NULL)
		fclose(pid_fstream);
}


/* 
 * Logging mode 
 */
static int 
task_mode_log()
{
	struct stat stats;
	struct tm tm_cur;
	time_t time_cur;
	FILE *log_fstream;

	if ((log_fstream = task_mode_log_open()) == NULL)
		return (-1);

	if (stat(log_file, &stats) < 0) {
		aprintf(stderr, "\n  ERROR: %s: %s\n\n", log_file, strerror(errno));
		return (-1);
	}

	task_mode_log_close(log_fstream);

	if (! S_ISREG(stats.st_mode)) {
		aprintf(stderr, "\n  ERROR: %s is not a regular file.\n\n", log_file);
		return (-1);
	}

	/* Logging mode on */
	log_mode = 0;

	aprintf(NULL, "  ArpON \"Arp handler inspection\" version %s (%s)\n", CUR_VERSION, WEB_SITE);

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);

	return (0);
}

/* 
 * Open log file 
 */
static FILE * 
task_mode_log_open()
{
	FILE *log_fstream;

	if ((log_fstream = fopen(log_file, "a+")) == NULL) {
		aprintf(stderr, "\n  ERROR: %s: %s\n\n", log_file, strerror(errno));
		return (NULL);
	}

	return (log_fstream); 
}

/* 
 * Close log file 
 */
static void
task_mode_log_close(FILE *log_fstream)
{

	if (log_fstream != NULL)
		fclose(log_fstream);
}

/*******************
 * Device Handler: *
 *******************/

/* 
 * Handles the network interface with following modes:
 *     - Manual, 
 *     - Automatic, 
 *     - Listing eventual network interfaces
 *
 * doing the following operations:
 *     - Verifying Datalink 
 *     - Selecting the network interface to be used
 *     - Putting down the promiscue flag if found set
 *     - Reading MAC Address
 *     - Reading IP, netmask inet4 addresses
 *     - Printing out network interfaces dates
 */
static int
device_manager(enum device_t op_dev, char *devarg)
{
	char errbuf[PCAP_ERRBUF_SIZE];
	pcap_if_t *ifc;

	if (pcap_findalldevs(&ifc, errbuf) < 0) { 
		ERROR(errbuf);
		return (-1);
	}	

	for (; ifc != NULL; ifc = ifc->next) { 
		switch (op_dev) {
		case (DEVICE_MANUAL):
			if (strncmp(devarg, ifc->name, IF_NAMESIZE) == 0) {
				if ((ifc->flags & PCAP_IF_LOOPBACK) == 0) { 
					if (device_check_datalink(ifc->name) < 0) { 
						aprintf(stderr, "\n  ERROR: %s device's datalink is not supported!\n\n", dev);
						goto exit_failure;
					}			
						
					device_set_name(ifc->name);

					if (device_del_promisc() < 0 || device_get_mac_address() < 0 ||
				 	    device_get_inet4_addresses(DEVICE_MANUAL) < 0)
						goto exit_failure;
					
					device_info_print();	
					goto exit_success;
				} else {
					aprintf(stderr, "\n  ERROR: %s device is not supported!\n\n", dev);
					goto exit_failure;
				}
			}
			break;
				
		case (DEVICE_AUTO):
			if ((ifc->flags & PCAP_IF_LOOPBACK) == 0) {	
				if (device_check_datalink(ifc->name) < 0)
					break;

				device_set_name(ifc->name);
					
				if (device_del_promisc() < 0 || device_get_mac_address() < 0 ||
				    device_get_inet4_addresses(DEVICE_AUTO) < 0)
					goto exit_failure;
					
				device_info_print();	
				goto exit_success;
			}
			break;
				
		case (DEVICE_LIST):
			if ((ifc->flags & PCAP_IF_LOOPBACK) == 0) {
				if (device_check_datalink(ifc->name) < 0)
					break;
					
				device_set_name(ifc->name);
					
				if (device_del_promisc() < 0 || device_get_mac_address() < 0 ||
				    device_get_inet4_addresses(DEVICE_LIST) < 0)
					goto exit_failure;
					
				device_info_print();
			}
			break;
		}
	}

	if (op_dev == DEVICE_MANUAL) {
		if (strncmp(devarg, dev.dev, IF_NAMESIZE) != 0) {
			aprintf(stderr, "\n  ERROR: %s device not found!\n\n", dev);
			goto exit_failure;
		}
	}
	
	pcap_freealldevs(ifc);
	return (0);
	
	exit_failure:
		pcap_freealldevs(ifc);
		return (-1);

	exit_success:
		pcap_freealldevs(ifc);
		return (0);
}

/*
 * Sets the network interface
 */
static void
device_set_name(char *name)
{

	memset(dev.dev, '\0',  IF_NAMESIZE);
	strncpy(dev.dev, name, IF_NAMESIZE);
}

/*
 * Checks for the datalink, accepting EN10MB only
 * (and so only ethernet and wireless devices)
 */
static int
device_check_datalink(char *devarg)
{
	char errbuf[PCAP_ERRBUF_SIZE];
	pcap_t *pcap;
	int datalink;	

	if ((pcap = pcap_open_live(devarg, BUFSIZ, 0, 0, errbuf)) == NULL) {
		ERROR(errbuf);
		return (-1);
	}
	
	if ((datalink = pcap_datalink(pcap)) < 0) {
		ERROR(pcap_geterr(pcap));
		return (-1);
	}
	
	/* 
	 * Set network offset if it is
	 * ethernet network interface
	 */
	if (datalink == DLT_EN10MB) {
		dev.dev_offset = 14;
	
		pcap_close(pcap);
		return (0);
	}
	
	pcap_close(pcap);
	return (-1);
}

/*
 * Putting down the promiscue flag if found set in network interface
 */
static int
device_del_promisc()
{	
	struct ifreq ifr;
	int sd;
	
	if ((sd = socket(AF_INET, SOCK_STREAM, 0)) < 0) {
		ERROR(strerror(errno));
		return (-1);
	}
	
	strncpy(ifr.ifr_name, dev.dev, sizeof(char) * IF_NAMESIZE);
	ifr.ifr_name[sizeof(char) * strlen(ifr.ifr_name)] = '\0';
	
	if (ioctl(sd, SIOCGIFFLAGS, &ifr) < 0) {
		ERROR(strerror(errno));
		goto exit_failure;
	}

	if (ifr.ifr_flags & IFF_PROMISC) {
		/* Remove Promisc flag */
		ifr.ifr_flags &= ~IFF_PROMISC;
		
		if (ioctl(sd, SIOCSIFFLAGS, &ifr) < 0) {
			ERROR(strerror(errno));
			goto exit_failure;
		}
	}

	close(sd);
	return (0);
	
	exit_failure:
		close(sd);
		return (-1);
}


/*
 * Reading network interfaces MAC address 
 */
static int
device_get_mac_address()
{
	eth_addr_t mac;
	eth_t *if_eth;
	register int i;
	
	if ((if_eth = eth_open(dev.dev)) == NULL) {
		ERROR(strerror(errno));
		return (-1);
	}
	
	if (eth_get(if_eth, &mac) < 0) {
		ERROR(strerror(errno));
		
		eth_close(if_eth);
		return (-1);
	}
	
	for (i = 0; i < ETHER_ADDR_LEN; i++)
		dev.dev_mac.octet[i] = mac.data[i];

	eth_close(if_eth);
	return (0);
}

/*
 * Checks and saves IPv4 and netmask network adresses.
 */
static int 
device_get_inet4_addresses(enum device_t op_dev)
{
	char errbuf_lnet[LIBNET_ERRBUF_SIZE], errbuf_pcap[PCAP_ERRBUF_SIZE];
	libnet_t *lnet;
	bpf_u_int32 network;

	if ((lnet = libnet_init(LIBNET_LINK, dev.dev, errbuf_lnet)) == NULL) {
		if (op_dev == DEVICE_LIST) {
			dev.dev_inet4.s_addr = 0x0;
			dev.dev_netmask.s_addr = 0x0;
			
			return (0);
		} else { 
			ERROR(errbuf_lnet);
			return (-1);
		}
	}

	dev.dev_inet4.s_addr = libnet_get_ipaddr4(lnet);
	libnet_destroy(lnet);

	/* Get netmask */
	if((pcap_lookupnet(dev.dev, &network, &(dev.dev_netmask.s_addr), errbuf_pcap)) < 0) {
		if (op_dev == DEVICE_LIST) {
			dev.dev_inet4.s_addr = 0x0;
			dev.dev_netmask.s_addr = 0x0;
			
			return (0);
		} else {
			ERROR(errbuf_pcap);
			return (-1);
		}
	}

	return (0);
}

/*
 * Prints network interfaces dates
 */
static void 
device_info_print()
{
	struct tm tm_cur;
	time_t time_cur;

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);

	PRINT_TIME("\n  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "Device: (%s) ", dev.dev);
	aprintf(stdout, "MAC: %s ", ether_ntoa(&(dev.dev_mac)));
	
	if (dev.dev_inet4.s_addr == 0x0) {
		aprintf(stdout, "Inet4: %s ", "unknown");
		aprintf(stdout, "Netmask: %s\n", "unknown");
	} else {
		aprintf(stdout, "Inet4: %s ", inet_ntoa(dev.dev_inet4));
		aprintf(stdout, "Netmask: %s\n", inet_ntoa(dev.dev_netmask));
	}
} 
	 
/*********************
 * Arp ping handler: *
 *********************/

/*
 * Set ping response timeout
 */
static void 
arp_ping_set_timeout(char *timeout)
{

	ping_timeout = atoi(timeout);
}

/*
 * Ping to single host, sends a sole Arp 
 * request and waits a sole Arp reply
 */
static int
arp_ping_host(char *arg)
{
	struct ether_addr dst_mac;
	struct in_addr dst_ip;
	struct tm tm_cur;
	pcap_t *pcap;
	time_t time_cur;

	memcpy(&dst_mac, ether_aton(MAC_ADDR_BROADCAST), sizeof(dst_mac));
	
	if (inet_aton(arg, &dst_ip) == 0) {
		aprintf(stderr, "\n  ERROR: %s inet4 address malformed!\n\n", arg);
		return (-1);
	}

	if ((pcap = arp_ping_open_packet()) == NULL)
		return (-1);

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");

	/* PID Scheduling */
	if (task_mode_cpu_priority(PRIO_PROCESS, getpid(), cpu_priority) < 0)
		goto exit_failure;
	
	aprintf(stdout, "Arp Ping to Host (%s) with timeout: %d %s.\n", 
		arg, ping_timeout, ping_timeout == 1 ? "millisecond" : "milliseconds"); 
	
	if (arp_ping_send_packet(PING_HOST, &dst_mac, &dst_ip) < 0 ||
	    arp_ping_recv_packet(PING_HOST, pcap, (unsigned char *) arg) < 0)
		goto exit_failure;
	
	arp_ping_close_packet(pcap);
	return (0);
	
	exit_failure:
		arp_ping_close_packet(pcap);
		return (-1);
}

/*
 * It sends out N Arp requests and waits for N Arp
 * reply, where N is the number of host
 */
static int
arp_ping_broadcast()
{
	enum netmask_class_t n_hosts;
	struct ether_addr dst_mac;
	struct in_addr dst_ip;
	struct tm tm_cur;
	char host[INET4_ADDR_LEN];
	pcap_t *pcap = NULL;
	time_t time_cur;
	register int c, b, a;
	int netmask_c, netmask_b, netmask_a;
	int inet4_c, inet4_b, inet4_a;
	
	/* Number hosts */
	n_hosts	= ntohl(~(dev.dev_netmask.s_addr));
	
#if BYTE_ORDER == LITTLE_ENDIAN
	/* Netmask Classes */
	netmask_c = dev.dev_netmask.s_addr >> 24 & 0xff,
	netmask_b = dev.dev_netmask.s_addr >> 16 & 0xff,
	netmask_a = dev.dev_netmask.s_addr >> 8 & 0xff;
	/* Broadcast Classes */
	inet4_c = dev.dev_inet4.s_addr >> 16 & 0xff,
	inet4_b = dev.dev_inet4.s_addr >> 8 & 0xff,
	inet4_a = dev.dev_inet4.s_addr & 0xff;
#else
	/* Netmask Classes */
	netmask_c = dev.dev_netmask.s_addr & 0xff,
	netmask_b = dev.dev_netmask.s_addr >> 8 & 0xff,
	netmask_a = dev.dev_netmask.s_addr >> 16 & 0xff,
	/* Broadcast Classes */
	inet4_c = dev.dev_inet4.s_addr >> 8 & 0xff,
	inet4_b = dev.dev_inet4.s_addr >> 16 & 0xff,
	inet4_a = dev.dev_inet4.s_addr >> 24 & 0xff;
#endif

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	
	/* PID Scheduling */
	if (task_mode_cpu_priority(PRIO_PROCESS, getpid(), cpu_priority) < 0)
		goto exit_failure;

	aprintf(stdout, "Arp Ping to Broadcast with timeout: %d %s, ", 
		ping_timeout, ping_timeout == 1 ? "millisecond" : "milliseconds"); 
	aprintf(stdout, "Class: \"%c\", Possible Hosts: %d.\n", 
		n_hosts <= NETMASK_CLASS_C ? 'C' : n_hosts <= NETMASK_CLASS_B ? 'B' :
		n_hosts <= NETMASK_CLASS_A ? 'A' : '?' , n_hosts);
	
	if (n_hosts <= NETMASK_CLASS_C) {
		for (c = netmask_c; c < NETMASK_CLASS_C; c++) {
			if ((pcap = arp_ping_open_packet()) == NULL)
				return (-1);

			memcpy(&dst_mac, ether_aton(MAC_ADDR_BROADCAST), sizeof(dst_mac));
			snprintf(host, INET4_ADDR_LEN, "%d.%d.%d.%d", inet4_a, inet4_b, inet4_c, c);	
			
			if (inet_aton(host, &dst_ip) == 0) {
				aprintf(stderr, "\n  ERROR: %s inet4 address malformed!\n\n", host);
				goto exit_failure;
			}
			
			if (arp_ping_send_packet(PING_BROADCAST, &dst_mac, &dst_ip) < 0 ||
			    arp_ping_recv_packet(PING_BROADCAST, pcap, (unsigned char *) host) < 0)
				goto exit_failure;
			   
			arp_ping_close_packet(pcap);
		}
	} else if (n_hosts <= NETMASK_CLASS_B) {
		for (b = netmask_b; b < NETMASK_CLASS_C; b++) {
			for (c = netmask_c; c < NETMASK_CLASS_C; c++) {
				if ((pcap = arp_ping_open_packet()) == NULL)
					return (-1);

				memcpy(&dst_mac, ether_aton(MAC_ADDR_BROADCAST), sizeof(dst_mac));
				snprintf(host, INET4_ADDR_LEN, "%d.%d.%d.%d", inet4_a, inet4_b, b, c);	
				
				if (inet_aton(host, &dst_ip) == 0) { 
					aprintf(stderr, "\n  ERROR: %s inet4 address malformed!\n\n", host);
					goto exit_failure;
				}
				
				if (arp_ping_send_packet(PING_BROADCAST, &dst_mac, &dst_ip) < 0 ||
				    arp_ping_recv_packet(PING_BROADCAST, pcap, (unsigned char *) host) < 0)
					goto exit_failure;
			   
				arp_ping_close_packet(pcap);
			}
		}
	} else if (n_hosts <= NETMASK_CLASS_A) {
		for (a = netmask_a; a < NETMASK_CLASS_C; a++) {
			for (b = netmask_b; b < NETMASK_CLASS_C; b++) {
				for (c = netmask_c; c < NETMASK_CLASS_C; c++) {
					if ((pcap = arp_ping_open_packet()) == NULL)
						return (-1);

					memcpy(&dst_mac, ether_aton(MAC_ADDR_BROADCAST), sizeof(dst_mac));
					snprintf(host, INET4_ADDR_LEN, "%d.%d.%d.%d", inet4_a, a, b, c);	
					
					if (inet_aton(host, &dst_ip) == 0) {
						aprintf(stderr, "\n  ERROR: %s inet4 address malformed!\n\n", host);
						goto exit_failure;
					}
				
					if (arp_ping_send_packet(PING_BROADCAST, &dst_mac, &dst_ip) < 0 ||
					    arp_ping_recv_packet(PING_BROADCAST, pcap, (unsigned char *) host) < 0)
						goto exit_failure;
			   
					arp_ping_close_packet(pcap);
				}
			}
		}
	} else {
		/* Network Class unknown */
		aprintf(stderr, "\n  ERROR: netmask is not correct!\n\n");
		return (-1);
	}
	
	return (0);	
	
	exit_failure:
		arp_ping_close_packet(pcap);
		return (-1);
}

/*
 * Open pcap file descriptor
 */
static pcap_t *
arp_ping_open_packet()
{
	struct bpf_program compiled_filter;
	char errbuf[PCAP_ERRBUF_SIZE], *filter = "arp";
	pcap_t *pcap;
#ifndef LINUX
	unsigned int op = 1;
	
	if ((pcap = pcap_open_live(dev.dev, BUFSIZ, 0, ping_timeout, errbuf)) == NULL) {
#else
	if ((pcap = pcap_open_live(dev.dev, BUFSIZ, 0, 0, errbuf)) == NULL) {
#endif
		ERROR(errbuf);
		return (NULL);
	}

#ifndef LINUX
	/* 
	 * BSD, differently from linux does not support automatic 
	 * socket real time (Linux Socket Filter). 
	 * Therefore on BSD platform it's necessary to use BIOCIMMEDIATE
	 */
	if (ioctl(pcap_fileno(pcap), BIOCIMMEDIATE, &op) < 0) {
		ERROR(strerror(errno));
		return (NULL);
	}
#endif

	if (pcap_compile(pcap, &compiled_filter, filter, 0, dev.dev_netmask.s_addr) < 0) {
		ERROR(pcap_geterr(pcap));	
		goto exit_failure;
	}
	
	if (pcap_setfilter(pcap, &compiled_filter) < 0) {
		ERROR(pcap_geterr(pcap));
		goto exit_failure;
	}	

	if (pcap_setdirection(pcap, PCAP_D_IN) < 0) {
		ERROR(pcap_geterr(pcap));
		goto exit_failure;
	}
		
	pcap_freecode(&compiled_filter);
	return (pcap);
	
	exit_failure:
		arp_ping_close_packet(pcap);
		return (NULL);
}

/*
 * Close pcap file descriptor
 */
static void
arp_ping_close_packet(pcap_t *pcap)
{

	pcap_close(pcap);
}

/*
 * It receives the packet using a 500/user defined ms timeout,
 * if it arrives it starts reading
 */
static int
arp_ping_recv_packet(enum ping_t type, pcap_t *pcap, unsigned char *arg)
{

	signal(SIGINT, 	arp_ping_recv_packet_signal);
	signal(SIGTERM, arp_ping_recv_packet_signal);
	signal(SIGKILL, arp_ping_recv_packet_signal);

	/* This variable is used by arp_ping_read_packet() */
	ping_type = type;
	ping_check_reply = PING_REPLY_FALSE;

#ifdef LINUX
	/*
	 * Libpcap doesn't supported timeout on Linux platform.
	 * I resolved with workaround using alarm() and pthread 
	 */
	switch (type) {
	case (PING_HOST):
		memset(ping_host, '\0', sizeof(char) * INET4_ADDR_LEN);
		strncpy(ping_host, (char *) arg, sizeof(char) * INET4_ADDR_LEN);		
	
		/* milliseconds to microseconds, no interval */
		ualarm(ping_timeout * 1000, 0);
		signal(SIGALRM, arp_ping_recv_packet_signal);
	
		if (pcap_loop(pcap, 1, arp_ping_read_packet, arg) < 0) {
        		ERROR(pcap_geterr(pcap));
        	        return (-1);
        	}

		alarm(0);	
		memset(ping_host, '\0', sizeof(char) * INET4_ADDR_LEN);
		break;

	case (PING_BROADCAST):
		memset(ping_host, '\0', sizeof(char) * INET4_ADDR_LEN);
		strncpy(ping_host, (char *) arg, sizeof(char) * INET4_ADDR_LEN);

		/* milliseconds to microseconds, no interval */
		ualarm(ping_timeout * 1000, 0);
		signal(SIGALRM, arp_ping_recv_packet_signal);

		if (pthread_create(&thread[0], NULL, arp_ping_recv_packet_thread, (void *) pcap) != 0) {
			ERROR(strerror(errno));
			return (-1);
		}

		pthread_join(thread[0], NULL);
		alarm(0);
		break;
	}
#else
	if (pcap_dispatch(pcap, 1, arp_ping_read_packet, arg) < 0) {
		ERROR(pcap_geterr(pcap));
		return (-1);
	}
#endif

	/* ping_check_reply shows the value signed by arp_ping_read_packet() */
	if (type == PING_HOST && ping_check_reply == PING_REPLY_FALSE) {
		aprintf(stderr, "\n  ERROR: %s host is down!\n\n", arg);
		return (-1);
	}
	
	return (0);
}

/*
 * SIGALRM signal received
 */
static void
arp_ping_recv_packet_signal(int sig)
{

#ifdef LINUX
	if (sig == SIGALRM) {
		switch (ping_type) {
		case (PING_HOST):
			aprintf(stderr, "\n  ERROR: %s host is down!\n\n", ping_host);
			exit(EXIT_FAILURE);

		case (PING_BROADCAST):
			pthread_cancel(thread[0]);
			return;
		}
	}
#endif

	aprintf(stdout, "\r\n  Signal received...\n\n");
	exit(EXIT_FAILURE);
}

#ifdef LINUX
/*
 * Use thread for non blocking to pcap_loop()
 */
static void *
arp_ping_recv_packet_thread(void *arg)
{
	pcap_t *pcap = (pcap_t *) arg;

	if (pcap_loop(pcap, 1, arp_ping_read_packet, (unsigned char *) ping_host) < 0) {
		ERROR(pcap_geterr(pcap));
		return (NULL);
	}

	pthread_exit((void *) 0);
}
#endif

/*
 * Reads the inbound packet from Arp ping,
 * accepts only the Arp reply from the trusted host of request
 */
static void
arp_ping_read_packet(unsigned char *arg, const struct pcap_pkthdr *header, const unsigned char *packet)
{
	struct ether_addr src_mac;
	struct in_addr src_ip;
	struct tm tm_cur;
	char c_src_ip[INET4_ADDR_LEN];
	time_t time_cur;
	register int i;

	arp_packet = (struct arp_header *) (packet + dev.dev_offset);
	
	if (ntohs(arp_packet->ah_header.ar_op) == ARP_OP_REPLY) {
		snprintf(c_src_ip, INET4_ADDR_LEN, "%d.%d.%d.%d", 
			arp_packet->ah_addresses.ar_spa[0], arp_packet->ah_addresses.ar_spa[1],
			arp_packet->ah_addresses.ar_spa[2], arp_packet->ah_addresses.ar_spa[3]);
		inet_aton(c_src_ip, &src_ip);
	
		if (strcmp((char *) arg, inet_ntoa(src_ip)) == 0) {
			for (i = 0; i < ETHER_ADDR_LEN; i++)
				src_mac.octet[i] = arp_packet->ah_addresses.ar_sha[i];

			/* Localtime */
			time_cur = time(NULL);
			tm_cur = *(struct tm *) localtime(&time_cur);
			PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");

			/* This variable is assigned by arp_ping_recv_packets() */			
			switch (ping_type) {
			case (PING_HOST):
				aprintf(stdout, "<- Arp reply %s is-at (%s)\n", 
					inet_ntoa(src_ip), ether_ntoa(&src_mac)); 
				break;
				
			case (PING_BROADCAST):
				aprintf(stdout, "%d) Inet4: %15s  ->  Mac: %17s\n", 
					ping_broadcast_count, inet_ntoa(src_ip), ether_ntoa(&src_mac));
				
				ping_broadcast_count++;
				break;
			}
		
			/* Successfully received Arp reply from the host */
			ping_check_reply = PING_REPLY_TRUE;
		} else {
			/* Received Arp reply doesn't come from host */
			ping_check_reply = PING_REPLY_FALSE;
		}
	} else {
		/* This packet is not Arp reply */
		ping_check_reply = PING_REPLY_FALSE;
	}			
	
	/* 
	 * ping_check_reply will checks later from 
	 * arp_ping_recv_packets()
	 */	
}

/*
 * Sends Arp request packet for Arp ping
 */
static int
arp_ping_send_packet(enum ping_t type, struct ether_addr *dst_mac, struct in_addr *dst_ip)
{
	struct tm tm_cur;
	char errbuf[LIBNET_ERRBUF_SIZE];
	libnet_t *lnet;
	time_t time_cur;
	
	if ((lnet = libnet_init(LIBNET_LINK, dev.dev, errbuf)) == NULL) {
		ERROR(errbuf);
		return (-1);
	}
	
	if (libnet_autobuild_arp(ARPOP_REQUEST, 
				 (u_int8_t *) &(dev.dev_mac),	/* Source Mac */
				 (u_int8_t *) &(dev.dev_inet4),	/* Source InetV4 */
				 (u_int8_t *) dst_mac,		/* Destination Mac */
				 (u_int8_t *) dst_ip,		/* Destination InetV4 */
				 lnet ) < 0) {
		ERROR(libnet_geterror(lnet));
		goto exit_failure;
	}
	
	if (libnet_autobuild_ethernet((u_int8_t *) dst_mac, ETHERTYPE_ARP, lnet) < 0) {
		ERROR(libnet_geterror(lnet));
		goto exit_failure;
	}
	
	/* Sends (Ethernet + Arp) Packet */
	if (libnet_write(lnet) < 0) {
		ERROR(libnet_geterror(lnet));
		goto exit_failure;
	}
	
	if (type == PING_HOST) {
		/* Localtime */
		time_cur = time(NULL);
		tm_cur = *(struct tm *) localtime(&time_cur);
		PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
		
		aprintf(stdout, "-> Arp who-has %s (%s)", inet_ntoa(*dst_ip), ether_ntoa(dst_mac)); 
		aprintf(stdout, " tell %s (%s)\n", inet_ntoa(dev.dev_inet4), ether_ntoa(&(dev.dev_mac)));
		       
	}
		
	libnet_destroy(lnet);
	return (0);
	
	exit_failure:
		libnet_destroy(lnet);
		return (-1);
}

/******************************
 * Arp sniff packets handler: *
 ******************************/

/*
 * Process only inbound packets:
 *     - Arp reply
 *     - Arp request
 */
static int
arp_sniff_packets()
{
	struct tm tm_cur;
	pcap_t *pcap;
	time_t time_cur;
	
	if ((pcap = arp_sniff_open_packets()) == NULL)
		return (-1);

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");

	/* PID Scheduling */	
	if (task_mode_cpu_priority(PRIO_PROCESS, getpid(), cpu_priority) < 0)
		goto exit_failure;
	
	aprintf(stdout, "Sniffing Arp packets:\n");

	/* Used by arp_sniff_signal */ 	
	pcap_sniff = pcap;
	/* Ctrl + C */
	signal(SIGINT, arp_sniff_signal);

	if (pthread_create(&thread[0], NULL, arp_sniff_thread, (void *) pcap) != 0) {
		ERROR(strerror(errno));
		return (-1);
	}
	pthread_join(thread[0], NULL);
	
	arp_sniff_close_packets(pcap);
	return (0);
	
	exit_failure:
		arp_sniff_close_packets(pcap);
		return (-1);

}

/*
 * If SIGINT signal (Ctrl + C) arrives,
 * it goes back to the passively sniffed
 * packet's stats and then it exits
 */
static void
arp_sniff_signal(int sig)
{
	struct tm tm_cur;
	time_t time_cur;

	aprintf(stdout, "\r");

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "Arp packets stats:\n");

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "Received \"Arp Total\": %d\n", arp_sniff_request_count + arp_sniff_reply_count);
	
	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "Received \"Arp Request\": %d\n", arp_sniff_request_count);
	
	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "Received \"Arp Reply\": %d\n\n", arp_sniff_reply_count);

	pthread_cancel(thread[0]);
	exit(EXIT_SUCCESS);
}

/*
 * Use thread for non blocking to pcap_loop()
 */
static void *
arp_sniff_thread(void *arg)
{
	pcap_t *pcap = (pcap_t *) arg;

	while (1) {
		if (pcap_loop(pcap, 1, arp_sniff_read_packets, NULL) < 0) {
			ERROR(pcap_geterr(pcap));
			return (NULL);
		}
	}

	pthread_exit((void *) 0);
}

/*
 * Open pcap file descriptor
 */
static pcap_t *
arp_sniff_open_packets()
{
	struct bpf_program compiled_filter;
	char errbuf[PCAP_ERRBUF_SIZE], *filter = "arp";
	pcap_t *pcap;
#ifndef LINUX
	unsigned int op = 1;
#endif
	
	if ((pcap = pcap_open_live(dev.dev, BUFSIZ, 0, 0, errbuf)) == NULL) {
		ERROR(errbuf);
		return (NULL);
	}

#ifndef LINUX
	/* 
	 * BSD, differently from linux does not support automatic 
	 * socket real time (Linux Socket Filter). 
	 * Therefore on BSD platform it's necessary to use BIOCIMMEDIATE
	 */
	if (ioctl(pcap_fileno(pcap), BIOCIMMEDIATE, &op) < 0) {
		ERROR(strerror(errno));
		return (NULL);
	}
#endif

	if (pcap_compile(pcap, &compiled_filter, filter, 0, dev.dev_netmask.s_addr) < 0) {
		ERROR(pcap_geterr(pcap));
		goto exit_failure;
	}
	
	if (pcap_setfilter(pcap, &compiled_filter) < 0) {
		ERROR(pcap_geterr(pcap));
		goto exit_failure;
	}	
	
	pcap_freecode(&compiled_filter);
	return (pcap);
	
	exit_failure:
		arp_sniff_close_packets(pcap);
		return (NULL);
}	

/*
 * Close pcap file descriptor
 */
static void
arp_sniff_close_packets(pcap_t *pcap)
{

	pcap_close(pcap);
}

/*
 * Reads inbound packets:
 *     - Arp reply
 *     - Arp request
 */
static void
arp_sniff_read_packets(unsigned char *arg, const struct pcap_pkthdr *header, const unsigned char *packet)
{
	struct ether_addr src_mac, dst_mac;
	struct in_addr src_ip, dst_ip;
	struct tm tm_cur;
	char c_src_ip[INET4_ADDR_LEN], c_dst_ip[INET4_ADDR_LEN];
	register int i;	
	time_t time_cur;

	arp_packet = (struct arp_header *) (packet + dev.dev_offset);

	if (ntohs(arp_packet->ah_header.ar_op) == ARP_OP_REPLY) {
		snprintf(c_src_ip, INET4_ADDR_LEN, "%d.%d.%d.%d", 
			arp_packet->ah_addresses.ar_spa[0], arp_packet->ah_addresses.ar_spa[1],
			arp_packet->ah_addresses.ar_spa[2], arp_packet->ah_addresses.ar_spa[3]);
		inet_aton(c_src_ip, &src_ip);
		
		for (i = 0; i < ETHER_ADDR_LEN; i++)
			src_mac.octet[i] = arp_packet->ah_addresses.ar_sha[i];
	
		/* Count Arp reply */
		arp_sniff_reply_count += 1;
		
		/* Localtime */
		time_cur = time(NULL);
		tm_cur = *(struct tm *) localtime(&time_cur);
		PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");	
		
		if (! strcmp(c_src_ip, inet_ntoa(dev.dev_inet4)))
			aprintf(stdout, "<- Arp reply %s is-at (%s)\n", inet_ntoa(src_ip), ether_ntoa(&src_mac));
		else
			aprintf(stdout, "-> Arp reply %s is-at (%s)\n", inet_ntoa(src_ip), ether_ntoa(&src_mac));
	} else if (ntohs(arp_packet->ah_header.ar_op) == ARP_OP_REQUEST) {
		snprintf(c_src_ip, INET4_ADDR_LEN, "%d.%d.%d.%d", 
			arp_packet->ah_addresses.ar_spa[0], arp_packet->ah_addresses.ar_spa[1],
			arp_packet->ah_addresses.ar_spa[2], arp_packet->ah_addresses.ar_spa[3]);
		inet_aton(c_src_ip, &src_ip);
		
		for (i = 0; i < ETHER_ADDR_LEN; i++)
			src_mac.octet[i] = arp_packet->ah_addresses.ar_sha[i];
	
		snprintf(c_dst_ip, INET4_ADDR_LEN, "%d.%d.%d.%d", 
			arp_packet->ah_addresses.ar_tpa[0], arp_packet->ah_addresses.ar_tpa[1],
			arp_packet->ah_addresses.ar_tpa[2], arp_packet->ah_addresses.ar_tpa[3]);
		inet_aton(c_dst_ip, &dst_ip);
	
		for (i = 0; i < ETHER_ADDR_LEN; i++)
			dst_mac.octet[i] = arp_packet->ah_addresses.ar_tha[i];
	
		/* Count Arp request */
		arp_sniff_request_count += 1;

		/* Localtime */
		time_cur = time(NULL);
		tm_cur = *(struct tm *) localtime(&time_cur);
		PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
		
		if (! strcmp(c_src_ip, inet_ntoa(dev.dev_inet4)))
			aprintf(stdout, "<- Arp who-has %s (%s)", inet_ntoa(dst_ip), ether_ntoa(&dst_mac));
		else
			aprintf(stdout, "-> Arp who-has %s (%s)", inet_ntoa(dst_ip), ether_ntoa(&dst_mac)); 
		
		aprintf(stdout, " tell %s (%s)\n", inet_ntoa(src_ip), ether_ntoa(&src_mac));
	}
}

/*
 * Arp Cache handler:
 */

/*
 * It handles the Arp cache's functionally:
 *     - Add 
 *     - Delete 
 *     - List Arp cache
 */
static int
arp_cache_manager(enum arp_cache_t op_ac, char *entry)
{
	arp_t *arp;

	if ((arp = arp_open()) == NULL) {
		ERROR(strerror(errno));
		return (-1);
	}
		
	switch (op_ac) {
	case (ARP_CACHE_ADD):
		if (arp_cache_add(ARP_CACHE, arp, entry) < 0)
			goto exit_failure;		
		break;
			
	case (ARP_CACHE_DEL):
		if (arp_cache_del(ARP_CACHE, arp, entry) < 0)
			goto exit_failure;
		break;
			
	case (ARP_CACHE_LIST):
		if (arp_cache_list(arp) < 0)
			goto exit_failure;
		break;
	}
	
	arp_close(arp);
	return (0);
	
	exit_failure:
		arp_close(arp);
		return (-1);
}

/*
 * Adds Arp cache entry
 */
static int
arp_cache_add(enum cache_t type, arp_t *arp, char *arg)
{
	struct arp_entry entry;
#ifdef LINUX
	struct arpreq ar;
#endif
	struct tm tm_cur;
	char c_mac[MAC_ADDR_LEN], c_ip[INET4_ADDR_LEN];
	time_t time_cur;
#ifdef LINUX
	int fd;
#endif

	sscanf(arg, "%15s %17s", c_ip, c_mac);
	c_mac[sizeof(char) * strlen(c_mac)] = '\0';
	c_ip[sizeof(char) * strlen(c_ip)] = '\0'; 
	
	if (addr_aton(c_mac, &entry.arp_ha) < 0) {
		ERROR(strerror(errno));
		return (-1);
	}
	
	if (addr_aton(c_ip, &entry.arp_pa) < 0) {
		ERROR(strerror(errno));	
		return (-1);
	}

#ifdef LINUX
	/*
	 * arp_add() for linux has grave problem,
	 * it doesn't choose correct network interface,
	 * with this method i resolve it
	 */

	if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
		ERROR(strerror(errno));
		return (-1);
	}

	memset(&ar, 0, sizeof(ar));

	/* Ip address */
	if (addr_ntos(&entry.arp_pa, &ar.arp_pa) < 0) {
		ERROR(strerror(errno));
		return (-1);
	}

	ar.arp_pa.sa_family = AF_INET;

	/* Mac address */
	if (addr_ntos(&entry.arp_ha, &ar.arp_ha) < 0) {
		ERROR(strerror(errno));
		return (-1);
	}
	
	ar.arp_ha.sa_family = ARP_HRD_ETH;

	/* Flags */
	ar.arp_flags = ATF_PERM | ATF_COM;

	/* Correct network interface */
	strncpy(ar.arp_dev, dev.dev, sizeof(ar.arp_dev));
	ar.arp_dev[strlen(ar.arp_dev)] = '\0';

	if (ioctl(fd, SIOCSARP, &ar) < 0) {
		ERROR(strerror(errno));
		return (-1);
	}

	close(fd);
#else
	if (arp_add(arp, &entry) < 0) { 
		ERROR(strerror(errno));
		return (-1);
	}
#endif
	
	/*
	 * These outputs depends by asked functionally
	 */	
	switch (type)
	{
	case (ARP_CACHE):
		/* Localtime */
		time_cur = time(NULL);
		tm_cur = *(struct tm *) localtime(&time_cur);
		PRINT_TIME("\n  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
		aprintf(stdout, "Arp Cache added %s -> %s entry.\n", c_ip, c_mac);
		break;
	
	case (SARPI):
		aprintf(stdout, "%s -> %s entry in Arp Cache.\n", c_ip, c_mac);
		break;
		
	case (SARPI_REFRESH):
		aprintf(stdout, "%15s -> %17s\n", c_ip, c_mac);
		break;

	case (DARPI):
		/* Localtime */
		time_cur = time(NULL);
		tm_cur = *(struct tm *) localtime(&time_cur);
		PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
		aprintf(stdout, "DARPI added %s -> %s entry in Arp Cache.\n", c_ip, c_mac);
		break;
	}
	
	return (0);
}

/*
 * Search entry in Arp cache, if is found it will delete 
 */
static int
arp_cache_del(enum cache_t type, arp_t *arp, char *arg)
{
	struct arp_entry entry;
#ifdef LINUX
	struct arpreq ar;
#endif
	struct tm tm_cur;
	unsigned int i;
	time_t time_cur;
#ifdef LINUX
	int fd;
#endif

	/* Read Arp cache */
	if (arp_loop(arp, arp_cache_list_create, NULL) < 0)
		return (-1);
	
	/* Search InetV4 or Mac in Arp Cache */
	TAILQ_FOREACH(arp_cache_pos, &arp_cache_head, entries) {
		if (strcmp(arg, inet_ntoa(arp_cache_pos->ac_ip)) == 0 || strcmp(arg, ether_ntoa(&arp_cache_pos->ac_mac)) == 0) { 
			for (i = 0; i < ETHER_ADDR_LEN; i++)
				entry.arp_ha.addr_eth.data[i] = arp_cache_pos->ac_mac.octet[i];
			
			if (addr_aton(inet_ntoa(arp_cache_pos->ac_ip), &entry.arp_pa) < 0) {
				ERROR(strerror(errno)); 
				return (-1);
			}

#ifdef LINUX
			/*
			 * arp_delete() for linux has grave problem,
			 * it doesn't choose correct network interface,
			 * with this method i resolve it
			 */

			if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
				ERROR(strerror(errno));
				return (-1);
			}

			memset(&ar, 0, sizeof(ar));

			/* Ip address */
			if (addr_ntos(&entry.arp_pa, &ar.arp_pa) < 0) {
				ERROR(strerror(errno));
				return (-1);
			}

			ar.arp_pa.sa_family = AF_INET;

			/* Correct network interface */
			strncpy(ar.arp_dev, dev.dev, sizeof(ar.arp_dev));
			ar.arp_dev[strlen(ar.arp_dev)] = '\0';

			if (ioctl(fd, SIOCDARP, &ar) < 0) {
				ERROR(strerror(errno));
				return (-1);
			}

			close(fd);
#else
			if (arp_delete(arp, &entry) < 0) {
				ERROR(strerror(errno));

				arp_cache_list_destroy();
				return (-1);
			}
#endif

			/*
			 * Depend by asked functionally
			 */	
			switch (type) {
			case (ARP_CACHE):
				/* Localtime */
				time_cur = time(NULL);
				tm_cur = *(struct tm *) localtime(&time_cur);
				PRINT_TIME("\n  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");

				aprintf(stdout, "Arp Cache deleted %s -> %s entry.\n",
			   		inet_ntoa(arp_cache_pos->ac_ip), ether_ntoa(&arp_cache_pos->ac_mac));
				break;
			
			case (SARPI):
			case (SARPI_REFRESH):
				break;
		
			case (DARPI):
				aprintf(stdout, "deleted %s -> %s entry from Arp Cache.\n",
			   		inet_ntoa(arp_cache_pos->ac_ip), ether_ntoa(&arp_cache_pos->ac_mac));
				break;
			}
			
			arp_cache_list_destroy();
			return (0);
		}
	}

	if (type == DARPI) {
		/* Localtime */
		time_cur = time(NULL);
		tm_cur = *(struct tm *) localtime(&time_cur);
		PRINT_TIME("\n  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
		aprintf(stdout, "DARPI deleted %s entry, not found in Arp Cache!\n", arg);
		arp_cache_list_destroy();
		return (0);
	}
	
	aprintf(stderr, "\n  ERROR: %s entry not found!\n\n", arg);
	arp_cache_list_destroy();
	return (-1);
}

/*
 * Prints Arp cache
 */
static int
arp_cache_list(arp_t *arp)
{
	struct tm tm_cur;
	time_t time_cur;

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("\n  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");

	aprintf(stdout, "Arp Cache list:\n");

	if (arp_loop(arp, arp_cache_list_create, NULL) < 0)
		return (-1);
			
	arp_cache_list_print();
	arp_cache_list_destroy();
	return (0);
}

/*
 * Create Arp cache tail using one entry
 * or it adds simply a new entry
 */
static int
arp_cache_list_create(const struct arp_entry *entry, void *arg)
{
	struct ether_addr mac;
	struct in_addr ip;
	register int i;

	for (i = 0; i < ETHER_ADDR_LEN; i++)
		mac.octet[i] = entry->arp_ha.addr_eth.data[i];
	
	ip.s_addr = entry->arp_pa.addr_ip;
	
	if (arp_cache_begin == NULL) { 
		TAILQ_INIT(&arp_cache_head);
	
		if ((arp_cache_begin = malloc(sizeof(struct arp_cache))) == NULL) { 
			ERROR(strerror(errno));
			return (-1);
		}
		
		memcpy(&arp_cache_begin->ac_mac, &mac, sizeof(mac));
		memcpy(&arp_cache_begin->ac_ip, &ip, sizeof(ip));
		TAILQ_INSERT_HEAD(&arp_cache_head, arp_cache_begin, entries);
	} else {
		if ((arp_cache_next = malloc(sizeof(struct arp_cache))) == NULL) { 
			ERROR(strerror(errno));
			return (-1);
		}
		
		memcpy(&arp_cache_next->ac_mac, &mac, sizeof(mac));
		memcpy(&arp_cache_next->ac_ip, &ip, sizeof(ip));
		TAILQ_INSERT_TAIL(&arp_cache_head, arp_cache_next, entries);
	}
	
	return (0);
}

/*
 * Prints Arp cache tail
 */
static void
arp_cache_list_print()
{
	struct tm tm_cur;
	time_t time_cur;
	unsigned int i = 1;

	TAILQ_FOREACH(arp_cache_pos, &arp_cache_head, entries) {
		/* Localtime */
		time_cur = time(NULL);
		tm_cur = *(struct tm *) localtime(&time_cur);
		PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");

		aprintf(stdout, "%d) %15s -> %17s\n", i++, 
			inet_ntoa(arp_cache_pos->ac_ip), ether_ntoa(&arp_cache_pos->ac_mac)); 
	}
}

/*
 * Destroy Arp cache tail
 */
static void
arp_cache_list_destroy()
{

	while (TAILQ_EMPTY(&arp_cache_head) == 0) {
		arp_cache_pos = TAILQ_FIRST(&arp_cache_head);
		
		TAILQ_REMOVE(&arp_cache_head, arp_cache_pos, entries);
		free(arp_cache_pos);
	}
}


/******************
 * SARPI handler: *
 ******************/

/*
 * Sets Arp cache timeout for automatic update
 */
static void
sarpi_set_timeout(char *timeout)
{

	sarpi_timeout = atoi(timeout);
}

/*
 * Handles SARPI through two processes:
 *     - Child:	
 *     	   	Update automatically the Arp cache
 *     - Parent:	
 *		Works in real time, in other words
 *		it listens to the inbound packets	
 */
static int
sarpi_manager()
{
	struct tm tm_cur;
	arp_t *arp = NULL;
	time_t time_cur;

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "SARPI Start...\n");

	/* PID Parent Scheduling */
	if (task_mode_cpu_priority(PRIO_PROCESS, getpid(), cpu_priority) < 0)
		goto exit_failure;

	if ((arp = sarpi_arp_cache_open()) == NULL)
		return (-1);
	
	/* Arp Cache entries protect */
	sarpi_arp_cache_list();

	/* Cache refresh with thread for parallelism */
	if (pthread_create(&thread[0], NULL, sarpi_arp_cache_refresh_thread, (void *) arp) != 0) {
		ERROR(strerror(errno));
		return (-1);
	}

	if (sarpi_realtime(arp) < 0)
		goto exit_failure;
	
	sarpi_arp_cache_close(arp);
	return (0);
		
	exit_failure:
		sarpi_arp_cache_close(arp);
		return (-1);
}

/*
 * Open or create Arp cache tail
 */
static arp_t * 
sarpi_arp_cache_open()
{
	arp_t *arp;
	
	if ((arp = arp_open()) == NULL) {
		ERROR(strerror(errno));
		return (NULL);
	}
	
	if (arp_loop(arp, arp_cache_list_create, NULL) < 0) {
		arp_close(arp);
		return (NULL);
	}
			
	return (arp);
}

/*
 * Destroy and close Arp cache tail
 */
static void
sarpi_arp_cache_close(arp_t *arp)
{

	arp_cache_list_destroy();
	arp_close(arp);
}

/*
 * Prints Arp cache tail which contains
 * the entries protected by SARPI
 */
static void
sarpi_arp_cache_list()
{
	struct tm tm_cur;
	time_t time_cur;

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "SARPI protects these Arp Cache's entries:\n");
	
	arp_cache_list_print();
}

/*
 * During every timeout it updates
 * static entries in kernel's Arp cache
 */
static void * 
sarpi_arp_cache_refresh_thread(void *arg)
{
	struct tm tm_cur;
	arp_t *arp = (arp_t *) arg;
	char *entry;
	time_t time_cur;
	int entry_len, i;

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "SARPI Arp Cache refresh timeout: %d %s\n", sarpi_timeout, sarpi_timeout == 1 ? "minut." : "minuts.");
			
	for (;;) {
		/* timeout * 60 sec = N minuts */
		sleep(sarpi_timeout * 60);
	
		/* Localtime */
		time_cur = time(NULL);
		tm_cur = *(struct tm *) localtime(&time_cur);
		PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
		aprintf(stdout, "SARPI refresh these Arp Cache entries:\n");
		
		/* Count entries */		
		i = 1;
	
		TAILQ_FOREACH(arp_cache_pos, &arp_cache_head, entries) {
			entry_len = strlen(inet_ntoa(arp_cache_pos->ac_ip)) + strlen(ether_ntoa(&arp_cache_pos->ac_mac)) + 2;
			
			if ((entry = malloc(sizeof(char) * entry_len)) == NULL) {
				ERROR(strerror(errno));
				pthread_exit((void *) -1);
			}
			snprintf(entry, entry_len, "%s %s", inet_ntoa(arp_cache_pos->ac_ip), ether_ntoa(&arp_cache_pos->ac_mac));

			/* Localtime */
			time_cur = time(NULL);
			tm_cur = *(struct tm *) localtime(&time_cur);
			PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
			aprintf(stdout, "%d) ", i);
			
			/* Refresh Arp Cache entry */
			if (arp_cache_add(SARPI_REFRESH, arp, entry) < 0) {
				free(entry);
				pthread_exit((void *) -1);
			}
		
			free(entry);
			i++;
		}
	}
	
	pthread_exit((void *) 0);
}

/*
 * SARPI Realtime,
 * process Arp reply inbound packets
 */
static int
sarpi_realtime(arp_t *arp)
{
	struct tm tm_cur;
	pcap_t *pcap;
	time_t time_cur;

	if ((pcap = sarpi_realtime_open_packets()) == NULL)
		return (-1);
	
	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "SARPI Realtime Protect actived!\n");
	
	/* Ctrl + C */
	signal(SIGINT, sarpi_realtime_signal);
	signal(SIGTERM, sarpi_realtime_signal);
	signal(SIGKILL, sarpi_realtime_signal);

	if (pthread_create(&thread[1], NULL, sarpi_realtime_thread, (void *) pcap) != 0) {
		ERROR(strerror(errno));
		return (-1);
	}

	pthread_join(thread[0], NULL);
	pthread_join(thread[1], NULL);
	
	sarpi_realtime_close_packets(pcap);
	return (0);
}

/*
 * If SIGINT signal (Ctrl + C) or SIGTERM/SIGKILL arrives,
 * it kills child process and then it exits
 */
static void
sarpi_realtime_signal(int sig)
{
	
	aprintf(stdout, "\r\n  Signal received...\n\n");
	
	pthread_cancel(thread[0]);
	pthread_cancel(thread[1]);
	exit(EXIT_SUCCESS);
}

/*
 * Use thread for non blocking to pcap_loop()
 */
static void *
sarpi_realtime_thread(void *arg)
{
	pcap_t *pcap = (pcap_t *) arg;

	while (1) {
		if (pcap_loop(pcap, 1, sarpi_realtime_read_packets, NULL) < 0) {
			ERROR(pcap_geterr(pcap));
			return (NULL);
		}
	}

	pthread_exit((void *) 0);
}

/*
 * Open pcap file descriptor
 */
static pcap_t *
sarpi_realtime_open_packets()
{
	struct bpf_program compiled_filter;
	char errbuf[PCAP_ERRBUF_SIZE], *filter = "arp";
	pcap_t *pcap;
#ifndef LINUX
	unsigned int op = 1;
#endif

	if ((pcap = pcap_open_live(dev.dev, BUFSIZ, 0, 0, errbuf)) == NULL) {
		ERROR(errbuf);
		return (NULL);
	}

#ifndef LINUX
	/* 
	 * BSD, differently from linux does not support automatic 
	 * socket real time (Linux Socket Filter). 
	 * Therefore on BSD platform it's necessary to use BIOCIMMEDIATE
	 */
	if (ioctl(pcap_fileno(pcap), BIOCIMMEDIATE, &op) < 0) {
		ERROR(strerror(errno));
		return (NULL);
	}
#endif

	if (pcap_compile(pcap, &compiled_filter, filter, 0, dev.dev_netmask.s_addr) < 0) {
		ERROR(pcap_geterr(pcap));
		goto exit_failure;
	}
	
	if (pcap_setfilter(pcap, &compiled_filter) < 0) {
		ERROR(pcap_geterr(pcap));
		goto exit_failure;
	}	
	
	pcap_freecode(&compiled_filter);
	return (pcap);
		
	exit_failure:
		sarpi_realtime_close_packets(pcap);
		return (NULL);
}

/*
 * Close pcap file descriptor
 */
static void
sarpi_realtime_close_packets(pcap_t *pcap)
{

	pcap_close(pcap);
}

/*
 * Reads the Arp reply/request packets (it checks destionation address).
 * If Arp reply/request's data like addresses IP and MAC match a static entry of
 * Arp cache protected by SARPI, update the entry using the static one
 */
static void
sarpi_realtime_read_packets(unsigned char *arg, const struct pcap_pkthdr *header, const unsigned char *packet)
{
	struct ether_addr src_mac, dst_mac;
	struct in_addr src_ip, dst_ip;
	struct tm tm_cur;
	arp_t *arp;	
	char *entry;
	char c_src_ip[INET4_ADDR_LEN], c_src_mac[MAC_ADDR_LEN], 
	     c_dst_ip[INET4_ADDR_LEN], c_dst_mac[MAC_ADDR_LEN], 
	     c_ap_ip[INET4_ADDR_LEN], c_ap_mac[MAC_ADDR_LEN];
	time_t time_cur;
	register int i;
	int entry_len;

	/* Arp Packet */	
	arp_packet = (struct arp_header *) (packet + dev.dev_offset);

	snprintf(c_src_ip, INET4_ADDR_LEN, "%d.%d.%d.%d", 
		arp_packet->ah_addresses.ar_spa[0], arp_packet->ah_addresses.ar_spa[1],
		arp_packet->ah_addresses.ar_spa[2], arp_packet->ah_addresses.ar_spa[3]);
	inet_aton(c_src_ip, &src_ip);
		
	for (i = 0; i < ETHER_ADDR_LEN; i++)
		src_mac.octet[i] = arp_packet->ah_addresses.ar_sha[i];	
	strncpy(c_src_mac, ether_ntoa(&src_mac), sizeof(char) * MAC_ADDR_LEN);
	c_src_mac[sizeof(char) * strlen(c_src_mac)] = '\0';

	snprintf(c_dst_ip, INET4_ADDR_LEN, "%d.%d.%d.%d", 
		arp_packet->ah_addresses.ar_tpa[0], arp_packet->ah_addresses.ar_tpa[1],
		arp_packet->ah_addresses.ar_tpa[2], arp_packet->ah_addresses.ar_tpa[3]);
	inet_aton(c_dst_ip, &dst_ip);

	for (i = 0; i < ETHER_ADDR_LEN; i++)
		dst_mac.octet[i] = arp_packet->ah_addresses.ar_tha[i];
	strncpy(c_dst_mac, ether_ntoa(&dst_mac), sizeof(char) * MAC_ADDR_LEN);
	c_dst_mac[sizeof(char) * strlen(c_dst_mac)] = '\0';

	if (ntohs(arp_packet->ah_header.ar_op) == ARP_OP_REPLY || ntohs(arp_packet->ah_header.ar_op) == ARP_OP_REQUEST) {
		/* Check destionation address */
		if (strcmp(c_dst_ip, inet_ntoa(dev.dev_inet4)) == 0) {
			/* Search Arp reply/request source in Arp cache entries */
			TAILQ_FOREACH(arp_cache_pos, &arp_cache_head, entries) {
				memset(c_ap_ip, '\0', sizeof(char) * INET4_ADDR_LEN);
				strncpy(c_ap_ip, inet_ntoa(arp_cache_pos->ac_ip), sizeof(char) * INET4_ADDR_LEN);
				c_ap_ip[sizeof(char) * strlen(c_ap_ip)] = '\0';
				
				memset(c_ap_mac, '\0', sizeof(char) * MAC_ADDR_LEN);
				strncpy(c_ap_mac, ether_ntoa(&arp_cache_pos->ac_mac), sizeof(char) * MAC_ADDR_LEN);
				c_ap_mac[sizeof(char) * strlen(c_ap_mac)] = '\0';

				/* Source check in SARPI Cache */
				if (strcmp(c_src_ip , c_ap_ip) == 0) {
					/* Entry Found! */
					switch (ntohs(arp_packet->ah_header.ar_op)) {
					case (ARP_OP_REPLY):
						/* Localtime */
						time_cur = time(NULL);
						tm_cur = *(struct tm *) localtime(&time_cur);
						PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
						aprintf(stdout, "SARPI Arp Reply Inbound: ");
						break;

					case (ARP_OP_REQUEST):
						/* Localtime */
						time_cur = time(NULL);
						tm_cur = *(struct tm *) localtime(&time_cur);
						PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
						aprintf(stdout, "SARPI Arp Request Inbound: ");
						break;
					}

					aprintf(stdout, "Refresh ");

					entry_len = strlen(inet_ntoa(arp_cache_pos->ac_ip)) + 
						    strlen(ether_ntoa(&arp_cache_pos->ac_mac)) + 2;
					
					if ((entry = malloc(sizeof(char) * entry_len)) == NULL)
					{
						ERROR(strerror(errno));
						exit(EXIT_FAILURE);
					}

					snprintf(entry, entry_len, "%s %s", 
						 inet_ntoa(arp_cache_pos->ac_ip), ether_ntoa(&arp_cache_pos->ac_mac)); 
			
					if ((arp = arp_open()) == NULL) {
						free(entry);
						exit(EXIT_FAILURE);
					} 
					
					/* Refresh Arp Cache entry */
					if (arp_cache_add(SARPI, arp, entry) < 0) {
						free(entry);
						exit(EXIT_FAILURE);
					} 
	
					arp_close(arp);	
					free(entry);
					return;
				}
			}
			
			/* Source not found in SARPI Cache */
			switch (ntohs(arp_packet->ah_header.ar_op)) {
			case (ARP_OP_REPLY):
				/* Localtime */
				time_cur = time(NULL);
				tm_cur = *(struct tm *) localtime(&time_cur);
				PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
				aprintf(stdout, "SARPI Arp Reply Inbound: ");
				break;

			case (ARP_OP_REQUEST):
				/* Localtime */
				time_cur = time(NULL);
				tm_cur = *(struct tm *) localtime(&time_cur);
				PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
				aprintf(stdout, "SARPI Arp Request Inbound: ");
				break;
			}

			aprintf(stdout, "Ignores entry in Arp Cache: ");
			aprintf(stdout, "%s -> %s\n", inet_ntoa(src_ip), ether_ntoa(&src_mac));
		} 
		/* We are Source address */
		else {
			switch (ntohs(arp_packet->ah_header.ar_op)) {
			case (ARP_OP_REPLY):
				/* Localtime */
				time_cur = time(NULL);
				tm_cur = *(struct tm *) localtime(&time_cur);
				PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
				aprintf(stdout, "SARPI Arp Reply Outbound: ");
				break;
			
			case (ARP_OP_REQUEST):
				/* Localtime */
				time_cur = time(NULL);
				tm_cur = *(struct tm *) localtime(&time_cur);
				PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
				aprintf(stdout, "SARPI Arp Request Outbound: ");
				break;
			}

			aprintf(stdout, "Send to ");
			aprintf(stdout, "%s -> %s\n", inet_ntoa(dst_ip), ether_ntoa(&dst_mac));
		}
	}
}

/******************
 * DARPI handler: *
 ******************/

/*
 * Set DARPI Cache entry timeout.
 */
static void
darpi_set_timeout(char *timeout)
{

	darpi_timeout = atoi(timeout);
}

/* 
 * Handles DARPI, delete all found entries in the Arp cache to
 * delete some poisoned hosts then it starts realtime execution
 * to reads the packets:
 *     - Arp request
 *     - Arp reply
 */
static int
darpi_manager()
{
	struct tm tm_cur;
	time_t time_cur;
	
	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "DARPI Start...\n");

	/* PID Scheduling */
	if (task_mode_cpu_priority(PRIO_PROCESS, getpid(), cpu_priority) < 0)
		return (-1);
	
	/* Delete all Arp Cache entries */
	if (darpi_arp_cache_delete_entries() < 0)
		return (-1);

	if (darpi_realtime() < 0)
		return (-1);
	
	return (0);
}

/*
 * Open and create Arp cache tail
 */
static arp_t *
darpi_arp_cache_open()
{
	arp_t *arp;
	
	if ((arp = arp_open()) == NULL) {
		ERROR(strerror(errno));
		return (NULL);
	}
	
	if (arp_loop(arp, arp_cache_list_create, NULL) < 0) {
		arp_close(arp);
		return (NULL);
	}
			
	return (arp);
}

/*
 * Destroy Arp cache tail
 */
static void
darpi_arp_cache_close(arp_t *arp)
{

	arp_cache_list_destroy();
	arp_close(arp);
}

/*
 * Delete all found entries in the
 * Arp cache reading Arp cache tail
 */
static int
darpi_arp_cache_delete_entries()
{
	struct arp_entry entry;
#ifdef LINUX
	struct arpreq ar;
#endif
	struct tm tm_cur;
	arp_t *arp;
	time_t time_cur;
	int i, j = 1;
#ifdef LINUX
	int fd;
#endif

	if ((arp = darpi_arp_cache_open()) == NULL)
		return (-1);

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "DARPI deletes these Arp Cache entries:\n");
	
	TAILQ_FOREACH(arp_cache_pos, &arp_cache_head, entries) {
		entry.arp_pa.addr_type = ADDR_TYPE_IP;
				
		for (i = 0; i < ETHER_ADDR_LEN; i++)
			entry.arp_ha.addr_eth.data[i] = arp_cache_pos->ac_mac.octet[i];
			
		entry.arp_pa.addr_ip = arp_cache_pos->ac_ip.s_addr;

#ifdef LINUX
		/*
		 * arp_delete() for linux has grave problem,
		 * it doesn't choose correct network interface,
		 * with this method i resolve it
		 */

		if ((fd = socket(AF_INET, SOCK_DGRAM, 0)) < 0) {
			ERROR(strerror(errno));
			return (-1);
		}

		memset(&ar, 0, sizeof(ar));

		/* Ip address */
		if (addr_ntos(&entry.arp_pa, &ar.arp_pa) < 0) {
			ERROR(strerror(errno));
			return (-1);
		}

		ar.arp_pa.sa_family = AF_INET;

		/* Correct network interface */
		strncpy(ar.arp_dev, dev.dev, sizeof(ar.arp_dev));
		ar.arp_dev[strlen(ar.arp_dev)] = '\0';

		if (ioctl(fd, SIOCDARP, &ar) < 0) {
			ERROR(strerror(errno));
			return (-1);
		}

		close(fd);
#else
		if (arp_delete(arp, &entry) < 0) {
			ERROR(strerror(errno));
			return (-1);
		}
#endif

		/* Localtime */
		time_cur = time(NULL);
		tm_cur = *(struct tm *) localtime(&time_cur);
		PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
		aprintf(stdout, "%d) %15s -> %17s\n", j++, 
			inet_ntoa(arp_cache_pos->ac_ip), ether_ntoa(&arp_cache_pos->ac_mac)); 
	}
	
	darpi_arp_cache_close(arp);

	return (0);
}

/*
 * Adds or refresh an entry in Arp cache
 * checked by DARPI during the Realtime execution
 */
static int
darpi_arp_cache_add_entry(struct in_addr *ip, struct ether_addr *mac)
{
	arp_t *arp;
	char entry[strlen(inet_ntoa(*ip)) + strlen(ether_ntoa(mac)) + 2];

	if ((arp = darpi_arp_cache_open()) == NULL)
		return (-1);

	sprintf(entry, "%s %s", inet_ntoa(*ip), ether_ntoa(mac));
	entry[strlen(entry)] = '\0';
 	
	/* Add/Refresh Arp Cache entry */
	if (arp_cache_add(DARPI, arp, entry) < 0) {
		free(entry);
		return (-1);
	}
		
	darpi_arp_cache_close(arp);	

	return (0);
}

/*
 * Delete an entry in the Arp cache
 * checked by DARPI during the Realtime execution
 */
static int
darpi_arp_cache_del_entry(char *arg)
{
	arp_t *arp;

	if ((arp = darpi_arp_cache_open()) == NULL)
		return (-1);

	if (arp_cache_del(DARPI, arp, arg) < 0)
		return (-1);
	
	darpi_arp_cache_close(arp);
	
	return (0);
}

/*
 * DARPI Realtime execution
 * Process all outbound/inbound packets
 */
static int
darpi_realtime()
{
	struct tm tm_cur;
	pcap_t *pcap;
	time_t time_cur;

	if ((pcap = darpi_realtime_open_packets()) == NULL)
		return (-1);

	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "DARPI Cache entry timeout: %d %s\n", darpi_timeout, darpi_timeout == 1 ? "millisecond." : "milliseconds.");
	
	/* Localtime */
	time_cur = time(NULL);
	tm_cur = *(struct tm *) localtime(&time_cur);
	PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
	aprintf(stdout, "DARPI Realtime Protect actived!\n");
	
	/* Ctrl + C */
	signal(SIGINT, darpi_realtime_signal);
	signal(SIGTERM, darpi_realtime_signal);
	signal(SIGKILL, darpi_realtime_signal);

	pthread_attr_init(&join_attr);
	pthread_attr_setschedpolicy(&join_attr, SCHED_RR);

	if (pthread_create(&thread[0], &join_attr, darpi_realtime_thread, (void *) pcap) != 0) {
		ERROR(strerror(errno));
		return (-1);
	}
	pthread_join(thread[0], NULL);

	pthread_attr_destroy(&join_attr);
	darpi_cache_list_destroy();
	darpi_realtime_close_packets(pcap);
	
	return (0);
}

/*
 * If SIGINT signal (Ctrl + C) or SIGTERM/SIGKILL arrives,
 * it destroy Arp cache tail and then it exits
 */
static void
darpi_realtime_signal(int sig)
{

	aprintf(stdout, "\r\n  Signal received...\n\n");

	darpi_cache_list_destroy();
	pthread_attr_destroy(&detached_attr);
	pthread_cancel(thread[0]);
	pthread_rwlock_destroy(&wlock);	
	pthread_rwlock_destroy(&rlock);
	pthread_attr_destroy(&join_attr);

	exit(EXIT_SUCCESS);
}

/*
 * Use thread for non blocking to pcap_loop()
 */
static void *
darpi_realtime_thread(void *arg)
{
	pcap_t *pcap = (pcap_t *) arg;

	pthread_attr_init(&detached_attr);
	pthread_attr_setdetachstate(&detached_attr, PTHREAD_CREATE_DETACHED);
	pthread_attr_setschedpolicy(&detached_attr, SCHED_RR);
	pthread_rwlock_init(&rlock, NULL);
	pthread_rwlock_init(&wlock, NULL);

	while (1) {
		if (pcap_loop(pcap, 1, darpi_realtime_read_packets, NULL) < 0) {
			ERROR(pcap_geterr(pcap));

			pthread_rwlock_destroy(&wlock);
			pthread_rwlock_destroy(&rlock);
			pthread_attr_destroy(&detached_attr);	
			pthread_exit((void *) NULL);
		}
	}

	pthread_rwlock_destroy(&wlock);
	pthread_rwlock_destroy(&rlock);
	pthread_attr_destroy(&detached_attr);
	pthread_exit((void *) 0);
}

/*
 * Open pcap file descriptor
 */
static pcap_t *
darpi_realtime_open_packets()
{
	struct bpf_program compiled_filter;
	char errbuf[PCAP_ERRBUF_SIZE], *filter = "arp";
	pcap_t *pcap;
#ifndef LINUX
	unsigned int op = 1;
#endif
	
	if ((pcap = pcap_open_live(dev.dev, BUFSIZ, 0, 0, errbuf)) == NULL) {
		ERROR(errbuf);
		return (NULL);
	}

#ifndef LINUX
	/* 
	 * BSD, differently from linux does not support automatic 
	 * socket real time (Linux Socket Filter). 
	 * Therefore on BSD platform it's necessary to use BIOCIMMEDIATE
	 */
	if (ioctl(pcap_fileno(pcap), BIOCIMMEDIATE, &op) < 0) {
		ERROR(strerror(errno));
		return (NULL);
	}
#endif

	if (pcap_compile(pcap, &compiled_filter, filter, 0, dev.dev_netmask.s_addr) < 0) {
		ERROR(pcap_geterr(pcap));
		goto exit_failure;
	}
	
	if (pcap_setfilter(pcap, &compiled_filter) < 0) {
		ERROR(pcap_geterr(pcap));
		goto exit_failure;
	}	
	
	pcap_freecode(&compiled_filter);
	return (pcap);
	
	exit_failure:
		darpi_realtime_close_packets(pcap);
		return (NULL);
}

/*
 * Close pcap file descriptor
 */
static void
darpi_realtime_close_packets(pcap_t *pcap)
{

	pcap_close(pcap);
}

/*
 * Reads inbound packets such as:
 *     - Arp request:
 *		Checks the source, if it matches us:
 *		    Save the destination address in the DARPI Cache
 *		    then it checks the inbound Arp reply packets
 *		Or checks the destination, if it matches us:
 *		    Delete source address in Arp cache
 *		    (possible address poisoned)
 *     - Arp reply:
 *		Checks the destination, if matches us:
 *		    Checks the source address, if it matches in
 *		    DARPI cache, they are accepted and inserted
 *		    in Arp cache, otherwise they are rejected
 *		    and deleted in Arp cache
 */
static void
darpi_realtime_read_packets(unsigned char *arg, const struct pcap_pkthdr *header, const unsigned char *packet)
{
	struct ether_addr src_mac, dst_mac, dst_mac_req;
	struct in_addr src_ip, dst_ip, dst_ip_req;
	struct arp_header *arp_pack;
	struct tm tm_cur;
	char c_src_ip[INET4_ADDR_LEN], c_src_mac[MAC_ADDR_LEN], c_dst_ip[INET4_ADDR_LEN], c_dst_mac[MAC_ADDR_LEN];
	pthread_t thr;
	int i;	
	time_t time_cur;

	/* Arp Packet */
	arp_pack = (struct arp_header *) (packet + dev.dev_offset);

	snprintf(c_src_ip, INET4_ADDR_LEN, "%d.%d.%d.%d", 
		arp_pack->ah_addresses.ar_spa[0], arp_pack->ah_addresses.ar_spa[1],
		arp_pack->ah_addresses.ar_spa[2], arp_pack->ah_addresses.ar_spa[3]);
	inet_aton(c_src_ip, &src_ip);
	
	for (i = 0; i < ETHER_ADDR_LEN; i++)
		src_mac.octet[i] = arp_pack->ah_addresses.ar_sha[i];
	strncpy(c_src_mac, ether_ntoa(&src_mac), sizeof(char) * MAC_ADDR_LEN);
	c_src_mac[sizeof(char) * strlen(c_src_mac)] = '\0';

	snprintf(c_dst_ip, INET4_ADDR_LEN, "%d.%d.%d.%d", 
		arp_pack->ah_addresses.ar_tpa[0], arp_pack->ah_addresses.ar_tpa[1],
		arp_pack->ah_addresses.ar_tpa[2], arp_pack->ah_addresses.ar_tpa[3]);
	inet_aton(c_dst_ip, &dst_ip);

	for (i = 0; i < ETHER_ADDR_LEN; i++)
		dst_mac.octet[i] = arp_pack->ah_addresses.ar_tha[i];
	strncpy(c_dst_mac, ether_ntoa(&dst_mac), sizeof(char) * MAC_ADDR_LEN);
	c_dst_mac[sizeof(char) * strlen(c_dst_mac)] = '\0';

	if (ntohs(arp_pack->ah_header.ar_op) == ARP_OP_REQUEST)
	{
		/* Check Source Arp Request */
		if (strcmp(c_src_mac, ether_ntoa(&(dev.dev_mac))) == 0 && strcmp(c_src_ip, inet_ntoa(dev.dev_inet4)) == 0) {
			/* Localtime */
			time_cur = time(NULL);
			tm_cur = *(struct tm *) localtime(&time_cur);
			PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
			aprintf(stdout, "DARPI Arp Request Outbound: ");

			/* DARPI Cache Add/Refresh */
			if (darpi_cache_list_create(&dst_ip) < 0)
				exit(EXIT_FAILURE);

			/* Check DARPI Cache entry with timeout */
			pthread_create(&thr, &detached_attr, darpi_cache_list_create_check_thread, (void *) &dst_ip);
		}
		/* Check Destination Arp Request */ 
		else if (strcmp(c_dst_ip, inet_ntoa(dev.dev_inet4)) == 0) {
			/* Localtime */
			time_cur = time(NULL);
			tm_cur = *(struct tm *) localtime(&time_cur);
			PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
			aprintf(stdout, "DARPI Arp Request Inbound: ");

			/* Delete it in Arp cache (possible source address poisoned) */
			if (darpi_arp_cache_del_entry(c_src_ip) < 0)
				exit(EXIT_FAILURE);
	
			memcpy(&dst_mac_req, ether_aton(MAC_ADDR_BROADCAST), sizeof(dst_mac_req));	
			
			if (inet_aton(c_src_ip, &dst_ip_req) == 0) {
				aprintf(stderr, "\n  ERROR: %s inet4 address malformed!\n\n", c_src_ip);
				exit(EXIT_FAILURE);
			}
	
			/* After darpi will handlers this arp request and reply */	
			if (darpi_realtime_send_packet(&dst_mac_req, &dst_ip_req) < 0)
				exit(EXIT_FAILURE);
		}
	} else if (ntohs(arp_pack->ah_header.ar_op) == ARP_OP_REPLY) {
		/* Check Source Arp Reply */
		if (strcmp(c_src_ip, inet_ntoa(dev.dev_inet4)) == 0) {
			/* Localtime */
			time_cur = time(NULL);
			tm_cur = *(struct tm *) localtime(&time_cur);
			PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
			aprintf(stdout, "DARPI Arp Reply Outbound: ");

			aprintf(stdout, "Send to ");
			aprintf(stdout, "%s -> %s\n", c_dst_ip, c_dst_mac);	
		}	
		/* Check Destination Arp Reply */
		else if (strcmp(c_dst_mac, ether_ntoa(&(dev.dev_mac))) == 0 && strcmp(c_dst_ip, inet_ntoa(dev.dev_inet4)) == 0) {
			/* Localtime */
			time_cur = time(NULL);
			tm_cur = *(struct tm *) localtime(&time_cur);
			PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
			aprintf(stdout, "DARPI Arp Reply Inbound: ");

			/* 
			 * Search entry in DARPI cache, if it matches it is to inserted in
			 * Arp cache, else if it doesn't exist, it is to deleted from Arp cache
			 */
			if (darpi_cache_list_search(c_src_ip, &src_mac, &src_ip) < 0)
				exit(EXIT_FAILURE);
		}
	}
}

/*
 * Sends arp request 
 * (See Arp request Inbound in darpi_realtime_read_packets())
 */
static int
darpi_realtime_send_packet(struct ether_addr *dst_mac, struct in_addr *dst_ip)
{
	char errbuf[LIBNET_ERRBUF_SIZE];
	libnet_t *lnet;

	if ((lnet = libnet_init(LIBNET_LINK, dev.dev, errbuf)) == NULL) {
		ERROR(errbuf);
		return (-1);
	}
	
	if (libnet_autobuild_arp(ARPOP_REQUEST, 
				 (u_int8_t *) &(dev.dev_mac),	/* Source Mac */
				 (u_int8_t *) &(dev.dev_inet4),	/* Source InetV4 */
				 (u_int8_t *) dst_mac,		/* Destination Mac */
				 (u_int8_t *) dst_ip,		/* Destination InetV4 */
				 lnet ) < 0) {
		ERROR(libnet_geterror(lnet));
		goto exit_failure;
	}
	
	if (libnet_autobuild_ethernet((u_int8_t *) dst_mac, ETHERTYPE_ARP, lnet) < 0) {
		ERROR(libnet_geterror(lnet));
		goto exit_failure;
	}
	
	/* Sends (Ethernet + Arp) Packet */
	if (libnet_write(lnet) < 0) {
		ERROR(libnet_geterror(lnet));
		goto exit_failure;
	}

	libnet_destroy(lnet);
	return (0);
	
	exit_failure:
		libnet_destroy(lnet);
		return (-1);
}

/*
 * Create or adds DARPI cache tail
 */
static int 
darpi_cache_list_create(struct in_addr *entry)
{
	struct in_addr ip;

	ip.s_addr = entry->s_addr;
	
	aprintf(stdout, "Added %s entry in DARPI Cache!\n", inet_ntoa(*entry));
	
	pthread_rwlock_wrlock(&wlock);

	if (darpi_cache_begin == NULL) { 
		TAILQ_INIT(&darpi_cache_head);
	
		if ((darpi_cache_begin = malloc(sizeof(struct darpi_cache))) == NULL) {
			ERROR(strerror(errno));

			pthread_rwlock_unlock(&wlock);
			return (-1);
		}
		
		memcpy(&darpi_cache_begin->dc_ip, &ip, sizeof(ip));
		TAILQ_INSERT_HEAD(&darpi_cache_head, darpi_cache_begin, entries);

		pthread_rwlock_unlock(&wlock);
		return (0);
	}
	/* else */
	if ((darpi_cache_next = malloc(sizeof(struct darpi_cache))) == NULL) {
		ERROR(strerror(errno));

		pthread_rwlock_unlock(&wlock);
		return (-1);
	}
		
	memcpy(&darpi_cache_next->dc_ip, &ip, sizeof(ip));
	TAILQ_INSERT_TAIL(&darpi_cache_head, darpi_cache_next, entries);

	pthread_rwlock_unlock(&wlock);
	return (0);	
}

/*
 * DARPI Cache entry timeout.
 * When DARPI reads Arp Request Outbound, it writes
 * an entry in DARPI Cache, if this entry replies in 
 * Arp Reply Inbound it's ok, else if it doesn't replies,
 * it probably doesn't exist. This timeout in this case,
 * delete this entry from DARPI Cache. 
 */
static void *
darpi_cache_list_create_check_thread(void *arg)
{
	struct in_addr *entry = (struct in_addr *)arg;
	struct tm tm_cur;
	char ip[INET4_ADDR_LEN];
	time_t time_cur;

	snprintf(ip, sizeof(char) * INET4_ADDR_LEN, "%s", inet_ntoa(*entry));

	/* Microseconds to milliseconds */
	usleep(darpi_timeout * 1000);

	pthread_rwlock_rdlock(&rlock);

	TAILQ_FOREACH(darpi_cache_pos, &darpi_cache_head, entries) {
		if (strcmp(inet_ntoa(darpi_cache_pos->dc_ip), ip) == 0) {
			pthread_rwlock_unlock(&rlock);
			pthread_rwlock_wrlock(&wlock);

			TAILQ_REMOVE(&darpi_cache_head, darpi_cache_pos, entries);
			free(darpi_cache_pos);
			
			pthread_rwlock_unlock(&wlock);

			/* Localtime */
			time_cur = time(NULL);
			tm_cur = *(struct tm *) localtime(&time_cur);
			PRINT_TIME("  [%02d/%02d/%04d - %02d:%02d:%02d %s] ");
			aprintf(stdout, "DARPI Arp Reply Inbound Timeout: ");
			aprintf(stdout, "Deleted %s entry from DARPI Cache!\n", ip);

			pthread_exit((void *) 0);
		}
	}
	
	pthread_rwlock_unlock(&rlock);
	pthread_exit((void *) 0);
}

/*
 * Search entry in DARPI cache, if it matches it is to inserted in
 * Arp cache, else if it doesn't exist, it is to deleted from Arp cache
 */
static int
darpi_cache_list_search(char *entry, struct ether_addr *mac, struct in_addr *ip)
{

	pthread_rwlock_rdlock(&rlock);

	TAILQ_FOREACH(darpi_cache_pos, &darpi_cache_head, entries) {
		if (strcmp(inet_ntoa(darpi_cache_pos->dc_ip), entry) == 0) {
			pthread_rwlock_unlock(&rlock);
			pthread_rwlock_wrlock(&wlock);
			
			TAILQ_REMOVE(&darpi_cache_head, darpi_cache_pos, entries);
			free(darpi_cache_pos);

			pthread_rwlock_unlock(&wlock);

			aprintf(stdout, "%s entry found in DARPI Cache, deleted it.\n", entry);
		
			if (darpi_arp_cache_add_entry(ip, mac) < 0)
				return (-1);
		
			return (0);
		}
	}

	pthread_rwlock_unlock(&rlock);

	aprintf(stdout, "%s entry not found in DARPI Cache!\n", entry);
	
	if (darpi_arp_cache_del_entry(entry) < 0)
		return (-1);

	return (0);
}

/*
 * Destroy DARPI cache tail
 */
static void
darpi_cache_list_destroy()
{

	pthread_rwlock_rdlock(&rlock);

	while (TAILQ_EMPTY(&darpi_cache_head) == 0) {
		pthread_rwlock_wrlock(&wlock);
	
		darpi_cache_pos = TAILQ_FIRST(&darpi_cache_head);
	
		aprintf(stdout, "  - DARPI Realtime deletes %s entry in DARPI Cache!\n", 
			inet_ntoa(darpi_cache_pos->dc_ip));
	
		TAILQ_REMOVE(&darpi_cache_head, darpi_cache_pos, entries);
		free(darpi_cache_pos);
	
		pthread_rwlock_unlock(&wlock);
	}

	pthread_rwlock_unlock(&rlock);
}

/*****************
 * Misc handler: *
 *****************/

/* 
 * My printf() with logging mode
 */
static void 
aprintf(FILE *stream, char *fmt, ...)
{
	FILE *log_fstream;
	va_list ap;

	if (stream != NULL) {
		va_start(ap, fmt);
		vfprintf(stream, fmt, ap);
		va_end(ap);
	}

	if (log_mode == 0) {
		if ((log_fstream = task_mode_log_open()) == NULL) {
			kill(pid_main, SIGTERM);
			exit(EXIT_FAILURE);
		}

		va_start(ap, fmt);
		vfprintf(log_fstream, fmt, ap);
		va_end(ap);
		
		fflush(log_fstream);

		task_mode_log_close(log_fstream);
	}
}

/*
 * Prints license
 */
static void
license()
{

	printf("\n");
	printf("  Copyright (c) 2008 Andrea Di Pasquale <spikey.it@gmail.com>\n");
 	printf("  All rights reserved.\n");
 	printf("\n");
 	printf("  Redistribution and use in source and binary forms, with or without\n");
 	printf("  modification, are permitted provided that the following conditions\n");
 	printf("  are met:\n");
 	printf("  1. Redistributions of source code must retain the above copyright\n");
 	printf("     notice(s), this list of conditions and the following disclaimer as\n");
 	printf("     the first lines of this file unmodified other than the possible\n");
 	printf("     addition of one or more copyright notices.\n");
 	printf("  2. Redistributions in binary form must reproduce the above copyright\n");
 	printf("     notice(s), this list of conditions and the following disclaimer in the\n");
 	printf("     documentation and/or other materials provided with the distribution.\n");
 	printf("\n");
 	printf("  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDER(S) ``AS IS'' AND ANY\n");
 	printf("  EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED\n");
 	printf("  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE\n");
 	printf("  DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER(S) BE LIABLE FOR ANY\n");
 	printf("  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES\n");
 	printf("  (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR\n");
 	printf("  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER\n");
 	printf("  CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT\n");
 	printf("  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY\n");
 	printf("  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH\n");
 	printf("  DAMAGE.\n\n");
}

/*
 * Prints version
 */
static void
version()
{
	
	printf("\n  ArpON \"Arp handler inspection\" version %s (%s)\n", CUR_VERSION, WEB_SITE);
}

/*
 * Prints help summary page
 */
static void
help()
{

	printf("\n");
	printf("  Usage: arpon [Task Mode] [Log Mode] [Device] {[Arping] | [Sniffer] | [Arp Cache] | [SARPI | DARPI]} [Misc]\n");
	printf("\n");	
	printf("  TASK MODE:\n");
	printf("    -n, --nice           <Nice Value>    Sets PID's CPU priority\n");
	printf("                                         (Default Nice: %d)\n", cpu_priority);
	printf("    -d, --daemon                         Works in background task\n");
	printf("                                         (Default: %s)\n", pid_file);	
	printf("  LOG MODE:\n");
	printf("    -f, --log-file       <Log file>      Sets log file\n");
	printf("                                         (Default: %s)\n", log_file);
	printf("    -g, --log                            Works in logging mode\n");
	printf("\n");
	printf("  DEVICE MANAGER:\n");
	printf("    -i, --dev-manual     <Device>        Sets your valid device manually\n");
	printf("    -o, --dev-auto                       Sets valid device automatically\n");
	printf("    -l, --dev-list                       Prints all valid devices\n");
	printf("                                         (Sets last valid device)\n");
	printf("  ARP PING:\n");
	printf("    -m, --ping-timeout   <Timeout>       Sets Arp Ping response timeout\n");
	printf("                                         (Default: %d milliseconds)\n", ping_timeout);
	printf("    -p, --ping-host      <Inet4>         Sends Arp Ping to Inet4 address\n");
	printf("    -b, --ping-broadcast                 Sends Arp Ping to Broadcast address\n");
	printf("                                         (Prints LAN's active hosts)\n");
	printf("  ARP PASSIVE SNIFFER:\n");
	printf("    -r, --sniff-arp                      Sniffs only Arp protocol\n");
	printf("                                         (I/O Arp Request/Reply)\n");
	printf("  ARP CACHE MANAGER:\n");
	printf("    -a, --cache-add      <\"Inet4  MAC\">  Adds Inet4 and MAC Arp entry\n");
	printf("    -e, --cache-del      <Inet4 | MAC>   Deletes Inet4 or MAC Arp entry\n");
	printf("    -t, --cache-list                     Prints total ARP Cache entries\n");
	printf("\n");	
	printf("  STATIC ARP INSPECTION:\n");
	printf("    -u, --sarpi-timeout  <Timeout>       Sets Arp Cache refresh timeout\n");
	printf("                                         (Default: %d minuts)\n", sarpi_timeout);
	printf("    -s, --sarpi                          Manages Arp Cache statically\n");
	printf("\n");
	printf("  DYNAMIC ARP INSPECTION:\n");
	printf("    -z, --darpi-timeout  <Timeout>       Sets DARPI Cache entry timeout\n");
	printf("                                         (Default: %d milliseconds)\n", darpi_timeout);
	printf("    -y, --darpi                          Manages Arp Cache dinamically\n");
	printf("\n");	
	printf("  MISC:\n");
	printf("    -c, --license                        Prints license page\n");
	printf("    -v, --version                        Prints version number\n");
	printf("    -h, --help                           Prints help summary page\n");
	printf("\n");	
	printf("  SEE THE MAN PAGE FOR MANY DESCRIPTIONS AND EXAMPLES\n\n");
}

/*****************
 * Main handler: *
 *****************/

/* 
 * Main signal handler
 */
static void
main_signal(int sig)
{

	/* Kill all childs */
	kill(0, sig);
	exit(EXIT_SUCCESS);
}

/*
 * Main 
 */
int
main(int argc, char *argv[], char *envp[])
{
	struct option longopts[] = {
		{ "nice",		required_argument,	NULL,	'n' },
		{ "daemon",		no_argument,		NULL,	'd' },
	
		{ "log-file",		required_argument,	NULL, 	'f' },
		{ "log",		no_argument,		NULL,	'g' },
	
		{ "dev-manual",		required_argument,	NULL,   'i' },
		{ "dev-auto",		no_argument,		NULL,	'o' },
		{ "dev-list",		no_argument,        	NULL,	'l' },

		{ "ping-timeout",	required_argument,	NULL,	'm' },	
		{ "ping-host",		required_argument,	NULL,   'p' },
		{ "ping-broadcast",	no_argument,		NULL,	'b' },
	
		{ "sniff-arp",		no_argument,		NULL,	'r' },
	
		{ "cache-add",		required_argument,  	NULL,	'a' },
		{ "cache-del",		required_argument,  	NULL,	'e' },
		{ "cache-list",		no_argument,		NULL,	't' },
	
		{ "sarpi-timeout",	required_argument,	NULL,	'u' },
		{ "sarpi",		no_argument,		NULL,	's' },
	
		{ "darpi-timeout",	required_argument,	NULL,	'z' },	
		{ "darpi",		no_argument,		NULL,	'y' },
	
		{ "license",		no_argument,		NULL,	'c' },
		{ "version",		no_argument,		NULL,	'v' },
		{ "help",		no_argument,		NULL,	'h' },
	
		{ NULL,			0,                  	NULL,	 0  }
	};
	register int gopt;
	int type;

	if (argc == 1) {
		version();
		help();
		goto exit_success;
	}

	/* Sanitize environment from LD_PRELOAD attacks */
	if (getenv("LD_PRELOAD") != NULL) {
		unsetenv("LD_PRELOAD");
		execve(argv[0], argv, envp);
	}
	
	version();
	
	if (getuid() != ROOT_USER) {
		aprintf(stderr, "\n  ERROR: You are not root user!\n\n");
		goto exit_failure;
	}

	pid_main = getpid();

	signal(SIGINT, main_signal);
	signal(SIGKILL, main_signal);
	signal(SIGTERM, main_signal);

	while ((gopt = getopt_long(argc, argv, "n:df:gi:olm:p:bra:e:tu:sz:ycvh", longopts, NULL)) != -1) { 
		switch(gopt) {
		case ('n'):
			cpu_priority = atoi(optarg);
			break;

		case ('d'):
			if (task_mode_daemon() < 0)
				goto exit_failure;
			break;
	
		case ('f'):
			log_file = optarg;
			break;

		case ('g'):
			if (task_mode_log() < 0)
				goto exit_failure;
			break;
				
		case ('i'):
			if (device_manager(DEVICE_MANUAL, optarg) < 0)
				goto exit_failure;
			break;
				
		case ('o'):
			if (device_manager(DEVICE_AUTO, NULL) < 0)
				goto exit_failure;
			break;
				
		case ('l'):
			if (device_manager(DEVICE_LIST, NULL) < 0)
				goto exit_failure;
				
			aprintf(stdout, "\n");
			goto exit_success;
	
		case ('m'):
			arp_ping_set_timeout(optarg);
			break;

		case ('p'):
			if (dev.dev[0] == '\0') {
				if (device_manager(DEVICE_AUTO, NULL) < 0)
					goto exit_failure;
			}
				
			if (arp_ping_host(optarg) < 0)
				goto exit_failure;
			break;
				
		case ('b'):
			if (dev.dev[0] == '\0') {
				if (device_manager(DEVICE_AUTO, NULL) < 0)
					goto exit_failure;
			}
				
			if (arp_ping_broadcast() < 0)
				goto exit_failure;
			break;
				
		case ('r'):
			if (dev.dev[0] == '\0') {
				if (device_manager(DEVICE_AUTO, NULL) < 0)
					goto exit_failure;
			}
				
			if (arp_sniff_packets() < 0)
				goto exit_failure;
			break;
				
		case ('a'):
			if (arp_cache_manager(ARP_CACHE_ADD, optarg) < 0)
				goto exit_failure;
			break;
			
		case ('e'):
			if (arp_cache_manager(ARP_CACHE_DEL, optarg) < 0)
				goto exit_failure;
			break;
			
		case ('t'):
			if (arp_cache_manager(ARP_CACHE_LIST, NULL) < 0)
				goto exit_failure;
			break;
				
		case ('u'):
			sarpi_set_timeout(optarg);
			break;
				
		case ('s'):
			switch (fork()) {
			case (-1):
				ERROR(strerror(errno));
				return (-1);

			/* Child for multiplexing */
			case (0):
				if (dev.dev[0] == '\0') {
					if (device_manager(DEVICE_AUTO, NULL) < 0)
						goto exit_failure;
				}
			
				if (sarpi_manager() < 0)
					goto exit_failure;
				break;
					
			default:
				break;
			}
			/* Parent */
			break;
			
		case ('z'):
			darpi_set_timeout(optarg);
			break;
	
		case ('y'):
			switch (fork()) {
			case (-1):
				ERROR(strerror(errno));
				return (-1);

			/* Child for multiplexing */
			case (0):
				if (dev.dev[0] == '\0') {
					if (device_manager(DEVICE_AUTO, NULL) < 0)
						goto exit_failure;
				}
				
				if (darpi_manager() < 0)
					goto exit_failure;
				break;
					
			default:
				break;	
			}
			/* Parent */
			break;
				
		case ('c'):
			license();
			goto exit_success;
	
		case ('v'):
			aprintf(stdout, "\n");
			goto exit_success;
	
		case ('h'):
			help();
			goto exit_success;
	
		default:
			break;
		}
	}

	/* Parent waits all childs */
	type = WNOHANG;
	wait(&type);
	
	argc -= optind;
	argv += optind;
	 
	aprintf(stdout, "\n");
	return (0);
	
	exit_failure:
		return (-1);

	exit_success:
		return (0);
}

/* EOF */ 
