/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	ISO file system creation code
 *
 *	by Tony Sideris	(06:06PM Dec 17, 2001)
 *================================================*/
#include "arson.h"

#include <qpushbutton.h>
#include <qtoolbutton.h>
#include <qlistview.h>
#include <qfileinfo.h>
#include <qdir.h>
#include <qxml.h>
#include <qcheckbox.h>
#include <qlineedit.h>
#include <qlayout.h>
#include <qlabel.h>

#include <klineeditdlg.h>
#include <kfiledialog.h>
#include <klocale.h>

#include "tempfile.h"
#include "xmlwriter.h"
#include "mainwnd.h"
#include "isofs.h"
#include "utils.h"

#define ISORC			"arsonisorc.xml"

/*========================================================*/
/*	Class encapsulating mkisofs flags
 *========================================================*/

ArsonIsoFlags::ArsonIsoFlags (void)
{
	//	Nothing...
}

/*========================================================*/

void ArsonIsoFlags::applyTo (ArsonProcessBase *proc, const QString &fn) const
{
	FLAGMAP::ConstIterator it, end;

	for (it = m_map.begin(), end = m_map.end(); it != end; ++it)
	{
		(*proc) << it.key();

		if (it.data() != QString::null)
			(*proc) << ArsonProcess::quoteFile(it.data());
	}

	if (fn != QString::null)
		(*proc) << "-o" << ArsonProcess::quoteFile(fn);
	else
		proc->invalidate(i18n("ISO filename not specified!"));
}

/*========================================================*/

void ArsonIsoFlags::save (ArsonXmlWriter &writer)
{
	FLAGMAP::Iterator it, end;

	for (it = m_map.begin(), end = m_map.end(); it != end; ++it)
	{
		ArsonXmlTag tag (writer, "option");
		tag.addAttribute("switch", it.key());

		if (it.data() != QString::null)
			tag.addAttribute("value", it.data());

		tag.doit();
	}
}

void ArsonIsoFlags::load (const QString &name, const QXmlAttributes &attr)
{
	if (name == "option")
	{
/*		Trace("s=%s v=%s\n",
			attr.value("switch").latin1(),
			attr.value("value").latin1());
*/
		addFlag(
			attr.value("switch"),
			attr.value("value"));
	}
}


/*========================================================*/
/*	Container class for various mkisofs presets
 *========================================================*/

class isoPresetParser : public ArsonUserCfg
{
public:
	isoPresetParser (ArsonIsoPresets *ptr)
		: ArsonUserCfg(),
		m_ptr(ptr)
	{}

	virtual bool startElement (const QString &ns, const QString &local,
		const QString &name, const QXmlAttributes &attr)
	{
//		Trace("Starting element: %s\n", name.latin1());
		
		if (name == "isopresets")
			return true;

		else if (name == "preset")
		{
			m_name = attr.value("name");
			m_cur.clear();
		}
		else
			m_cur.load(name, attr);

		return true;
	}

	virtual bool endElement (const QString &ns,
		const QString &local, const QString &name)
	{
		if (name == "preset")
			m_ptr->addPreset(m_name, m_cur);

		return true;
	}

private:
	ArsonIsoPresets *m_ptr;
	ArsonIsoFlags m_cur;
	QString m_name;
};

ArsonIsoPresets::ArsonIsoPresets (void)
{
	isoPresetParser parser (this);
	parser.readConfig(ISORC);
	
	//	Must at least contain a default entry
	if (!m_map.contains(defaultName()))
	{
		ArsonIsoFlags dflt;
		dflt.addFlag("-r")
			.addFlag("-V", "ARSONIMG");

		addPreset(
			defaultName(), dflt);
	}
}

/*========================================================*/

const QString &ArsonIsoPresets::defaultName (void)
{
	const static QString dft (i18n("Default"));
	return dft;
}

/*========================================================*/

QStringList ArsonIsoPresets::names (void) const
{
	QStringList sl;
	PRESETMAP::ConstIterator it, end;

	for (it = m_map.begin(), end = m_map.end(); it != end; ++it)
		sl.append(it.key());

	return sl;
}

/*========================================================*/

void ArsonIsoPresets::addPreset (const QString &name, ArsonIsoFlags &flags)
{
	m_map.insert(name, flags);
}

/*========================================================*/

void ArsonIsoPresets::save (void)
{
	const QString path = arsonDataFile(ISORC, "config", true);
	Trace("IsoPresets::save %d presets\n", m_map.count());

	if (path != QString::null)
	{
		QFile file (QFile::encodeName(path));

		if (file.open(IO_WriteOnly | IO_Truncate))
		{
			QTextStream ts (&file);
			PRESETMAP::Iterator it, end;
			ArsonXmlWriter writer (ts, "isopresets");

			writer.begin();
			
			/*	Have each preset create a preset node,
			 *	and save itself in that new node.
			 */
			for (it = m_map.begin(), end = m_map.end(); it != end; ++it)
			{
				ArsonXmlTag tag (writer, "preset");
				tag.addAttribute("name", it.key());

				tag.begin();

				it.data().save(writer);
				tag.end();
			}

			writer.end();
		}
	}
}

/*========================================================*/

void ArsonIsoPresets::remove (const QString &name)
{
	m_map.remove(name);
}

/*========================================================*/
/*	ISO creation manager implementation
 *========================================================*/

ArsonIsoMgr::ArsonIsoMgr (ArsonProcessUI *pUI, const QStringList &what)
	: ArsonIsoWriter(pUI, NULL),
	m_what(what),
	m_pFifo(NULL)
{
	ArsonIsoPresets temp;
	ArsonIsoFlags *pl;

	Verify((pl = temp.flags(ArsonIsoPresets::defaultName())));
	m_isoFlags = *pl;
}

ArsonIsoMgr::~ArsonIsoMgr (void)
{
	const QString isofile = opts().getString("isofile");
	
	//	Delete the ISO file
	if (opts().getBool("delete") && isofile != QString::null)
		ArsonFileCleanup() << isofile;

	delete m_pFifo;
}

/*========================================================*/

ArsonIsoSizeProcess::ArsonIsoSizeProcess (const QString &fifo, const ArsonIsoFlags *pFlags, const QStringList &what)
	: ArsonUtilityProcess(ACONFIG, Communication(Stdout | Stderr),
		QString::null, ArsonConfig::PROGRAM_MKISOFS),
	m_size(0)
{
	QStringList::ConstIterator it, end;
	ArsonIsoFlags flags (*pFlags);

	flags.addFlag("-quiet").addFlag("-print-size");
	flags.applyTo(this, fifo);

	for (it = what.begin(), end = what.end(); it != end; ++it)
		deferArg(quoteFile(*it));
}

void ArsonIsoSizeProcess::output (const QString &out, bool error)
{
	bool ok;
	const uint temp = out.toUInt(&ok);

	if (ok)
		m_size = temp;
}

/*========================================================*/
/*	A more specialized cdrecord process class...
 *	need to add tsize= param.
 *========================================================*/

class moreSpecializedCdrecordProcess : public ArsonCdrecordProcess
{
public:
	moreSpecializedCdrecordProcess (ArsonProcessMgr *pMgr,
		uint size, const QString &isofile)
		: ArsonCdrecordProcess(pMgr)
	{
		THIS << QString().sprintf("tsize=%us", size);

		deferArg("-data");
		deferArg(quoteFile(isofile));
	}
};

/*========================================================*/

ArsonWriterProcess *ArsonIsoMgr::createWriterProcess (void)
{
	if (m_pFifo && m_size != QString::null)
		return new moreSpecializedCdrecordProcess(this, m_size, m_pFifo->filename());

	return ArsonIsoWriter::createWriterProcess();
}

/*========================================================*/

ArsonMkisoProcess *ArsonIsoMgr::mkisofsProcess (const ArsonIsoFlags *pFlags,
	const QString &outfile, const QStringList &what)
{
	return new ArsonMkisoProcess(this, pFlags, outfile, what);
}

ArsonIsoSizeProcess *ArsonIsoMgr::sizeProcess (const ArsonIsoFlags *pFlags,
	const QString &outfile, const QStringList &what)
{
	return new ArsonIsoSizeProcess(outfile, pFlags, what);
}

/*========================================================*/

void ArsonIsoMgr::setupFifo (void)
{
	ArsonIsoFlags flags (m_isoFlags);

	m_pFifo = new ArsonFifo("image", "fifo.iso");

	if (m_pFifo->valid())
	{
		ArsonIsoSizeProcess *ps = sizeProcess(
			&m_isoFlags, m_pFifo->filename(), m_what);

		//	Precalc the size of the ISO image
		if (!ps->execute())
			arsonErrorMsg(
				i18n("Failed to execute mkisofs to determine image size"));
		else
		{
			ArsonMkisoProcess *pIso = mkisofsProcess(&flags,
				m_pFifo->filename(), m_what);

			//	This process actually creates the ISO image
			if (pIso->execute())
			{
				m_isoFile = m_pFifo->filename();
				m_size = ps->size();

				writeCd();				
			}
		}

		delete ps;

		if (!m_isoFile.isEmpty())
			return;
	}
	else
		arsonErrorMsg(
			i18n("Failed to create FIFO"));

	setProcess(NULL);
}

/*========================================================*/

void ArsonIsoMgr::begin (const ArsonProcessOpts &o)
{
	ArsonIsoWriter::begin(o);

	if (o.getBool("direct"))
	{
		if (opts().programPref(PROGGRP_DATACD) == "cdrdao")
			arsonErrorMsg(
				i18n("Direct burning with cdrdao is not currently supported, use cdrecord."));
		else
			setupFifo();
	}
	else
	{
		try {
			setProcess(
				mkisofsProcess(&m_isoFlags,
					o.getString("isofile"),
					m_what));
		}
		catch (ArsonError &err) {
			err.report();
		}
	}
}

/*========================================================*/

void ArsonIsoMgr::taskComplete (ArsonProcess *ptr)
{
	const bool success = ptr->successful();
	
	ArsonIsoWriter::taskComplete(ptr);

	if (success && m_isoFile == QString::null)
	{
		m_isoFile = opts().getString("isofile");
		writeCd();
	}
}

/*========================================================*/
/*	The progress dialog base class
 *========================================================*/

ArsonIsoProgress::ArsonIsoProgress (QWidget *parent, const char *name)
	: ArsonImgCreateProgress(parent, name)
{
	QLabel *pl;
	QToolButton *pt;

	m_pDirect = addCheckbox(
		i18n("Write image directly to burner &process (no intermediate file)"),
		"direct");

	m_pDelete = new ArsonProgressCheckbox(
		i18n("&Delete image after burn"), ctrlParent(), "deliso");
	m_pIsoFile = new QLineEdit(ctrlParent());

	pt = new QToolButton(ctrlParent());
	pt->setText(i18n("..."));

	pl = new QLabel(i18n("&ISO Options:"), ctrlParent());
	m_pPresets = new ArsonProgressCombobox(ctrlParent(), "isopreset");
	pl->setBuddy(m_pPresets);
	m_pPresets->setSizePolicy(
		QSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::Fixed));

	layoutRow(true) << m_pDelete;
	layoutRow(true) << m_pIsoFile << pt;

	LayoutRow lr (layoutRow());
	lr << pl << m_pPresets << spacer();
	showLabelEdit(lr.layout());

	QObject::connect(m_pDirect, SIGNAL(toggled(bool)), m_pDelete, SLOT(setDisabled(bool)));
	QObject::connect(m_pDirect, SIGNAL(toggled(bool)), m_pIsoFile, SLOT(setDisabled(bool)));
	QObject::connect(m_pDirect, SIGNAL(toggled(bool)), pt, SLOT(setDisabled(bool)));
	QObject::connect(pt, SIGNAL(clicked()), this, SLOT(iso_browse_clicked()));

	m_pIsoFile->setText(ArsonTempFile::tempFileName("img", "iso"));
	m_pDelete->setChecked(true);
	fillPresets();
}

/*========================================================*/

void ArsonIsoProgress::iso_browse_clicked (void)
{
	const QString path = KFileDialog::getSaveFileName();

	if (path != QString::null)
		m_pIsoFile->setText(path);
}

/*========================================================*/

void ArsonIsoProgress::fillPresets (void)
{
	ArsonIsoPresets presets;
	const QStringList sl = presets.names();

	m_pPresets->clear();

	for (uint index = 0; index < presets.count(); ++index)
		m_pPresets->insertItem(sl[index]);

	m_pPresets->setCurrentItem(0);
}

/*========================================================*/

void ArsonIsoProgress::processOpts (ArsonProcessOpts &opts)
{
	ArsonImgCreateProgress::processOpts(opts);

	opts.addBool("direct", (m_pDirect->isChecked() && m_pDirect->isEnabled()));
	opts.addBool("delete", m_pDelete->isChecked());
	opts.addString("isofile", m_pIsoFile->text());
}

/*========================================================*/

void ArsonIsoProgress::reconfigure (void)
{
	ArsonCdWriterProgress::reconfigure();

	m_pDirect->setEnabled(
		ACONFIG.programPref(PROGGRP_DATACD) == "cdrecord");
	
	fillPresets();
}

/*========================================================*/

ArsonProcessMgr *ArsonIsoProgress::createProcessMgr (void)
{
	ArsonIsoPresets presets;

	if (ArsonIsoFlags *pf = presets.flags(m_pPresets->currentText()))
	{
		if (ArsonIsoMgr *ptr = createIsoMgr())
		{
			ptr->setIsoFlags(*pf);
			return ptr;
		}
	}

	return NULL;
}

/*========================================================*/
/*	The Progress Dialog
 *========================================================*/

ArsonIsoWriterProgress::ArsonIsoWriterProgress (QWidget *parent, const QStringList &dirs)
	: ArsonIsoProgress(parent, "isofsdlg"), m_dirs(dirs)
{
	//	Nothing...
}

ArsonIsoMgr *ArsonIsoWriterProgress::createIsoMgr (void)
{
	return new ArsonIsoMgr(ui(), m_dirs);
}

/*========================================================*/
