/*
 * Copyright (c) 2001,2002 Tony Sideris
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */
/*================================================*/
/*	List window for arson, class implementation
 *
 *	by Tony Sideris	(06:40AM Jul 31, 2001)
 *================================================*/
#include "arson.h"

#include <qdragobject.h>
#include <qheader.h>

#include <klocale.h>

#include "listwnd.h"
#include "docwidget.h"

/*========================================================*/
/*	Function implementations
 *========================================================*/

ArsonListWndBase::ArsonListWndBase (QWidget *parent, const char *name)
	: KListView(parent, name),
	m_nLastCol(-1),
	m_bAscend(false)
{
	header()->setClickEnabled(false);
	setSorting(-1);

	memset (&m_headers, 0, sizeof(m_headers));
	setAllColumnsShowFocus(true);
	QListView::setSelectionMode(QListView::Extended);
}

/*========================================================*/

void ArsonListWndBase::setSortable (bool sortable)
{
	header()->setClickEnabled(sortable);
	setShowSortIndicator(sortable);

	if (sortable)
		QObject::connect(header(), SIGNAL(clicked(int)),
			this, SLOT(slotHdrClicked(int)));
	else
		QObject::disconnect(header(), SIGNAL(clicked(int)),
			this, SLOT(slotHdrClicked(int)));

	m_nLastCol = -1;
	m_bAscend = false;
}

/*========================================================*/

void ArsonListWndBase::resizeEvent (QResizeEvent *pEvent)
{
	KListView::resizeEvent(pEvent);
	resizeColumns(visibleWidth());
}

/*========================================================*/

void ArsonListWndBase::resizeColumns (int width)
{
	if (m_headers.pHdrs)
	{		
		//	Resize each of the column headers to the proper percentage
		for (int index = 0; index < m_headers.count; ++index)
		{
			const int colwidth = m_headers.pHdrs[index]
				.calc_width(width);

			setColumnWidth(index, colwidth);
		}
	}
}

/*========================================================*/

QSize ArsonListWndBase::sizeHint (void) const
{
	return QSize(-1,32);
}

/*========================================================*/

void ArsonListWndBase::setListHeaders (ArsonListHeader *pHdrs, int count)
{
	m_headers.pHdrs = pHdrs;
	m_headers.count = count;

	for (int index = 0; index < count; index++)
	{
		addColumn(i18n(pHdrs[index].text));
		setColumnWidthMode(index, QListView::Manual);
	}

	resizeColumns(visibleWidth());
}

/*========================================================*/

void ArsonListWndBase::slotHdrClicked (int col)
{
	bool ascend = m_bAscend;

	/*	Reset the sort order to ascending when
	 *	a different column is clicked from before
	 */
	if (col != m_nLastCol)
		ascend = true;

	/*	Do the sort, then turn off sorting so
	 *	drag-movable items still works.
	 */
	setSorting(col, ascend);
	sort();
	setSorting(-1);

	/*	Remember the column clicked, and the
	 *	inverse sort order for next time.
	 */
	m_nLastCol = col;
	m_bAscend = !ascend;
}

/*========================================================*/

uint ArsonListWndBase::itemListStates (void) const
{
	uint res = 0;

	if (QListViewItem *pi = selectedItem())
	{
		QListViewItem *pt;
		res |= itemDelete;

		if ((pt = pi->itemAbove()) && pt->parent() == pi->parent())
			res |= itemUp;

		if ((pt = pi->itemBelow()) && pt->parent() == pi->parent())
			res |= itemDown;
	}

	return res; 
}

/*========================================================*/

bool ArsonListWndBase::moveItemUp (void)
{
	QListViewItem *pi, *pre;

	if ((pi = selectedItem()) &&
		(pre = pi->itemAbove()) &&
		(pi->parent() == pre->parent()))
	{
		moveItem(pre, NULL, pi);
		setSelected(pi, true);
		return true;
	}

	return false;
}

bool ArsonListWndBase::moveItemDown (void)
{
	QListViewItem *pi, *next;

	if ((pi = selectedItem()) &&
		(next = pi->nextSibling()))
	{
		moveItem(pi, NULL, next);
		setSelected(pi, true);
		return true;
	}

	return false;
}

/*========================================================*/
/*	List window class containsing a view reference
 *========================================================*/

ArsonListWnd::ArsonListWnd (ArsonListDoc *parent)
	: ArsonListWndBase(parent, NULL)
{
	setSortable(true);

	QObject::connect(this, SIGNAL(contextMenu(KListView*,QListViewItem*,const QPoint&)),
		parent, SLOT(slotContextMenu(KListView*,QListViewItem*,const QPoint&)));
}

/*========================================================*/
/*	A listwnd that accepts file drops
 *========================================================*/

ArsonFileListWnd::ArsonFileListWnd (ArsonFileListDoc *parent)
	: ArsonListWnd(parent)
{
	setItemsMovable(true);
	setAcceptDrops(true);
	setDragEnabled(true);
}

/*========================================================*/

bool ArsonFileListWnd::acceptDrag (QDropEvent *pDrop) const
{
	QListViewItem *p, *a;
	ArsonFileListWnd *pw = (ArsonFileListWnd *) this;
	ArsonFileListDoc *pd = (ArsonFileListDoc *) parent();

	pw->findDrop(pDrop->pos(), p, a);

	if (pd->rootItem() && !p)
		return false;

	if (QUriDrag::canDecode(pDrop))
		return true;

	return ArsonListWnd::acceptDrag(pDrop);
}

/*========================================================*/
/*	Audio-specific list window
 *========================================================*/

ArsonListHeader ArsonMp3List::sHeaders[_HEADER_MAX] = {
	ArsonListHeader(I18N_NOOP("Filename"), 45),
	ArsonListHeader(I18N_NOOP("Title"), 45),
	ArsonListHeader(I18N_NOOP("Length"), 10),
};

ArsonMp3List::ArsonMp3List (ArsonFileListDoc *parent)
	: ArsonFileListWnd(parent)
{
	setListHeaders(sHeaders, _HEADER_MAX);
}

/*========================================================*/
