# -*- mode: python; coding: utf-8 -*-

"""types module provides more powerful structures and functions than built-in

.. moduleauthor:: Arco Research Group

"""
from __future__ import with_statement

import string, csv

class Record(object):
    "kargs are automatic attributes"

    def __init__(self, **kargs):
        self._attrs = kargs.keys()

        for k,v in kargs.items():
            setattr(self, k, v)

    def __str__(self):
        retval = ""
        for k in self._attrs:
            retval += "%s:'%s' " % (k, getattr(self, k))

        return "<Record %s>" % retval

# FIXME: probaly may be replaced with collections.OrderedDict in python2.7
class SortedDict(dict):
    """A fixed-position dictionary. The keys will be stored on a list in the
    same order as are inserted."""

    def __init__(self, other={}, default=None):
        self.__keylist = []
        self.default = default
        if isinstance(other, dict):
            self.update(other)
        if isinstance(other, list):
            for key,val in other:
                self[key] = val

    def __getitem__(self, key):
        if not key in dict.keys(self):
            if self.default is None:
                raise KeyError(key)

            self[key] = self.default

        return dict.__getitem__(self, key)


    def __setitem__(self, key, value):
        dict.__setitem__(self, key, value)
        if key in self.__keylist:
            self.__keylist.remove(key)
        self.__keylist.append(key)


    def update(self, other):
        for k,v in other.items():
            self[k] = v


    def keys(self):
        return self.__keylist


    def values(self):
        return [self[k] for k in self.__keylist]


    def items(self):
        return [(k, self[k]) for k in self.__keylist]


    def __iter__(self):
        return self.__keylist.__iter__()


    def clear(self):
        self.__keylist = []
        dict.clear(self)


#
# DEPRECATED: Use 'default' param in SortedDict
#
#class ListDict(SortedDict):
#    """A special case of a SortedDict. Here, if a key is unknown, it is created
#    en empty list for it."""
#
#    def __getitem__(self, key):
#        if not key in dict.keys(self):
#            self[key] = []
#        return SortedDict.__getitem__(self, key)


class DictTemplate:
    '''Read a template from file and generate a dict allowing substitutions

    template_file:
       full_name = Mr. $firstname %surname
       The $title = $value

    dt = DictTemplate('template_file', delimiter='=')
    result = t.substitute(dict(firstname='John', surname='Doe', title='job', value='fireman'))
    DictTemplate.render(result, 'out_file')

    out_file:
       full_name = Mr. John Doe
       The job = fireman
    '''

    def __init__(self, fname, delimiter=','):
        self.fname = fname
        self.delimiter = delimiter

    def substitute(self, values):
        with file(self.fname) as fd:
            result = string.Template(fd.read()).safe_substitute(values)

        csv_file = csv.reader(result.split('\n'), delimiter=self.delimiter)

        retval = {}
        for row in csv_file:
            if not row: continue
            assert len(row) == 2
            if any(['$' in x for x in row]): continue
            retval[row[0].strip()] = row[1].strip()

        return retval

    @classmethod
    def render(cls, data, fname, delimiter=','):
        """Render dict 'data' for file 'fname' using 'delimiter' """
        assert isinstance(data, dict)

        with open(fname, 'w') as fd:
            for x in data.items():
                fd.write("%s = %s\n" % x)


def merge(*input):
    """
    >>> merge([1,2], [2,4], [5, 6])
    [1, 2, 2, 4, 5, 6]
    >>> merge([[1,2], [2,4]])
    [[1, 2], [2, 4]]
    >>> merge(*[[1,2], [2,4]])
    [1, 2, 2, 4]
    """
    return reduce(list.__add__, input, list())


def merge_uniq(*input):
    """
    >>> merge_uniq([1,2], [2,4], [5, 6])
    [1, 2, 4, 5, 6]
    >>> merge_uniq([1,2])
    [1, 2]
    >>> merge_uniq(*[[1,2], [2,4]])
    [1, 2, 4]
    """
    return list(set(merge(*input)))

def get_supercls(cls):
    "Returns all ancestor superclasses as a list"
    return [cls] + merge(*[get_supercls(x) for x in cls.__bases__])
