# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::man;
use base 'AXP::Command';

use Arch::Util qw(load_file save_file);
use AXP::ModuleFinder;

sub infoline {
	"show manual pages of modules, like man(1)"
}

sub optusage {
	"[options] [Arch::Class]"
}

sub options {
	my $pager = $_[0]->default_pager;
	(
		format => { sh => 'f', type => '=s', desc => "produce another format: man/raw/text/html/pod" },
		pager  => { sh => 'p', type => '=s', desc => "use alternative pager ($pager)", init => $pager },
		output => { sh => 'o', type => "=s", arg => 'FILE', desc => "write to FILE rather than stdout" },
	)
}

sub helptext {
	q{
		Show or dump the documentation of axp and perl modules.
		Supported formats are: "man" (this is the default, invokes a
		pager), "raw" (raw man format), "text", "html", "pod".

		The --pager option only applies to the "man" format.
		The --output option only applies to any non "man" format.

		Without Arch::Class parameter, show the list of all found arch
		related perl modules with non-empty inline pod, this includes
		Arch::*, AXP::*, ArchZoom::* and ArchWay::* modules.
	}
}

sub default_pager {
	$ENV{PAGER} || "less -e"
}

sub show_index {
	my $finder = AXP::ModuleFinder->new;
	my @classes = sort keys %{$finder->find_modules_with_pod(undef, 1)};
	print "Available topics (please supply one):\n\n";
	print map { "    $_\n" } @classes;
	print "\n";
	exit 0;
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	my $module = shift @ARGV;
	$self->show_index unless $module;

	my $subfile = $module . ".pm";
	$subfile =~ s!::!/!g;
	my $file;
	foreach my $dir (@INC) {
		my $file0 = "$dir/$subfile";
		next unless -f $file0;
		$file = $file0;
		last;
	}
	die "No topic $module found\n" unless $file;

	my $format = $opt{format} || 'man';

	# process "man" format
	if ($format eq 'man') {
		my $pager = $self->default_pager;
		open(MANPIPE,
			"| pod2man --section 3 --release 'Arch' "
			. "--center 'Arch Perl library' $file "
			. "| nroff -man | $pager"
		);
		close(MANPIPE);
		exit 0;
	}

	my $content = load_file($file);
	die "No documentation in $module\n" unless $content =~ /^=head1/m;
	my $output = $opt{output} || "-";

	# process "pod" format
	if ($format eq 'pod') {
		my $pod = join('', $content =~ /(\n=\w.*?\n=cut\n)/);
		save_file($output, \$pod);
		exit 0;
	}

	my $subclass = { raw => 'Man', text => 'Text', html => 'Html' }->{$format};
	die "Unsupported pod conversion format: $format\n" unless $subclass;
	my $pod_class = "Pod::$subclass";
	eval "use $pod_class;"; die $@ if $@;

	# fork, since Pod::Parser does not support parsing from string
	my $child_pid = open(PODPIPE, "|-");
	defined $child_pid or die "Can't fork: $!";
	if ($child_pid) {
		print PODPIPE $content;
		close PODPIPE;
		wait;
		exit 0;
	}

	# process "html" format
	if ($format eq 'html') {
		pod2html("--infile=-", "--outfile=$output");
		unlink('pod2htmd.tmp');
		unlink('pod2htmi.tmp');
		unlink('pod2html-dircache');
		unlink('pod2html-itemcache');
	}
	# process "raw" and "text" formats
	else {
		my $parser = $pod_class->new;
		$parser->parse_from_file("-", $output);
	}
}

1;
