<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    user-defs.php                                           */
/* Author:      Paul Waite                                              */
/* Description: Definitions for managing USERS                          */
/*                                                                      */
/* ******************************************************************** */
/** @package core */

//-----------------------------------------------------------------------
/**
* The user class
* This class managed users. It pre-supposes a particular database
* structure based on three tables: uuser, ugroup, and uuser_group.
* Please see the example schemas for Phplib for further details.
* @package core
*/
class user {
  /** Login user id, string */
  var $userid = "";
  /** True if user record is valid */
  var $valid = false;
  /** Optional authorisation hash code */
  var $auth_code = "";
  /** Full name */
  var $name = "";
  /** Text password (encrypted or plain) */
  var $password = "";
  /** Password encryption flag */
  var $encrypted_passwords = false;
  /** User e-mail address */
  var $email = "";
  /** User type: arbitrary textual type */
  var $user_type = "";
  /** True of user is active/enabled */
  var $enabled = false;
  /** Total logins so far */
  var $total_logins = 0;
  /** Limit of logins allowed (0=unlimited) */
  var $limit_logins = 0;
  /** True if user has a group membership */
  var $hasgroups = false;
  /** Array of group membership names (strings) */
  var $group_names = array();
  /** Group membership details in full, as associative array */
  var $group_info;
  /** Group membership count */
  var $user_groups_cnt = 0;
  /** Complete user record as an associative array */
  var $user_record;
  /** List of IP addresses this user will be auto-logged-in from. */
  var $IP;
  /** Flag, true if user has auto-login IP addresses */
  var $hasIPlist = false;
  // .....................................................................
  /**
  * Constructor
  * Create a new user object.
  * @param string $userid User ID of the user
  */
  function user($userid="") {
    $this->userid = $userid;
    // A special case provided for making a brand new
    // user object for a new user record..
    if ($userid == "#new#") {
      // A new user profile..
      $this->valid = true;
      $this->name = "--enter user name--";
      $this->user_type = "user";
      $this->group_info[2] = "User"; // User
      $this->group_names[] = "User"; // User
      $this->user_groups_cnt = 1;
      $this->hasgroups = true;
      $this->userid = "<enter user id>";
    }
    // A normal user..
    else {
      // If id supplied, get user details..
      if ($userid != "") {
        $this->get_user_by_id($userid);
      }
    }
    return $this->valid;
  } // user
  // .....................................................................
  /**
  * Set the password encryption flag. If passed in true, then we will be
  * assuming that new passwords should be encrypted, and password checks
  * will try encryption first, then fall back to basic.
  * @param string $mode True if we are encrypting passwords, else false
  */
  function set_encrypted_passwords($mode=true) {
    $this->encrypted_passwords = $mode;
  } // set_encrypted_passwords
  // .....................................................................
  /**
  * Set the user login password. Store it according to the encryption
  * mode. We assume a plain text password is being supplied.
  * NB: Axyl-encrypted passwords always have an 'axenc_' prefix.
  * @param string $plainpass Plain text password to set for this user
  * $return string The password which is going to be stored
  */
  function set_password($plainpass) {
    if ($this->encrypted_passwords) {
      $password = "axenc_" . md5($password);
    }
    $this->password = $password;
    $this->user_record["password"] = $password;
    return $password;
  } // set_password
  // .....................................................................
  /**
  * Authenticate a user
  * Tries all types of authentication we know about using the parameters
  * passed to it.
  * @param string $authid   Unique user ID, authorization code or IP
  * @param string $password Password for the user
  * @return integer Login type code
  */
  function authenticate($authid, $password="") {
    $login_type = authenticate_userid($authid, $password);
    if ($login_type == LOGIN_UNKNOWN) {
      $login_type = authenticate_ip($authid);
      if ($login_type == LOGIN_UNKNOWN) {
        $login_type = authenticate_authid($authid);
      }
    }
    return $login_type;
  } // authenticate
  // .....................................................................
  /**
  * Authenticate a user by userid/password.
  * @param string $userid   Unique user ID of the user
  * @param string $password Password for the user
  * @return integer Login type code
  */
  function authenticate_userid($userid, $password="") {
    $login_type = LOGIN_UNKNOWN;
    // Guest authentication..
    if (stristr($userid, "guest")) {
      if ($this->user("guest") && $this->enabled) {
        $login_type = LOGIN_BY_GUEST;
      }
    }
    // Authentication by userid and password..
    elseif ($this->get_user_by_id($userid)) {
      if ($this->enabled) {
        // First try to match plain text..
        if ($this->password == $password) {
          $login_type = LOGIN_BY_PASSWD;
          debugbr("authenticated: plaintext password", DBG_DEBUG);
          if ($this->encrypted_passwords) {
            debugbr("warning: password should be encrypted!", DBG_DEBUG);
          }
        }
        else {
          // Try encrypted password match..
          $password = "axenc_" . md5($password);
          if ($this->password == $password) {
            $login_type = LOGIN_BY_PASSWD;
            debugbr("authenticated: encrypted password", DBG_DEBUG);
            if (!$this->encrypted_passwords) {
              debugbr("warning: password should be UN-encrypted!", DBG_DEBUG);
            }
          }
        }
      }
    }
    // Flag and return result..
    if ($login_type != LOGIN_UNKNOWN) debugbr("user '$userid' was authenticated", DBG_DEBUG);
    else {
      $this->valid = false;
      debugbr("user '$userid' failed authentication.", DBG_DEBUG);
    }
    return $login_type;
  } // authenticate_userid
  // .....................................................................
  /**
  * Authenticate a user by IP address
  * @param string $ip IP address of remote host accessing this site
  * @return integer Login type code
  */
  function authenticate_ipaddress($ip) {
    $login_type = LOGIN_UNKNOWN;
    // Authentication by IP..
    if ($this->get_user_by_ip($ip)) {
      if ($this->enabled) {
        $login_type = LOGIN_BY_IP;
      }
    }
    // Flag and return result..
    if ($login_type != LOGIN_UNKNOWN) debugbr("IP address '$ip' was authenticated", DBG_DEBUG);
    else {
      $this->valid = false;
      debugbr("IP address '$ip' failed authentication.", DBG_DEBUG);
    }
    return $login_type;
  } // authenticate_ipaddress
  // .....................................................................
  /**
  * Authenticate a user by authorisation ID
  * @param string $authid Authorisation code/id of the user
  * @return integer Login type code
  */
  function authenticate_authid($authid) {
    $login_type = LOGIN_UNKNOWN;
    // Authentication by unique authorsation code..
    if ($this->get_user_by_auth_code($authid)) {
      if ($this->enabled) {
        $login_type = LOGIN_BY_AUTHCODE;
      }
    }
    // Flag and return result..
    if ($login_type != LOGIN_UNKNOWN) debugbr("authid '$authid' was authenticated", DBG_DEBUG);
    else {
      $this->valid = false;
      debugbr("user '$authid' failed authentication.", DBG_DEBUG);
    }
    return $login_type;
  } // authenticate_authid
  // .....................................................................
  /**
  * Get user by ID
  * Internal function to return the user record from id.
  * @param string $userid Unique user ID
  * @return bool True if the user was found with the given user ID
  */
  function get_user_by_id($userid) {
    debug_trace($this);
    $this->valid = false;
    debugbr("get_user_by_id: getting '$userid'", DBG_DEBUG);
    $qu = dbrecordset("SELECT * FROM ax_user WHERE user_id='" . addslashes($userid) . "'");
    if ($qu->hasdata) {
      $this->user_record = $qu->current_row;
      $this->valid = true;
      $this->assign_vars();
    }
    debug_trace();
    return $this->valid;
  } // get_user_by_id
  // .....................................................................
  /**
  * Get user by Authorisation Code
  * Internal function to return the user record from auth_code. The
  * authorisation code is usually a string containing a complex key
  * generated by something like MD5 or better.
  * @param string $auth_code   Authorisation code to match for this user
  * @return bool True if the user was found with the given authorisation code
  */
  function get_user_by_auth_code($auth_code) {
    debug_trace($this);
    $this->valid = false;
    debugbr("get_user_by_auth_code: getting '$auth_code'", DBG_DEBUG);
    $qu = dbrecordset("SELECT * FROM ax_user WHERE auth_code='$auth_code'");
    if ($qu->hasdata) {
      $this->user_record = $qu->current_row;
      $this->valid = true;
      $this->assign_vars();
    }
    debug_trace();
    return $this->valid;
  } // get_user_by_auth_code
  // .....................................................................
  /**
  * Get user by IP
  * Internal function to return the user record which has IP address(es)
  * which coincide with the client IP address being used for this access.
  * @param string $ip Allowed IP host or network to allow logins from
  * @return bool True if a user was found with matching IP address
  */
  function get_user_by_ip($ip) {
    global $RESPONSE;
    debug_trace($this);
    debugbr("get_user_by_ip: getting '$ip'", DBG_DEBUG);
    $this->valid = false;
    if ($ip != "") {
      if (is_ipaddress($ip)) {
        switch ($RESPONSE->datasource->dbtype()) {
          case "postgres":
            // PostgreSQL support special inet datatype..
            $qu = dbrecordset("SELECT * FROM ax_user_ip WHERE ip >>= inet '$ip'");
            break;
          default:
            // All others use string comparison..
            $qu = dbrecordset("SELECT * FROM ax_user_ip WHERE ip='$ip'");
        }
        if ($qu->hasdata) {
          $userid = $qu->field("user_id");
          $ipcount = $qu->rowcount;
          $qu = dbrecordset("SELECT * FROM ax_user WHERE user_id='". addslashes($userid) . "'");
          if ($qu->hasdata) {
            $this->user_record = $qu->current_row;
            $this->valid = true;
            $this->assign_vars();
            debugbr("IP auth success. User=$this->userid", DBG_DEBUG);
          }
          $qip = dbrecordset("SELECT * FROM ax_user_ip WHERE user_id='". addslashes($userid) . "'");
          if ($qip->hasdata) {
            if (isset($this->IP)) unset($this->IP);
            $this->hasIPlist = true;
            do {
              $this->IP[] = $qip->field("ip");
            } while ($qip->get_next());
          }
          // Warning for badly defined IP login data..
          if ($ipcount > 1) {
            log_sys("WARNING: IP-based login overlap: $ipcount matches for Remote IP='$ip'");
          }
        }
        else {
          debugbr("get_user_by_ip: failed to authenticate '$ip'", DBG_DEBUG);
        }
      }
    }
    debug_trace();
    return $this->valid;
  } // get_user_by_ip
  // ....................................................................
  /**
  * Get user Authorisation Code
  * Return this user's unique authorisation code; generate
  * one if it isn't there yet, from userid and current time.
  * @return string The authorisation code for the current user
  */
  function get_auth_code() {
    if ($this->valid) {
      if ($this->auth_code == "") {
        debug_trace($this);
        $seed = $this->userid . $this->name . microtime();
        $this->auth_code = md5($seed);
        $this->user_record["auth_code"] = $this->auth_code;
        dbcommand("UPDATE ax_user SET auth_code='$this->auth_code' WHERE user_id='" . addslashes($this->userid) . "'");
        debug_trace();
      }
      return $this->auth_code;
    }
    else {
      return false;
    }
  } // get_auth_code
  // ....................................................................
  /**
  * Get user groups info
  * For this user, populate the group data for this object. We
  * read the uuser_group and ugroup tables and populate the
  * two variables @see $user_groups and @see $group_info
  * @return string The groups list for the user, delimited by pipe ("|")
  */
  function get_groups() {
    // Initialise..
    $ugroups = "";
    if (isset($this->group_info)) unset($this->group_info);

    // User group data acquisition..
    $q  = "SELECT *";
    $q .= "  FROM ax_user_group ug, ax_group g";
    $q .= " WHERE ug.user_id='" . addslashes($this->userid) . "'";
    $q .= "   AND g.group_id=ug.group_id";
    $group = dbrecordset($q);
    if ($group->hasdata) {
      $this->hasgroups = true;
      do {
        $groupid   = $group->field("group_id");
        $groupname = $group->field("group_desc");
        $this->group_info[$groupid] = $groupname;
        if ($ugroups != "") $ugroups .= "|";
        $ugroups .= $groupname;
      } while ($group->get_next());
    }
    // Force guest users into Guest group..
    if (stristr($this->userid, "guest")) {
      $ugroups = "Guest";
    }
    // Assign the group list and count..
    $this->group_names = explode("|", $ugroups);
    $this->user_groups_cnt = count($this->group_names);

    // A by-product..
    return $ugroups;
  } // get_groups
  // ....................................................................
  /**
  * Assign user variables
  * Internal function to assign variables from new record..
  * @access private
  */
  function assign_vars() {
    global $RESPONSE;
    debug_trace($this);
    if ($this->valid) {
      $this->userid = stripslashes($this->user_record["user_id"]);
      $this->name = stripslashes($this->user_record["full_name"]);
      $this->email = $this->user_record["email"];
      $this->password = $this->user_record["password"];
      if (!$this->encrypted_passwords) {
        $this->password = stripslashes($this->password);
      }
      $this->auth_code = $this->user_record["auth_code"];
      $this->user_type = $this->user_record["user_type"];
      $this->enabled = $RESPONSE->datasource->bool_from_db_value($this->user_record["enabled"]);
      $this->total_logins = $this->user_record["total_logins"];
      $this->limit_logins = $this->user_record["limit_logins"];
      // Get groups info..
      $this->get_groups();
    }
    else {
      $this->name = "Error: User not found";
      $this->user_type = "user";
    }
    debug_trace();
  } // assign_vars
  // ....................................................................
  /**
  * Is user a member of a named group. The argument passed in must be a
  * single group name string (ie. not a numeric group id) which is defined
  * in the database.
  * Return true if the user is a member of the named group.
  * @param string $groupname Name of the group we are checking user membership of
  * @return bool True if the user is a member of the group, else false
  */
  function ismemberof_group($groupname) {
    $found = false;
    for ($i = 0; $i < $this->user_groups_cnt; $i++) {
      if (!strcasecmp(trim($this->group_names[$i]), trim($groupname))) {
        $found = true;
        break;
      }
    } // for
    return $found;
  } // ismemberof_group
  // ....................................................................
  /**
  * Is user a member of one group of many
  * Check user against a list of groups, return true if member of at
  * least one of them. The list in $groupnames can be either a comma-delimited
  * string of group names, OR an array of group names.
  * @param mixed $groupnames_list Comma-delimited list OR array of group names
  * @return bool True if user is member of at least one of the groups, else false
  */
  function ismemberof_group_in($groupnames_list) {
    if (is_string($groupnames_list)) {
      if (trim($groupnames_list) == "") {
        return true;
      }
      $groupnames = explode(",", $groupnames_list);
    }
    else {
      $groupnames = $groupnames_list;
    }
    $found = false;
    if (count($groupnames) == 0) {
      return true;
    }
    foreach ($groupnames as $groupname) {
      $found = $this->ismemberof_group($groupname);
      if ($found) break;
    }
    return $found;
  } // ismemberof_group_in
  // ....................................................................
  /**
  * Is user a member of a group with ID
  * Return true if the user is a member of the group with given ID.
  * @param string $groupid   ID of the group we are checking user membership of
  * @return bool True if the user is a member of the group, else false
  */
  function ismemberof_group_with_id($groupid) {
    if (!isset($this->group_info)) {
      $this->get_groups();
    }
    return (isset($this->group_info[$groupid]));
  } // ismemberwith_groupid
  // ....................................................................
  /**
  * Return true if the current user is a valid one. This is false when the
  * user has not been authorised, or the user ID wasn't found etc. It is
  * an error condition for this to be false.
  * @return bool True if the current user object is valid
  */
  function isvalid() {
    return $this->valid;
  } // isvalid
  // ....................................................................
  /**
  * Get group IDs list
  * Return a string with the comma-delimited list of group ids which this
  * user belongs to in it. This is useful for using in an SQL statement like:
  *    WHERE group_id IN (group_ids_list())
  * for example. Note we only access the database to populate $this->group_info
  * when we need to, not every session.
  * @param string $delim Delimiter character (defaults to comma)
  * @return string List of group ID's comma-delimited
  */
  function group_ids_list($delim=",") {
    if (!isset($this->group_info)) {
      $this->get_groups();
    }
    $gplist = array();
    if (isset($this->group_info)) {
      foreach ($this->group_info as $gid => $gdesc) {
        $gplist[] = $gid;
      }
    }
    return implode($delim, $gplist);
  } // group_ids_list
  // ....................................................................
  /**
  * Get group names list
  * Return a string with the comma-delimited list of group names which this
  * user belongs to in it. Eg. "Editor,Author,Admin"
  * @param string $delim Delimiter character (defaults to comma)
  * @return string List of group name's comma-delimited
  */
  function group_names_list($delim=",") {
    if (!isset($this->group_info)) {
      $this->get_groups();
    }
    $gplist = array();
    if (isset($this->group_info)) {
      foreach ($this->group_info as $gid => $gdesc) {
        $gplist[] = $gdesc;
      }
    }
    return implode($delim, $gplist);
  } // group_names_list
  // ....................................................................
  /**
  * Get friendly name
  * Make a 'friendly' name from a full one. Good for "Dear... ,"
  * @return string Friendly name for the current user
  */
  function friendlyName() {
    if ($this->valid) {
      $splitname = explode(" ", $this->name);
      $mate = trim($splitname[0]);
      if ($mate == "") $mate = $this->name;
      return $mate;
    }
    else return "Invalid User";
  } // friendlyName

} // user class

// ----------------------------------------------------------------------
/**
* The Authorised User class
* This derived class just allows us a different way of defining
* a new user, when we know their authorisation code.
* @package core
*/
class authorised_user extends user {
  // .....................................................................
  /**
  * Constructor
  * Create a new authorised user object.
  * @param string $auth_code Authorisation code of the user
  */
  function authorised_user($auth_code="") {
    $this->user();
    if ($auth_code != "") {
      $this->get_user_by_auth_code($auth_code);
    }
  } // authorised_user

} // authorised_user class

// ----------------------------------------------------------------------
/**
* The Permissions class. This generic class manages permissions for a
* set of "agents" which are identified by a supplied "id". The permissions
* are the standard Create, Read, Update, Delete or any combination by
* ORing these values together.
* @package core
*/
// ACCESS MODES
/** Permission to create items */
define("PERM_CREATE",      0x01);
/** Permission to read/view items */
define("PERM_READ",        0x02);
/** Permission to update/modify items */
define("PERM_UPDATE",      0x04);
/** Permission to delete items */
define("PERM_DELETE",      0x08);

/** All permitted */
define("PERM_ALL",         0x0f);
/** Nothing permitted */
define("PERM_NONE",        0x00);

// PERMISSION RETURN CODES
/** Permission is given */
define("PERM_ALLOWED",     1);
/** Permission is refused */
define("PERM_DISALLOWED",  2);
/** Permission is undefined */
define("PERM_UNDEFINED",   3);

/** The default agent ID */
define("DEFAULT_AGENT", "__perm_default_agent__");

// ......................................................................
/**
* The permissions class. This class encpasulates a set of permissions
* which can be managed and tested by the associated methods.
* @package core
*/
class permissions {
  /** Array of permisssions. This is an associative array with the
      key being the identifier of an agent which can be permitted or
      disallowed from accessing things, and the value being a
      permission code as defined above. */
  var $perms = array();
  // .....................................................................
  /**
  * Constructor
  * Create a new permissions object with an optional permissions set.
  * @param mixed $perms If provided, must be an array of permissions
  */
  function permissions($perms=false) {
    // Always include default perm..
    $this->permit(DEFAULT_AGENT, PERM_READ);
    if ( is_array($perms) ) {
      $this->perms = $perms;
    }
  } // permissions
  // .....................................................................
  /**
  * Assign the default permission. This is the permission which is applied
  * if the supplied agent is not recognised.
  * @param integer $perm The default permission to apply for unrecognised agents
  */
  function setdefault($perm) {
    $this->permit(DEFAULT_AGENT, $perm);
  } // setdefault
  // .....................................................................
  /**
  * Assign the given agent(s) the given access permission. The first paramter
  * is a (comma) delimited list of agent IDs to assign the permission to.
  * @param mixed $agentids Agents to assign the permission to (array or delimited string)
  * @param integer $perm The permission of combination of perms to assign
  * @param string $delim The delimiter string separating agent IDs (default comma)
  */
  function permit($agentids, $perm, $delim=",") {
    if (is_array($agentids)) $agents = $agentids;
    else $agents = explode($delim, $agentids);
    foreach ($agents as $agentid) {
      $this->perms[$agentid] = $perm;
    }
  } // permit
  // .....................................................................
  /**
  * Un-assign the given agent(s) the given access permission. The first paramter
  * is a (comma) delimited list of agent IDs to unassign the permission from.
  * @param mixed $agentids Agents to unassign the permission from (array or delimited string)
  * @param integer $perm The permission of combination of perms to unassign
  * @param string $delim The delimiter string separating agent IDs (default comma)
  */
  function unpermit($agentids, $perm, $delim=",") {
    if (is_array($agentids)) $agents = $agentids;
    else $agents = explode($delim, $agentids);
    foreach ($agents as $agentid) {
      if (isset($this->perms[$agentid])) {
        $unperm  = $this->perms[$agentid] & $perm;
        $newperm = $this->perms[$agentid] ^ $unperm;
        $this->perms[$agentid] = $newperm;
      }
    }
  } // unpermit
  // .....................................................................
  /**
  * Low-level method for returning the permission for the given agent and
  * perm. We return one of three states: agent is allowed, agent is disallowed,
  * or agent permission status is undefined/unknown. The latter would occur
  * if the agent ID is unrecognised in this class (ie. not in the $perms array).
  * @param integer $agentid The unique agent ID to return the permission of
  * @param integer $perm The permission of combination of perms to assign
  * @return integer The permission status: allowed, disallowed or undefined
  */
  function permission($agentid, $perm) {
    if ( isset($this->perms[$agentid]) ) {
      return ($perm & $this->perms[$agentid] ? PERM_ALLOWED : PERM_DISALLOWED );
    }
    else {
      return PERM_UNDEFINED;
    }
  } // permission
  // .....................................................................
  /**
  * This is the main method for querying permission access rights for a given
  * agent. Returns a boolean value, true if the agent is permitted to access
  * in the given way, else false. If the agent ID is unrecognised, then the
  * method uses the 'default agent' permissions.
  * @param integer $agentid The agent to query the access permission of
  * @param integer $perm The access permission
  * @return boolean True if the agent is permitted access in given ways
  */
  function ispermitted($agentid, $perm) {
    $permission = $this->permission($agentid, $perm);
    if ($permission == PERM_UNDEFINED) {
      $permission = $this->permission(DEFAULT_AGENT, $perm);
    }
    return ($permission == PERM_ALLOWED);
  } // ispermitted
  // .....................................................................
  /**
  * This is a variant permitted query method, which takes a comma-delimited
  * list of agent IDs, and returns true if ANY one or more of these has the
  * required permissions. This facilitates passing of a group membership
  * list for a given user, for example.
  * @param mixed $agentids Agents to query the permission of (array or delimited string)
  * @param integer $perm The access permission
  * @param string $delim Delimiter character used (default is a comma)
  * @return boolean True if the agent is permitted access in given ways
  */
  function anypermitted($agentids, $perm, $delim=",") {
    $permitted = false;
    if (is_array($agentids)) $agents = $agentids;
    else $agents = explode($delim, $agentids);
    foreach ($agents as $agentid) {
      if ($this->ispermitted($agentid, $perm)) {
        $permitted = true;
        break;
      }
    }
    return $permitted;
  } // anypermitted
  // .....................................................................
  /**
  * Decode permission as a string of the form 'crud'
  * @param integer $perm The access permission to decode
  * @access private
  */
  function decode($perm) {
    $s = "";
    $s .= ($perm & PERM_CREATE ? "c" : "-");
    $s .= ($perm & PERM_READ   ? "r" : "-");
    $s .= ($perm & PERM_UPDATE ? "u" : "-");
    $s .= ($perm & PERM_DELETE ? "d" : "-");
    return $s;
  } // decode
  // .....................................................................
  /** Dump these permissions as text. Mainly a debugging aid.
  * @access private
  */
  function dump() {
    $s = "";
    reset($this->perms);
    while (list($agentid, $perm) = each($this->perms)) {
      if ($agentid != DEFAULT_AGENT) {
        $s .= $agentid . "&nbsp;" . "(" . $this->decode($perm) . ") ";
      }
    }
    if ($s == "") $s = "no perms";
    return $s;
  } // dump

} // permissions class

// ----------------------------------------------------------------------
?>