#!/usr/bin/php4 -q
<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    lu-optimize.php                                         */
/* Author:      Paul Waite                                              */
/* Description: Wrapper for sending the control message to Lucene to    */
/*              optimize its index. This script allows you to make sure */
/*              that your Axyl Lucene index is as small and efficient   */
/*              as it can be. Recommended practice is to optimize the   */
/*              index infrequently, say every night via cron, or after  */
/*              a large batch of documents has been indexed. After the  */
/*              optimize, if successful, a backup is done.              */
/*                                                                      */
/*              If the $WEBMASTER_EMAIL is defined, a success report is */
/*              sent. This is recommended to positively confirm the     */
/*              status of your index.                                   */
/*                                                                      */
/*       usage: ./lu-optimize.php                                       */
/*                                                                      */
/* ******************************************************************** */
// In older versions of Php CLI Program working dir is always its
// own directory so we need to get into the website root directory..
if (PHP_VERSION < 4.2) {
  chdir("..");
}
include_once("application.php");
include_once("lucene-defs.php");
include_once("mail-defs.php");
include_once("timer-defs.php");

$tinow = timestamp_to_displaydate(NICE_FULLDATETIME);

// Define a timer to report durations..
$timer = new microtimer();

// Initialise statuses..
$backed_up = false;
$optimized = false;

$rpt = "This email contains the status report for the Lucene optimize\n"
     . "and backup process completed at $tinow\n\n";

// Optimize the index after the batch has completed..
$timer->restart();
$optimizer = new lucene_utilitymsg("OPTIMIZE");
$optimizer->send(SOCK_FOREVER);
$success = $optimizer->response->valid;
$timer->stop();

if ($success) {
  $optimized = true;
  $rpt .= "Optimization was successful.\n";
  $rpt .= "Execution time: " . $timer->formatted_time() . "\n\n";
  $timer->restart();
  $backup = new lucene_utilitymsg("BACKUP");
  $backup->send(SOCK_FOREVER);
  $success = $backup->response->valid;
  $timer->stop();
  if ($success) {
    $backed_up = true;
    $rpt .= "Index backup was successful.\n";
    $rpt .= "Execution time: " . $timer->formatted_time() . "\n\n";
  }
  else {
    $rpt .= "Index backup FAILED.\n";
    $rpt .= "Error message: " . $backup->response->error_message . "\n";
    $rpt .= "Execution time: " . $timer->formatted_time() . "\n\n";
  }
}
else {
  $rpt .= "Optimization FAILED.\n";
  $rpt .= "Error message: " . $optimizer->response->error_message . "\n";
  $rpt .= "Execution time: " . $timer->formatted_time() . "\n\n";
  $rpt .= "No backup operation was performed.\n\n";
}

// Summary..
if ($success) {
  $rpt .= "Your Lucene index was optimized, and backed up.\n";
  $subject = APP_NAME . " Lucene Optimize and Backup OK";
}
else {
  $rpt .= "Problems were encountered. ";
  if (!$optimized) {
    $rpt .= "The optimization failed, which usually means that the\n"
          . "Lucene index has become corrupted. In this case you should immediately\n"
          . "try to switch Lucene to your last known good backup index.\n\n";
  }
  else {
    $rpt .= "The index backup failed. This could be due to a permissions problem, or\n";
    $rpt .= "perhaps the disk is full.\n\n";
  }
  $subject = APP_NAME . " Lucene Optimize and Backup FAILURE";
}

// Only send email if we have one defined..
if (isset($WEBMASTER_EMAIL) && $WEBMASTER_EMAIL != "") {
  $ereport = new email(
              // From: e-mail address, usually your website..
              "\"Axyl\" <axyl@axyl.org>",
              // To: e-mail, usually your administrator..
              $WEBMASTER_EMAIL,
              $subject,
              $rpt
              );
  $ereport->send();
}

?>