/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Application.h"
#include "OperationLog.h"
#include "TrayIcon.h"
#include "TrayMenu.h"
#include "LogDialog.h"
#include "RequestLogWindow.h"
#include "BasicConfigDialog.h"
#include "AdvancedConfigWindow.h"
#include "FilterConfigWindow.h"
#include "Conf.h"
#include "ConfigErrorHandler.h"
#include "RegexFilterDescriptor.h"
#include "FilterGroupTag.h"
#include "GlobalState.h"
#include "UrlPatterns.h"
#include "InetAddr.h"
#include "SymbolicInetAddr.h"
#include "DnsResolver.h"
#include "IntrusivePtr.h"
#include "RefCountableSAP.h"
#include "AutoClosingSAP.h"
#include "WorkerThreadPool.h"
#include "ScopedIncDec.h"
#include "EffectiveFileTimestamps.h"
#include "AtomicOps.h"
#include <ace/config-lite.h>
#include <ace/Dirent.h>
#include <ace/FILE_Addr.h>
#include <ace/FILE_IO.h>
#include <ace/FILE_Connector.h>
#include <ace/SOCK_Acceptor.h>
#include <ace/OS_NS_sys_stat.h>
#include <ace/OS_NS_stdio.h>
#include <ace/OS_NS_unistd.h>
#include <boost/tokenizer.hpp>
#include <wx/wx.h>
#include <wx/event.h>
#include <wx/dir.h>
#include <wx/file.h>
#include <wx/log.h>
#include <wx/filefn.h>
#include <sstream>
#include <fstream>
#include <string>
#include <list>
#include <deque>
#include <vector>
#include <set>
#include <algorithm>
#include <stddef.h>

using namespace std;

struct Application::ContentFilterFnameComparator
{
	bool operator()(
		IntrusivePtr<ContentFilterGroup> const& lhs,
		IntrusivePtr<ContentFilterGroup> const& rhs) {
		return lhs->fileName() < rhs->fileName();
	}
};


IMPLEMENT_APP_NO_MAIN(Application)

BEGIN_EVENT_TABLE(Application, wxApp)
	EVT_IDLE(onIdle)
END_EVENT_TABLE()


Application::Application()
:	m_isExiting(false),
	m_commandNestLevel(0),
	m_havePendingCommands(0),
	m_commandQueue(COMMAND_QUEUE_CAPACITY, *this),
	m_networkActivityHandler(m_commandQueue),
	m_filterJsLogHandler(m_commandQueue),
	m_requestLogHandler(m_commandQueue),
	m_isAcceptingConnections(false)
{
}

Application::~Application()
{
}

bool
Application::OnInit()
{
	m_appDir.Assign(argv[0]);
	m_appDir.AssignDir(m_appDir.GetPath(
		wxPATH_GET_VOLUME|wxPATH_GET_SEPARATOR
	));
	m_confDir.Assign(m_appDir);
	m_confDir.AppendDir(_T("conf"));
	m_ptrTrayIcon.reset(new TrayIcon);
	m_ptrTrayMenu.reset(new TrayMenu);
	m_ptrWorkerPool.reset(new WorkerThreadPool);
	
	bool errors = false;
	errors |= !loadConfig();
	loadStandardUrlPatterns();
	loadLocalUrlPatterns();
	loadContentFilters();
	if (errors) {
		showLogDialog();
	}
	m_ptrWorkerPool->start();
	return true;
}

int
Application::OnExit()
{
	// This will finish the worker threads.
	m_ptrWorkerPool.reset(0);
	
	/*
	Initialyzing these things in the constructor
	or deleting them in the destructor makes WxWidgets crash.
	That's why we initialize them in OnInit() and delete them in OnExit().
	*/
	m_ptrTrayIcon.reset(0);
	m_ptrTrayMenu.reset(0);
	return 0;
}

void
Application::showTrayMenu()
{
	m_ptrTrayIcon->PopupMenu(m_ptrTrayMenu.get());
}

void
Application::showLogDialog()
{
	LogDialog::show();
}

void
Application::showRequestLogWindow()
{
	RequestLogWindow::show();
}

void
Application::showBasicConfigDialog()
{
	BasicConfigDialog::show();
}

void
Application::showAdvancedConfigWindow()
{
	AdvancedConfigWindow::show();
}

void
Application::showFilterConfigWindow()
{
	FilterConfigWindow::show();
}

void
Application::setFilteringEnabled(bool val)
{
	GlobalState::WriteAccessor()->setFilteringEnabled(val);
}

void
Application::requestAppExit()
{
	// If we just call ExitMainLoop(), we may get a deadlock
	// while waiting for worker threads to finish
	// (in AcceptorThread DTOR). This happens if any worker thread
	// is blocked while sending a command through m_commandQueue.
	if (m_commandQueue.close() == 0) {
		// command queue is empty
		ExitMainLoop();
	} else {
		// Wait for the pending commands to be processed.
		// The actual quit will be done in notifyQueueEmpty().
		m_isExiting = true;
	}
}

bool
Application::fileToStream(ACE_FILE_IO& file, std::ostream& strm)
{
	char buf[4096];
	ssize_t read = 0;
	while ((read = file.recv(buf, sizeof(buf))) > 0) {
		strm.write(buf, read);
	}
	return (read == 0);
}

bool
Application::readFile(
	wxFileName const& fname, std::string& target,
	time_t* mtime, bool log_errors)
{
	AutoClosingSAP<ACE_FILE_IO> file;
	ACE_FILE_Addr addr(fname.GetFullPath().c_str());
	ACE_FILE_Connector connector;
	
	if (connector.connect(file, addr, 0, ACE_Addr::sap_any, 0, O_RDONLY) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Could not open ")+fname.GetFullPath(),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	ostringstream strm;
	
	if (!fileToStream(file, strm)) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Error reading ")+fname.GetFullPath(),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	target = strm.str();
	
	if (mtime) {
		ACE_stat st;
		ACE_OS::fstat(file.get_handle(), &st);
		*mtime = st.st_mtime;
	}
	
	return true;
}

bool
Application::writeFile(
	wxFileName const& fname, std::string const& data,
	time_t* mtime, bool log_errors)
{
	AutoClosingSAP<ACE_FILE_IO> file;
	ACE_FILE_Addr addr(fname.GetFullPath().c_str());
	ACE_FILE_Connector connector;
	
	int const flags = O_WRONLY|O_CREAT|O_TRUNC;
	if (connector.connect(file, addr, 0, ACE_Addr::sap_any, 0, flags) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Could not open ")+fname.GetFullPath(),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	if (file.send_n(data.c_str(), data.size()) == -1) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Error writing ")+fname.GetFullPath(),
				log->getErrorStyle()
			);
		}
		return false;
	}
	
	if (mtime) {
		ACE_stat st;
		ACE_OS::fstat(file.get_handle(), &st);
		*mtime = st.st_mtime;
	}
	
	return true;
}

bool
Application::renameFile(
	wxFileName const& from, wxFileName const& to, bool log_errors)
{
	wxLogNull nolog;
	if (!wxRenameFile(from.GetFullPath(), to.GetFullPath())) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Error renaming ") + from.GetFullPath() +
				_T(" to ") + to.GetFullPath(),
				log->getErrorStyle()
			);
		}
		return false;
	}
	return true;
}

bool
Application::deleteFile(wxFileName const& fname, bool log_errors)
{
	wxLogNull nolog;
	if (!fname.FileExists()) {
		return true;
	}
	if (!wxRemoveFile(fname.GetFullPath())) {
		if (log_errors) {
			Log* log = OperationLog::instance();
			log->appendRecord(
				_T("Error deleting ") + fname.GetFullPath(),
				log->getErrorStyle()
			);
		}
		return false;
	}
	return true;
}

bool
Application::processConfig(
	string const& text, Config& target, ConfigFile& file_structure)
{
	ConfigErrorHandler eh(_T("config"));
	file_structure.load(text, target, eh);
	return eh.numErrors() == 0;
}

bool
Application::applyConfig(Config const& config)
{
	typedef RefCountableSAP<ACE_SOCK_Acceptor> Acceptor;
	typedef IntrusivePtr<Acceptor> AcceptorPtr;
	
	Log* log = OperationLog::instance();

	GlobalState::WriteAccessor()->config() = config;
	// We apply the config independently of the result of the
	// bind operation, to make the Basic Config dialog work as expected.
	
	m_ptrWorkerPool->removeAllAcceptors();
	setAcceptingConnections(false);
	
	bool ok = true;
	deque<AcceptorPtr> acceptors;
	list<SymbolicInetAddr> addrs = config.getListenAddresses();
	for (; !addrs.empty(); addrs.pop_front()) {
		SymbolicInetAddr const& addr = addrs.front();
		vector<InetAddr> resolved_addrs = DnsResolver::resolve(addr);
		if (resolved_addrs.empty()) {
			ostringstream err;
			err << "Could not resolve listen address \"" << addr << '"';
			log->appendRecord(err.str().c_str(), log->getErrorStyle());
			ok = false;
			continue;
		}
		AcceptorPtr acceptor(new Acceptor);
		if (acceptor->open(resolved_addrs[0], true) == -1) {
			ostringstream err;
			err << "Could not bind to \"" << addr << '"';
			log->appendRecord(err.str().c_str(), log->getErrorStyle());
			ok = false;
			continue;
		}
		acceptors.push_back(acceptor);
	}
	if (!ok) {
		return false;
	}
	if (acceptors.empty()) {
		log->appendRecord(
			_T("No addresses to listen on!"),
			log->getErrorStyle()
		);
		return false;
	}
	
	for (; !acceptors.empty(); acceptors.pop_front()) {
		m_ptrWorkerPool->addAcceptor(acceptors.front());
	}
	setAcceptingConnections(true);
	
	return true;
}

void
Application::applyBrokenConfig(Config const& broken_config)
{
	GlobalState::WriteAccessor()->config() = broken_config;
	// We apply it anyway, to make the Basic Config dialog work as expected.
	
	m_ptrWorkerPool->removeAllAcceptors();
	setAcceptingConnections(false);
}

bool
Application::loadConfig()
{
	Log* log = OperationLog::instance();
	log->appendRecord(_T("Loading config ... "));
	size_t num_records = log->getNumRecords();

	wxFileName fname(m_confDir);
	fname.SetName(_T("config"));
	string text;
	time_t mtime = 0;
	if (!readFile(fname, text, &mtime)) {
		return false;
	}
	EffectiveFileTimestamps::config = mtime;
	
	Config config;
	if (!processConfig(text, config, m_configFile)) {
		applyBrokenConfig(config);
		return false;
	}
	
	if (!applyConfig(config)) {
		return false;
	}
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord(_T("done"), log->getSuccessStyle());
	}
	
	return true;
}

bool
Application::processUrlPatterns(
	std::string const& text, wxString const& fname,
	UrlPatterns& target, UrlPatternsFile& file_structure)
{
	ConfigErrorHandler eh(fname);
	file_structure.load(text, target, eh);
	return eh.numErrors() == 0;
}

void
Application::applyStandardUrlPatterns(UrlPatterns const& patterns)
{
	GlobalState::WriteAccessor()->urlPatterns().standardPatterns() = patterns;
}

void
Application::applyLocalUrlPatterns(UrlPatterns const& patterns)
{
	GlobalState::WriteAccessor()->urlPatterns().localPatterns() = patterns;
}

void
Application::loadStandardUrlPatterns()
{
	Log* log = OperationLog::instance();
	log->appendRecord(_T("Loading standard url patterns ... "));
	size_t num_records = log->getNumRecords();

	wxFileName fname(m_confDir);
	fname.SetName(_T("urls"));
	string text;
	readFile(fname, text);
	
	UrlPatterns patterns;
	processUrlPatterns(
		text, fname.GetFullName(),
		patterns, m_standardUrlPatternsFile
	);
	applyStandardUrlPatterns(patterns);
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord(_T("done"), log->getSuccessStyle());
	}
}

void
Application::loadLocalUrlPatterns()
{
	Log* log = OperationLog::instance();
	log->appendRecord(_T("Loading local url patterns ... "));
	size_t num_records = log->getNumRecords();

	wxFileName fname(m_confDir);
	fname.SetName(_T("urls.local"));
	string text;
	time_t mtime = 0;
	readFile(fname, text, &mtime);
	
	UrlPatterns patterns;
	processUrlPatterns(
		text, fname.GetFullName(),
		patterns, m_localUrlPatternsFile
	);
	applyLocalUrlPatterns(patterns);
	
	EffectiveFileTimestamps::urls_local = mtime;
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord(_T("done"), log->getSuccessStyle());
	}
}

void
Application::loadContentFilters()
{
	Log* log = OperationLog::instance();
	log->appendRecord(_T("Loading content filters ... "));
	size_t num_records = log->getNumRecords();
	
	wxFileName dirname = getFiltersDir();
	deque<wxString> fnames;
	loadFilterFnames(dirname, fnames);
	
	std::sort(fnames.begin(), fnames.end());
	m_contentFilters.clear();
	
	for (; !fnames.empty(); fnames.pop_front()) {
		wxString fname(fnames.front());
		wxFileName abs_fname(dirname);
		abs_fname.SetName(fname);
		
		string text;
		if (!readFile(abs_fname, text)) {
			continue;
		}
		
		IntrusivePtr<ContentFilterGroup> group(
			new ContentFilterGroup(FilterGroupTag(), fname)
		);
		
		ConfigErrorHandler eh(fname);
		group->fileStructure().load(
			text, group->filters(), eh, group->getTag()
		);
		if (eh.numErrors() == 0) {
			loadEnabledFilterList(*group, dirname);
		}
		
		m_contentFilters.push_back(group);
	}
	
	updateGlobalContentFilters();
	
	if (num_records == log->getNumRecords()) {
		log->appendToLastRecord(_T("done"), log->getSuccessStyle());
	}
}

bool
Application::loadFilterFnames(
	wxFileName const& dirname, std::deque<wxString>& fnames)
{
	Log* log = OperationLog::instance();
	wxString dirname_str = dirname.GetFullPath();
	
	ACE_Dirent dir;
	if (dir.open(dirname_str.c_str()) == -1) {
		log->appendRecord(
			_T("Could not open directory ") + dirname_str,
			log->getErrorStyle()
		);
		return false;
	}
	
	for (dirent* ent; (ent = dir.read()); ) {
		wxString const fname(ent->d_name);
		if (fname.Find(_T('.')) != -1) {
			// skip files with extensions
			continue;
		}
		fnames.push_back(fname);
	}
	
	return true;
}

void
Application::loadEnabledFilterList(
	ContentFilterGroup& group, wxFileName const& dirname)
{
	wxFileName fname(dirname);
	fname.SetName(group.fileName() + _T(".enabled"));
	if (!fname.FileExists()) {
		return;
	}
	
	string data;
	if (!readFile(fname, data)) {
		return;
	}
	
	typedef boost::tokenizer<boost::char_separator<char> > Tokenizer;
	boost::char_separator<char> sep("\r\n");
	Tokenizer tokens(data, sep);
	set<string> enabled_set(tokens.begin(), tokens.end());
	bool const all_enabled = (enabled_set.find("*") != enabled_set.end());
	
	typedef ContentFilters::FilterList FilterList;
	FilterList& list = group.filters().filters();
	FilterList::iterator it = list.begin();
	FilterList::iterator const end = list.end();
	for (; it != end; ++it) {
		RegexFilterDescriptor& filter = **it;
		bool const enabled = all_enabled ||
			(enabled_set.find(filter.name()) != enabled_set.end());
		filter.setEnabled(enabled);
	}
}

void
Application::appendContentFilterGroup(
	IntrusivePtr<ContentFilterGroup> const& group)
{
	m_contentFilters.push_back(group);
}

bool
Application::removeContentFilterGroup(
	IntrusivePtr<ContentFilterGroup> const& group)
{
	ContentFilterList::iterator it = std::find(
		m_contentFilters.begin(), m_contentFilters.end(), group
	);
	if (it == m_contentFilters.end()) {
		return false;
	}
	m_contentFilters.erase(it);
	return true;
}

void
Application::sortContentFiltersByFileName()
{
	std::stable_sort(
		m_contentFilters.begin(),
		m_contentFilters.end(),
		ContentFilterFnameComparator()
	);
}

void
Application::updateGlobalContentFilters()
{
	ContentFilters composition;
	
	ContentFilterList::iterator it = m_contentFilters.begin();
	ContentFilterList::iterator const end = m_contentFilters.end();
	for (; it != end; ++it) { 
		typedef ContentFilters::FilterList FilterList;
		FilterList& filters = (*it)->filters().filters();
		FilterList::iterator it2 = filters.begin();
		FilterList::iterator const end2 = filters.end();
		for (; it2 != end2; ++it2) {
			if ((*it2)->isEnabled()) {
				IntrusivePtr<RegexFilterDescriptor> new_filter(
					new RegexFilterDescriptor(**it2)
				);
				// Q: Why do we make a copy?
				// A: Because filters in the global filter list are going
				// to be accessed without locks.
				composition.filters().push_back(new_filter);
			}
		}
	}
	
	composition.sortByOrder();
	GlobalState::WriteAccessor gstate;
	gstate->contentFilters().filters().swap(composition.filters());
}

wxFileName
Application::getConfigFileName() const
{
	wxFileName fname(m_confDir);
	fname.SetName(_T("config"));
	return fname;
}

wxFileName
Application::getConfigDefaultFileName() const
{
	wxFileName fname(m_confDir);
	fname.SetName(_T("config.default"));
	return fname;
}

wxFileName
Application::getStandardUrlPatternsFileName() const
{
	wxFileName fname(m_confDir);
	fname.SetName(_T("urls"));
	return fname;
}

wxFileName
Application::getLocalUrlPatternsFileName() const
{
	wxFileName fname(m_confDir);
	fname.SetName(_T("urls.local"));
	return fname;
}

wxFileName
Application::getFiltersDir() const
{
	wxFileName dir(m_confDir);
	dir.AppendDir(_T("filters"));
	return dir;
}

void
Application::handleNetworkActivity()
{
	m_ptrTrayIcon->displayNetworkActivity();
}

void
Application::notifyCommandsPending()
{
	AtomicOps::set(&m_havePendingCommands, 1);
	wxWakeUpIdle();
}

// Note: this function is called from the main thread only.
void
Application::notifyQueueEmpty()
{
	if (m_isExiting) {
		ExitMainLoop();
	}
}

void
Application::onIdle(wxIdleEvent& evt)
{
	processCommands();
}

void
Application::processCommands()
{
	if (m_commandNestLevel) {
		// If a command does things that invoke a recursive main loop,
		// we can end up here.
		return;
	}
	
	if (AtomicOps::set(&m_havePendingCommands, 0) == 0) {
		return;
	}
	
	ScopedIncDec<int> nest_level_manager(m_commandNestLevel);
	
	unsigned max_commands = m_commandQueue.getCapacity();
	/*
	We limit the number of iterations to give the GUI thread
	a chance to draw something if other threads keep posting commands.
	*/
	for (; max_commands > 0; --max_commands) {
		InterthreadCommandQueue::CommandPtr command = m_commandQueue.pop();
		if (!command) {
			return;
		}
		(*command)();
	}
	notifyCommandsPending(); // there could be more commands in the queue
}

void
Application::setAcceptingConnections(bool val)
{
	if (val != m_isAcceptingConnections) {
		m_isAcceptingConnections = val;
		m_acceptingConnectionsSignal.emit(val);
	}
}
