/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "pch.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "Log.h"
#include "LogObserver.h"
#include <ace/config-lite.h>
#include <wx/colour.h>
#include <wx/font.h>


Log::Log()
:	m_numRecords(0),
	m_numVisibleViews(0),
	m_haveUnseenWarnings(false),
	m_visibilityReporter(sigc::mem_fun(*this, &Log::onViewVisibilityChange)),
	m_ptrDefaultStyle(new TextStyle),
	m_ptrSuccessStyle(new TextStyle),
	m_ptrWarningStyle(new TextStyle),
	m_ptrErrorStyle(new TextStyle)
{
	int const flags = wxTEXT_ATTR_TEXT_COLOUR|wxTEXT_ATTR_FONT_WEIGHT;

	{
		wxFont font;
		font.SetWeight(wxNORMAL);
		m_ptrDefaultStyle->SetFont(font);
		m_ptrDefaultStyle->SetTextColour(wxColour(0x00, 0x00, 0x00));
		m_ptrDefaultStyle->SetFlags(flags);
	}
	{
		wxFont font;
		font.SetWeight(wxBOLD);
		m_ptrSuccessStyle->SetFont(font);
		m_ptrSuccessStyle->SetTextColour(wxColour(0x38, 0x89, 0x3b));
		m_ptrSuccessStyle->SetFlags(flags);
	}
	{
		wxFont font;
		font.SetWeight(wxNORMAL);
		m_ptrWarningStyle->SetFont(font);
		m_ptrWarningStyle->SetTextColour(wxColour(0xf4, 0x1d, 0x00));
		m_ptrWarningStyle->SetFlags(flags);
	}
	{
		wxFont font;
		font.SetWeight(wxBOLD);
		m_ptrErrorStyle->SetFont(font);
		m_ptrErrorStyle->SetTextColour(wxColour(0xff, 0x00, 0x00));
		m_ptrErrorStyle->SetFlags(flags);
	}
}

Log::~Log()
{
}

void
Log::appendRecord(StyledText const& text)
{
	++m_numRecords;
	appendText(text, !m_text.empty());
}

void
Log::appendToLastRecord(StyledText const& text)
{
	appendText(text, false);
}

void
Log::clear()
{
	m_text.clear();
	m_numRecords = 0;
	
	std::set<LogObserver*>::iterator it = m_observers.begin();
	std::set<LogObserver*>::iterator const end = m_observers.end();
	for (; it != end; ++it) {
		(*it)->clear();
	}
	
	if (m_haveUnseenWarnings) {
		m_haveUnseenWarnings = false;
		m_unseenWarningsSignal(false);
	}
}

void
Log::appendText(StyledText const& text, bool prepend_newline)
{
	StyledText txt(text);
	if (prepend_newline) {
		txt.text() = _T("\n") + txt.text();
	}
	m_text.push_back(txt);
	std::set<LogObserver*>::iterator it = m_observers.begin();
	std::set<LogObserver*>::iterator const end = m_observers.end();
	for (; it != end; ++it) {
		(*it)->appendText(txt);
	}
	
	if (&txt.style() == m_ptrErrorStyle.get() ||
	    &txt.style() == m_ptrWarningStyle.get()) {
		if (!m_haveUnseenWarnings && m_numVisibleViews == 0) {
			m_haveUnseenWarnings = true;
			m_unseenWarningsSignal(true);
		}
	}
}

void
Log::onViewVisibilityChange(bool visible)
{
	if (!visible) {
		--m_numVisibleViews;
	} else {
		if (m_numVisibleViews++ == 0) {
			if (m_haveUnseenWarnings) {
				m_haveUnseenWarnings = false;
				m_unseenWarningsSignal(false);
			}
		}
	}
}
