/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "MonotonicTimer.h"
#include "TimeStamp.h"
#include "types.h"

#if defined(_WIN32)

#include "AtomicOps.h"
#include <mmsystem.h> // for timeGetTime()

#define MONOTONIC_TIMER_IMPL Win32MonotonicTimer

class Win32MonotonicTimer
{
public:
	static TimeStamp getTimestamp();
private:
	static volatile int32_t m_sWraparoundInfo;
};


volatile int32_t Win32MonotonicTimer::m_sWraparoundInfo = 0;

TimeStamp
Win32MonotonicTimer::getTimestamp()
{
	int32_t wraparound_info = m_sWraparoundInfo;
	// Non-atomic read is OK here. It doesn't really matter if we get
	// the old value or the new one.
	
	DWORD msec = ::timeGetTime();
	
#	define MSB_MASK(type) ((type)1 << (sizeof(type) * 8 - 1))
	
	DWORD msec_msb = msec & MSB_MASK(DWORD);
	int32_t wraparound_msb = wraparound_info & MSB_MASK(int32_t);
	int32_t num_wraparounds = wraparound_info & ~MSB_MASK(int32_t);
	
	if (msec_msb != (DWORD)wraparound_msb) {
		if (msec_msb) {
			wraparound_info = num_wraparounds | MSB_MASK(int32_t);
		} else {
			++num_wraparounds;
			wraparound_info = num_wraparounds;
		}
		AtomicOps::set(&m_sWraparoundInfo, wraparound_info);
	}
	
	uint64_t msec64(msec);
	msec64 += uint64_t(uint32_t(num_wraparounds)) << (sizeof(DWORD)*8);
	return TimeStamp::fromMsec(int64_t(msec64));
}

#elif defined(__MACH__)

#include <mach/mach_time.h>

#define MONOTONIC_TIMER_IMPL MachMonotonicTimer

class MachMonotonicTimer
{
public:
	static TimeStamp getTimestamp();
private:
	class Initializer
	{
	public:
		Initializer();
	};
	friend class MachMonotonicTimer::Initializer;
	
	static long double m_sTimeFactor;
	static Initializer m_sInitializer;
};


long double MachMonotonicTimer::m_sTimeFactor = 0.0L;
MachMonotonicTimer::Initializer MachMonotonicTimer::m_sInitializer;

MachMonotonicTimer::Initializer::Initializer()
{
	if (m_sTimeFactor != 0.0L) {
		return;
	}
	
	struct mach_timebase_info info;
	mach_timebase_info(&info);
		
	m_sTimeFactor = (long double)info.numer / ((long double)info.denom * 1000.0L);
}

TimeStamp
MachMonotonicTimer::getTimestamp()
{
	Initializer initializer;
	
	uint64_t abstime = mach_absolute_time();
	return TimeStamp::fromUsec(int64_t(m_sTimeFactor * abstime));
}

#else

#include <ace/OS_NS_unistd.h>
#include <ace/OS_NS_sys_time.h>

#define MONOTONIC_TIMER_IMPL PosixMonotonicTimer

class PosixMonotonicTimer
{
public:
	static TimeStamp getTimestamp();
};

TimeStamp
PosixMonotonicTimer::getTimestamp()
{
#if defined(HAVE_CLOCK_GETTIME) && _POSIX_MONOTONIC_CLOCK > 0
	struct timespec tv;
	if (clock_gettime(CLOCK_MONOTONIC, &tv) == 0) {
		return TimeStamp::fromTimespec(tv);
	} else // can happen for example in Linux kernel 2.4
#endif
	return TimeStamp::fromTimeval(ACE_OS::gettimeofday());
}

#endif

TimeStamp
MonotonicTimer::getTimestamp()
{
	return MONOTONIC_TIMER_IMPL::getTimestamp();
}
